/** @file resdel.c
 *  @brief Remove specified regions from result file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Removes region(s) from result file.",
  "Optionally, modified results are written in a new file.",
  " ",
  "Usage: @P filename tacid",
  " ",
  "Options:",
  " -o=<filename>",
  "     Remaining results are written in specified file;",
  "     by default, input file is modified.",
  " --dry",
  "     Dry run: nothing is actually written on disk.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P ug2345ki.res cereb",
  " ",
  "See also: reslist, resrmdpl, rescoll, parcoll, pardiff, paradd, tacdel",
  " ",
  "Keywords: results, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   ri, n=0, dry_run=0;
  RES   res;
  char *cptr, resfile1[FILENAME_MAX], resfile2[FILENAME_MAX], tmp[128];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  resfile1[0]=resfile2[0]=(char)0;
  resInit(&res);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!cptr[0]) continue;
    if(strncasecmp(cptr, "O=", 2)==0) {
      strlcpy(resfile2, cptr+2, FILENAME_MAX); if(strlen(resfile2)>0) continue;
    } else if(strcasecmp(cptr, "DRY")==0) {
      dry_run=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!resfile1[0]) {strlcpy(resfile1, argv[ai], FILENAME_MAX); continue;}
    else break;
  }

  /* Is something missing? */
  if(!resfile1[0] || ai>=argc) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(!resfile2[0]) strcpy(resfile2, resfile1);


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("dry_run := %d\n", dry_run);
    printf("resfile1 := %s\n", resfile1);
    printf("resfile2 := %s\n", resfile2);
  }
  if(verbose>4) RESULT_TEST=verbose-4; else RESULT_TEST=0;


  /*
   *  Read result file
   */
  if(verbose>1) printf("reading %s\n", resfile1);
  if(resRead(resfile1, &res, verbose-2)) {
    fprintf(stderr, "Error in reading '%s': %s\n", resfile1, reserrmsg);
    resEmpty(&res); 
    return(2);
  }


  /*
   *  Select the regions with user-defined IDs
   */
  for(; ai<argc; ai++) {
    if(verbose>1) printf("searching for region %s\n", argv[ai]);
    n=0; ri=atoi(argv[ai])-1; /* maybe its the region number? */
    if(ri<res.voiNr && ri>=0) { /* select that region if it exists */
      res.voi[ri].sw2=1; n=1;
    } else { /* if not then it is the region name */
      n=resSelectRegions(&res, argv[ai], 0);
      for(ri=0; ri<res.voiNr; ri++) if(res.voi[ri].sw) res.voi[ri].sw2=1;
    }
    if(n==0) {
      fprintf(stderr, "Error: %s did not match with any of the regions.\n",
              argv[ai]);
      resEmpty(&res); return(1);
    }
  }


  /*
   *  Delete the selected regions
   */
  if(verbose>0 || dry_run==1) {
    if(dry_run==0) strcpy(tmp, "deleted"); else strcpy(tmp, "would be deleted");
    for(ri=n=0; ri<res.voiNr; ri++) {
      printf("  %3d: %s  ", ri+1, res.voi[ri].name);
      if(res.voi[ri].sw2) {n++; printf("%s\n", tmp);} else printf("\n");
    }
  }
  /* Give warning if no regions were deleted, and quit */
  if(n<1) {
    fprintf(stderr, "Warning: datafile does not contain specified regions.\n");
    resEmpty(&res); return(0);
  }
  /* and if all were deleted, tell that too, and quit */
  if(res.voiNr==n) {
    fprintf(stderr, "Warning: all regions were selected for removal; aborted.\n");
    resEmpty(&res); return(0);
  }
  /* now do the deleting */
  ri=res.voiNr-1;
  while(ri>=0) {
    if(res.voi[ri].sw2) resDelete(&res, ri);
    ri--;
  }


  if(dry_run==0) {
    /*
     *  Write the remaining results
     */
    if(verbose>2) printf("writing %s\n", resfile2);
    if(resWrite(&res, resfile2, verbose-3)) {
      fprintf(stderr, "Error in writing '%s': %s\n", resfile2, reserrmsg);
      resEmpty(&res); return(11);
    }
    if(verbose>0) printf("  %s written.\n", resfile2);
  } else {
    printf("  would write %s\n", resfile2);
  }

  /* Free memory */
  resEmpty(&res);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
