#!/bin/bash
#: Title      : test_b2t_h2o
#: Date       : 2014-08-05
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ ! -z $OS ] && [ $OS eq "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../b2t_h2o$EXT;

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

# Check that required files exist
if [ ! -f $PROGRAM ]; then
  printf "\nError: application file missing.\n\n"
  exit 1
fi
if [ ! -f b2t_h2o_blood.dat ]; then
  printf "\nError: required file missing.\n\n"
  exit 1
fi

if [ ! -f pet01.dft ]; then
  printf "\n\nCreating pet01.dft\n"
# Flow=30 ml/min/dl = 0.005 ml/s/ml
# pH2O=0.5 ml/ml
# Vb=20% = 0.20 ml/ml
# tissue K1=Flow/0.80 = 0.00625
# k2 = K1/pH2O = 0.0125
# E=1.0
  rm -f tissue.dft
  p2t_3c -nosub b2t_h2o_blood.dat 0.00625 0.0125 0 0 0 0 tissue.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
# Calculate the contribution of tissue activity to PET tac
  rm -f tissue_part.dft
  taccalc tissue.dft x 0.80 tissue_part.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
# Calculate PET curve assuming that Vb=Va
  rm -f arterial20_part.dft
  taccalc b2t_h2o_blood.dat x 0.20 arterial20_part.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  taccalc tissue_part.dft + arterial20_part.dft pet01.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
fi


if [ ! -f pet02.dft ]; then
  printf "\n\nCreating pet02.dft\n"
# Flow=30 ml/min/dl = 0.005 ml/s/ml
# pH2O=0.5 ml/ml
# Vb=20% = 0.20 ml/ml
# Va=5% Vv=15%
# tissue K1=Flow/0.80 = 0.00625
# k2 = K1/pH2O = 0.0125
# E=1.0
  rm -f tissue.dft
  p2t_3c -nosub b2t_h2o_blood.dat 0.00625 0.0125 0 0 0 0 tissue.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f arterial05_part.dft
  taccalc b2t_h2o_blood.dat x 0.05 arterial05_part.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f vena_e1.dft
  taccalc tissue.dft : 0.5 vena_e1.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f vena_e1_part.dft
  taccalc vena_e1.dft x 0.15 vena_e1_part.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f tissue_part.dft
  taccalc tissue.dft x 0.80 tissue_part.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f temp.dft
  taccalc tissue_part.dft + arterial05_part.dft temp.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  taccalc temp.dft + vena_e1_part.dft pet02.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
fi

if [ ! -f pet03.dft ]; then
  printf "\n\nCreating pet03.dft\n"
# Calculate PET curve assuming that Va=5% and Vv=15% and extraction of water is 0.9
# tissue K1=E*Flow/0.80 = 0.005625
# k2 = K1/pH2O = 0.01125
  rm -f tissue.dft
  p2t_3c -nosub b2t_h2o_blood.dat 0.005625 0.01125 0 0 0 0 tissue.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
# Calculate the contribution of tissue activity to PET tac
  rm -f tissue_part.dft
  taccalc tissue.dft x 0.80 tissue_part.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
# Calculate the contribution of venous activity to PET tac
  rm -f vena_e09_first.dft
  taccalc tissue.dft : 0.5555555556 vena_e09_first.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f vena_e09_second.dft
  taccalc b2t_h2o_blood.dat x 0.1 vena_e09_second.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f vena_e09.dft
  taccalc vena_e09_first.dft + vena_e09_second.dft vena_e09.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f vena_e09_part.dft
  taccalc vena_e09.dft x 0.15 vena_e09_part.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f arterial05_part.dft
  taccalc b2t_h2o_blood.dat x 0.05 arterial05_part.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  rm -f temp.dft
  taccalc tissue_part.dft + arterial05_part.dft temp.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
  taccalc temp.dft + vena_e09_part.dft pet03.dft
  if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
fi

tac2svg b2t_h2o_testdata.svg -l pet01.dft pet02.dft pet03.dft
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"
printf "\n\n"

printf "\n0.1.0\n"
printf "Unknown option should give an error\n"
$PROGRAM -stupidoption b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"

printf "\n0.1.1\n"
printf "Extra argument should give an error\n"
$PROGRAM b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat extra
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"

printf "\n0.1.2\n"
printf "Missing argument should give an error\n"
$PROGRAM b2t_h2o_blood.dat 0.97 1.0 5 30 output.dat
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"

printf "\n0.1.3\n"
printf "Binary file as input file should not crash program\n"
$PROGRAM -d $PROGRAM 3 0.97 1.0 5 30 output.dat
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"

printf "===============================================================\n\n"

printf "\n0.2.0\n"
rm -f stdout.txt
printf "Default and given variables are correct\n"
$PROGRAM -d2 b2t_h2o_blood.dat 60.0 0.97 1.0 5.0 30.0 output.dat > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

iftisval stdout.txt save_only_total "1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt flow_per_tissue "0"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt add_to_previous "0"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

iftisval stdout.txt voiname
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt vfile
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

iftisval stdout.txt Flow "0.01"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt pH2O "0.97"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt E "1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt Vb "0.05"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt fA "0.3"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

printf "\n0.2.1\n"
rm -f stdout.txt
printf "Default and given variables are correct with decimal commas\n"
$PROGRAM -d2 b2t_h2o_blood.dat 66,6 0,97 0,9 5,1 33,3 output.dat > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

iftisval stdout.txt Flow "0.0111"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt pH2O "0.97"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt E "0.9"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt Vb "0.051"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt fA "0.333"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

printf "===============================================================\n\n"

printf "\n0.3.0\n"
rm -f stdout.txt
printf "Options -sub and -nosub\n"

$PROGRAM -d2 -sub b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt save_only_total "0"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

$PROGRAM -d2 -nosub b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt save_only_total "1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


printf "\n0.3.1\n"
rm -f stdout.txt
printf "Option -add\n"

$PROGRAM -d2 -add b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt add_to_previous "1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


printf "\n0.3.2\n"
rm -f stdout.txt
printf "Option -voiname\n"

$PROGRAM -d2 -voiname=nasse b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt voiname "nasse"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

$PROGRAM -d2 -voiname= b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

$PROGRAM -d2 -voiname=thisistoolongnameforavoi b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

rm -f stdout.txt
$PROGRAM -d2 -sub -voiname='this_is_long' b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
taclist -ift output.dat > stdout.txt
iftisval stdout.txt tacname[1] "this_is_long"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

rm -f stdout.txt
$PROGRAM -d2 -sub -voiname=. b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
taclist -ift output.dat > stdout.txt
iftisval stdout.txt tacname[1] ""
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"



printf "\n0.3.3\n"
rm -f stdout.txt
printf "Option -fpt\n"

$PROGRAM -d2 -fpt b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt flow_per_tissue "1"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


printf "\n0.3.4\n"
rm -f stdout.txt
printf "Option -vena\n"

$PROGRAM -d2 -vena=vena.dat b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat > stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval stdout.txt vfile "vena.dat"
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

$PROGRAM -d2 -vena= b2t_h2o_blood.dat 3 0.97 1.0 5 30 output.dat > stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


printf "===============================================================\n\n"
print "Simulation with reasonable values gives correct result\n\n"

printf "\n1.0.0 \n"
print "Test with E=1 and fA=1 \n"
rm -f output1.dat
$PROGRAM -nosub b2t_h2o_blood.dat 30 0.5 1 20 100 output1.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
tacmatch -abs=0.5 -rel=1 pet01.dft output1.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

printf "\n1.0.1 \n"
printf "Test with E=1 and fA=0.25 \n"
rm -f output2.dat
$PROGRAM -nosub b2t_h2o_blood.dat 30 0.5 1 20 25 output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
tacmatch -abs=0.5 -rel=1 pet02.dft output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

printf "\n1.0.2 \n"
printf "Test with E=0.9 and fA=0.25 \n"
rm -f output3.dat
$PROGRAM -nosub b2t_h2o_blood.dat 30 0.5 0.9 20 25 output3.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
tacmatch -abs=0.5 -rel=1  pet03.dft output3.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


printf "\n===============================================================\n\n"
printf "Simulation with nonphysiological values gives error\n\n"

printf "\n1.1.0 \n"

$PROGRAM -nosub b2t_h2o_blood.dat 30 0.5 1.1 20 100 output.dat
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

$PROGRAM -nosub b2t_h2o_blood.dat 30 0.5 1.0 120 10 output.dat
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

$PROGRAM -nosub b2t_h2o_blood.dat 30 0.5 1.0 10 110 output.dat
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

printf "\n===============================================================\n\n"
printf "\n1.2.0 \n"
printf "Program can overwrite and backup existing output files\n\n"
rm -f vena.dat output.dat output.dat.bak

$PROGRAM -vena=vena.dat -nosub b2t_h2o_blood.dat 30 0.5 1.0 10 33 output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
if [ ! -f output.dat ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
if [ ! -f vena.dat ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


$PROGRAM -vena=vena.dat -nosub b2t_h2o_blood.dat 300 0.5 1.0 10 33 output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
if [ ! -f output.dat ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
if [ ! -f vena.dat ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
if [ ! -f output.dat.bak ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
if [ ! -f vena.dat.bak ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

tacmatch output.dat output.dat.bak
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
tacmatch vena.dat vena.dat.bak
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


printf "\n---------------------------------------------------------------\n\n"
printf "\n1.2.1 \n"
printf "Option -add can insert TAC in existing file\n\n"
rm -f output.dat output1.dat output2.dat output3.dat

$PROGRAM -nosub b2t_h2o_blood.dat 1   0.9 1.0 10 33 output1.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM -nosub b2t_h2o_blood.dat 10  0.9 1.0 10 33 output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM -nosub b2t_h2o_blood.dat 100 0.9 1.0 10 33 output3.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi

tacadd -ovr outputx.dat output1.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacadd outputx.dat output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacadd outputx.dat output3.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

$PROGRAM -nosub -add b2t_h2o_blood.dat 1   0.9 1.0 10 33 output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
$PROGRAM -nosub -add b2t_h2o_blood.dat 10  0.9 1.0 10 33 output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
$PROGRAM -nosub -add b2t_h2o_blood.dat 100 0.9 1.0 10 33 output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

tacmatch outputx.dat output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


printf "\n"
printf "===============================================================\n\n"

printf "\n2.0.0 \n"
printf "With zero flow, output contains only vascular contribution\n\n"

rm -f output.dat
$PROGRAM -nosub b2t_h2o_blood.dat 0 0.8 1.0 10 40 output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
taccalc b2t_h2o_blood.dat x 0.04 temp.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
tacmatch -abs=0.5 -rel=1  temp.dat output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"


printf "\n2.0.1 \n"
printf "Water input is zero\n\n"

rm -f output.dat output.ift
taccalc b2t_h2o_blood.dat x 0 zeroinput.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM -nosub zeroinput.dat 20 0.8 1.0 10 33 output.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
interpol -i -x=600 output.dat outputi.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
taclist -ift outputi.dat > output.ift
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
iftisval -abs=0.00001 output.ift value[1][1] 0
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"

printf "\n"
printf "===============================================================\n\n"
printf "Option -fpt\n\n"

printf "\n2.1.0 \n"
printf "Test with E=1 and fA=1 \n"

rm -f output1.dat
$PROGRAM -fpt -nosub b2t_h2o_blood.dat 37,5 0.5 1 20 100 output1.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
tacmatch -abs=0.5 -rel=1  pet01.dft output1.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"


printf "\n"
printf "===============================================================\n\n"
printf "Input data\n\n"

printf "\n3.0.0 \n"
printf "Too few samples \n"
rm -f short.dat output.dat stdout.txt
echo 0 0 > short.dat
cat short.dat
$PROGRAM -sub short.dat 20 0.8 1.0 10 33 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"
iftisval stdout.txt Error "too few samples in 'short.dat'."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"

rm -f output.dat stdout.txt
echo 30 56 >> short.dat
cat short.dat
$PROGRAM -sub short.dat 20 0.8 1.0 10 33 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"
iftisval stdout.txt Warning "few samples in 'short.dat'."
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"


printf "---------------------------------------------------------------\n\n"

printf "\n3.1.0 \n"
printf "CSV with tabs accepted \n"

rm -f b2t_h2o_blood.tsv output2.dat
tacformat -mid -hdr=no -f=tsv-uk b2t_h2o_blood.dat b2t_h2o_blood.tsv
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM -nosub b2t_h2o_blood.tsv 30 0.5 1 20 25 output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
tacmatch -abs=0.5 -rel=1 pet02.dft output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


printf "\n3.1.1 \n"
printf "CSV with commas accepted \n"

rm -f b2t_h2o_blood.csv output2.dat
tacformat -mid -hdr=no -f=csv-uk b2t_h2o_blood.dat b2t_h2o_blood.csv
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM -nosub b2t_h2o_blood.csv 30 0.5 1 20 25 output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
tacmatch -abs=0.5 -rel=1  pet02.dft output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"


printf "\n3.1.2 \n"
printf "CSV with semicolons accepted \n"

rm -f b2t_h2o_blood.csv output2.dat
tacformat -mid -hdr=no -f=csv-int b2t_h2o_blood.dat b2t_h2o_blood.csv
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
$PROGRAM -nosub b2t_h2o_blood.csv 30 0.5 1 20 25 output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n"
tacmatch -abs=0.5 -rel=1  pet02.dft output2.dat
if [ $? -ne 0 ] ; then echo Failed! ; exit 1 ; fi
printf "passed.\n\n"





printf "=====================================================================\n"
printf "  All passed!\n"
printf "=====================================================================\n"
exit 0
