#!/bin/bash
#: Title      : test_p2t_3c
#: Date       : 2014-11-04
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../p2t_3c$EXT;

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

if [ ! -f bolus.dat ] || [ ! -f stead100.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f bolus_toofew.dat ] || [ ! -f bolus_wrongorder.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_pb_1cm.dat ] || [ ! -f correct_pb_2cm.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_pb_3cm.dat ] || [ ! -f correct_pb_p3cm.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_ss_1cm.dat ] || [ ! -f correct_ss_2cm.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_ss_3cm.dat ] || [ ! -f correct_ss_p3cm.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f o15_h2o.dft ] || [ ! -f o15_h2o.fra ] || [ ! -f o15_h2o_rfit.dft ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f f18_fdg.dft ] || [ ! -f f18_fdg.fra ] || [ ! -f f18_fdg_rfit.dft ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f c11_way100635.dft ] || [ ! -f c11_way100635.fra ] || [ ! -f c11_way_rfit.dft ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f multi.dat ] ; then
  tacadd -ovr multi.dat bolus.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -force multi.dat bolus.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f bolus_auc.dat ] ; then
  interpol -i -x=1000 bolus.dat bolus_auc.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f o15_h2o_min.dft ] ; then
  cp -f o15_h2o.dft o15_h2o_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xconv=min -yset=kBq/mL o15_h2o_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f f18_fdg_rfit_min.dft ] ; then
  cp -f f18_fdg_rfit.dft f18_fdg_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xset=sec -yset=kBq/mL f18_fdg_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xconv=min f18_fdg_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f c11_way_rfit_min.dft ] ; then
  cp -f c11_way_rfit.dft c11_way_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xset=sec -yset=kBq/mL c11_way_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xconv=min c11_way_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 0.1.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -stupidoption bolus.dat 0.1 0.2 0 0 0 0 tissue.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.1.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM bolus.dat 0.1 0.2 0 0 0 0 tissue.dat extra &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.1.2 \n"
printf " Test case: Missing argument(s). \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM  0.1 0.2 0 0 0 0 tissue.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM bolus.dat 0.1 0.2 0 0 0 0  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM bolus.dat 0.1 0.2 0 0  tissue.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM bolus.dat  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n passed. \n\n"


printf "\n-------------------------------------------------------------------\n"

printf "\n 0.2.0 \n"
printf " Test case: Correct input, no options. \n"
printf " Expected result 1: Values of variables are correct. \n"
printf " Expected result 2: Output file is created. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 bolus.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt pfile "bolus.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt sfile "output.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parameters "0.1 0.2 0 0 0 0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt save_only_total "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

if [ ! -f output.dat ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.1 \n"
printf " Test case: Options -sub and -nosub. \n"
printf " Expected result: Options are correctly identified. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -sub bolus.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt save_only_total "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -nosub bolus.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt save_only_total "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.2 \n"
printf " Test case: Options -parallel and -series. \n"
printf " Expected result: Options are correctly identified. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -parallel bolus.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -series bolus.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -paral bolus.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -ser bolus.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Input file \n"
printf "===================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case: Non-existing input file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dat
$PROGRAM nonexisting.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Wrong input file format. \n"
printf " Expected result: Error. \n\n"

printf "abc defg\nbubudum\nmeh\n" > bad.dat
rm -f stdout.txt output.dat
$PROGRAM bad.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'bad.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case: Input file contains more than one TAC. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dat
$PROGRAM multi.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "plasma data contains more than one curve."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.3 \n"
printf " Test case: Input file contains less than 3 samples. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dat
$PROGRAM bolus_toofew.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "too few samples in plasma data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.4 \n"
printf " Test case: Input file samples are not in ascending order. \n"
printf " Expected result: Error (from simulation routine). \n\n"

rm -f stdout.txt output.dat
$PROGRAM bolus_wrongorder.dat 0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Rate constants \n"
printf "===================================================================\n"

printf "\n 1.1.0 \n"
printf " Test case 1: All but the first rate constant can be set to zero. \n"
printf " Test case 2: Negative rate constants are not allowed. \n"
printf " Test case 3: Rate constant must be samller than 1.0E+08. \n"
printf " Expected result 1: Error if first is zero or less. \n"
printf " Expected result 2: Error if any rate constant is negative. \n"
printf " Expected result 3: Error if any rate constant is too large. \n\n"

rm -f stdout.txt output.dat
$PROGRAM bolus.dat 0 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "k1 must be > 0."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -parallel -d2 bolus.dat 0.1 0 0 0 0 0 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "parameters" "0.1 0 0 0 0 0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -series -d2 bolus.dat 0.1 0 0 0 0 0 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "parameters" "0.1 0 0 0 0 0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



rm -f stdout.txt output.dat
$PROGRAM bolus.dat -0.1 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 -0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 0.2 -0.05 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 0.2 0.05 -0.1 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 0.2 0.1 0.05 -0.01 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 0.2 0.05 0.1 0.2 -0.000001 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


rm -f stdout.txt output.dat
$PROGRAM bolus.dat 2.0E+08 0.2 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 2.0E+08 0 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 0.2 2.0E+08 0 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 0.2 0.05 2.0E+08 0 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 0.2 0.1 0.05 2.0E+08 0 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat 0.1 0.2 0.05 0.1 0.2 2.0E+08 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Output file \n"
printf "===================================================================\n"

printf "\n 1.2.0 \n"
printf " Test case: Output file exists. \n"
printf " Expected result: Backup is made from the existing one. \n\n"
rm -f stdout.txt output.dat output.dat.bak
$PROGRAM -nosub bolus.dat 0.4 0.2 0.2 0.2 0.1 0.2 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub bolus.dat 0.3 0.3 0.3 0.3 0.3 0.3 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -xunit=n -yunit=n output.dat output.dat.bak
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Validity in perfect steady-state and bolus \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case: Steady-state and 1TCM. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat
$PROGRAM stead100.dat 0.6 0.3 0 0 0 0 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -rel=0.01 correct_ss_1cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.1 \n"
printf " Test case: Steady-state and 2TCM. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat
$PROGRAM stead100.dat 0.6 0.3 0.2 0.4 0 0 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -rel=0.01 correct_ss_2cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.2 \n"
printf " Test case: Steady-state and 3TCM, compartments in series. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat
$PROGRAM stead100.dat 0.6 0.3 0.2 0.4 0.1 0.4 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -rel=0.01 correct_ss_3cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.3 \n"
printf " Test case: Steady-state and 3TCM, compartments in parallel. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat
$PROGRAM -parallel stead100.dat 0.6 0.4 0.5 0.2 1.0 1.2 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -rel=0.01 correct_ss_p3cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-------------------------------------------------------------------\n"

printf "\n 2.1.0 \n"
printf " Test case: Bolus and 1TCM. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f output.dat output_auc.dat
$PROGRAM bolus.dat 0.5 5.0 0 0 0 0 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=1000 output.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -rel=0.01 correct_pb_1cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case: Bolus and 2TCM. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f output.dat output_auc.dat
$PROGRAM bolus.dat 0.5 5.0 0.8 0.2 0 0 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=1000 output.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -rel=0.01 correct_pb_2cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.2 \n"
printf " Test case: Bolus and 3TCM, compartments in series. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f output.dat output_auc.dat
$PROGRAM bolus.dat 0.5 5.0 1.2 0.3 0.5 10 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=1000 output.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -rel=0.01 correct_pb_3cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.3 \n"
printf " Test case: Bolus and 3TCM, compartments in parallel. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f output.dat output_auc.dat
$PROGRAM -parallel bolus.dat 0.6 0.4 0.5 0.2 1.0 1.2 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=1000 output.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -rel=0.01 correct_pb_p3cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Comparison to simulations made using RFIT 4.4 \n"
printf "===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Radiowater blood input, range of k1 and k2 values. \n"
printf " Expected result: Simulated TACs match the ones made with RFIT. \n\n"

rm -f output.dat output.dft temp.dat
$PROGRAM -nosub o15_h2o_min.dft 0.01 0.01 0.0 0.0 0.0 0.0 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub o15_h2o_min.dft 0.04 0.05 0.0 0.0 0.0 0.0 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub o15_h2o_min.dft 0.30 0.40 0.0 0.0 0.0 0.0 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub o15_h2o_min.dft 0.80 1.00 0.0 0.0 0.0 0.0 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub o15_h2o_min.dft 1.50 2.00 0.0 0.0 0.0 0.0 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacunit -xconv=sec output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
simframe output.dat o15_h2o.fra output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="p2t_3c vs RFIT 4.4: radiowater" o15_h2o_rfit.svg -s o15_h2o_rfit.dft -l output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

interpol -y -x=20,50,75,90,120,180,300 o15_h2o_rfit.dft temp1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -x=20,50,75,90,120,180,300 output.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=n -yunit=n -abs=0.1 -rel=0.01 temp1.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 3.1.0 \n"
printf " Test case: FDG plasma input, 1-3 TCM. \n"
printf " Expected result: Simulated TACs match the ones made with RFIT. \n\n"

rm -f output.dat output.dft temp.dat
$PROGRAM -nosub f18_fdg.dft 0.100 0.130 0.060 0.000 0.000 0.000 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub f18_fdg.dft 0.100 0.130 0.060 0.010 0.000 0.000 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub f18_fdg.dft 0.017 0.224 0.039 0.019 0.025 0.000 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub f18_fdg.dft 0.022 0.178 0.123 0.003 0.720 0.000 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub f18_fdg.dft 0.800 0.900 0.600 0.400 0.400 0.600 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

simframe -mid output.dat f18_fdg.fra output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="p2t_3c vs RFIT 4.4: FDG" f18_fdg_rfit.svg -s f18_fdg_rfit_min.dft -l output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

interpol -y -x=1.3,2,2.5,5,7,10,20,30,40,50,60,70,80 f18_fdg_rfit_min.dft temp1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -x=1.3,2,2.5,5,7,10,20,30,40,50,60,70,80 output.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=n -yunit=n -abs=0.01 -rel=0.01 temp1.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 3.2.0 \n"
printf " Test case: [C-11]WAY-100635 plasma input, 1-2 TCM. \n"
printf " Expected result: Simulated TACs match the ones made with RFIT. \n\n"

rm -f output.dat output.dft temp.dat
$PROGRAM -nosub c11_way100635.dft 0.160 0.360 0.046 0.046 0.000 0.000 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub c11_way100635.dft 0.160 0.360 0.230 0.023 0.000 0.000 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd --force --silent output.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

simframe -mid output.dat c11_way100635.fra output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="p2t_3c vs RFIT 4.4: [C-11]WAY-100635" c11_way_rfit.svg -s c11_way_rfit_min.dft -l output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

interpol -y -x=1,2,3,4,5,7,10,15,20,30,40,50,60 c11_way_rfit_min.dft temp1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -x=1,2,3,4,5,7,10,15,20,30,40,50,60 output.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=n -yunit=n -abs=0.01 -rel=0.01 temp1.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

