#!/bin/bash
#: Title      : test_p2t_di
#: Date       : 2016-02-21
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../p2t_di$EXT;

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

if [ ! -f frames.dat ] || [ ! -f framel.dat ] || [ ! -f frames5.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f parent.fit ] || [ ! -f ssinput.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

printf "\n simulating data for testing \n"

if [ ! -f parent.dat ] ; then
  rm -f metabolite1.dat metabolite2.dat plasma.dat blood.dat multi.dat
  printf "\n Making parent plasma TAC \n"
  fit2dat -c=0,10,0.05 parent.fit temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=10,120,0.25 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftcat temp1.dat temp2.dat parent.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy -force parent.dat parent
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f metabolite1.dat ] ; then
  rm -f metabolite2.dat plasma.dat
  printf "\n Making metabolite1 plasma TAC \n"
  tactime -nogap parent.dat +1 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub temp1.dat 0.1 0.05 0 0 0 0 metabolite1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy -force metabolite1.dat metab1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f metabolite2.dat ] ; then
  rm -f plasma.dat
  printf "\n Making metabolite2 plasma TAC \n"
  tactime -nogap metabolite1.dat +1 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub temp2.dat 0.025 0.04 0 0 0 0 metabolite2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy -force metabolite2.dat metab2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma.dat ] || [ ! -f metabolites.dat ] || [ ! -f parent.rat ]; then
  printf "\n Making plasma TAC \n"
  taccalc --force metabolite1.dat + metabolite2.dat metabolites.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force metabolites.dat metab
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force parent.dat + metabolites.dat plasma.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force plasma.dat plasma
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n Making parent ratio \n"
  taccalc --force parent.dat : plasma.dat parent.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force parent.rat parent
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -y1=0 parentrat.svg -l parent.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f blood.dat ] || [ ! -f bp.rat] ; then
  printf "\n Simulating blood TAC, assuming that metab1 goes to RBC \n"
  p2blood norbc parent.dat 0.40 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2blood inwater metabolite1.dat 0.40 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2blood norbc metabolite2.dat 0.40 temp3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force temp1.dat + temp2.dat temp4.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force temp4.dat + temp3.dat blood.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force blood.dat blood
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force blood.dat : plasma.dat bp.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force bp.rat bl-to-pl
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg bprat.svg -l bp.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force blood.dat x 0 blood0.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

printf "\n\n Plotting input TACs \n"
tac2svg -legend=y input.svg -l plasma.dat blood.dat parent.dat metabolite1.dat metabolite2.dat metabolites.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

printf "\n Making short versions of input TACs \n"
interpol -y -c=0,30,1 plasma.dat plasma_short.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -c=0,30,1 parent.dat parent_short.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -c=0,30,1 metabolite1.dat metabolite1_short.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -c=0,30,1 blood.dat blood_short.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

printf "\n Making too short versions of input TACs \n"
interpol -y -x=0,30 plasma.dat plasma_tooshort.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -x=0,30 parent.dat parent_tooshort.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -x=0,30 metabolite1.dat metabolite1_tooshort.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -x=0,30 blood.dat blood_tooshort.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

printf "\n Making input file with several TACs \n"
if [ ! -f multi.dat ] ; then
  tacadd -ovr multi.dat parent.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -y -f=multi.dat metabolite1.dat temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --force multi.dat temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -y -f=multi.dat blood.dat temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd --force multi.dat temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n Simulating data with p2t_3c for comparison \n"

if [ ! -f test0.dat ] ; then
  printf "\n Simulating tissue data with metab1 input and assumption K1p=K1m and Vb=0 \n"
  p2t_3c -nosub parent.dat 0.6 0.2 0 0 0 0 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub metabolite1.dat 0.6 0.6 0 0 0 0 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force temp1.dat + temp2.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub parent.dat 0.12 0.04 0 0 0 0 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub metabolite1.dat 0.12 0.12 0 0 0 0 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force temp1.dat + temp2.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr -sn temp3.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -sn temp3.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp3.dat frames.dat test0.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test0.dat test0
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg test0.svg test0.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f test1.dat ] ; then
  printf "\n Simulating tissue data with metab1 input and assumption K1p=K1m \n"
  p2t_v3c -nosub parent.dat blood.dat 0.6 0.2 0 0 0 0 0 8 100 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub metabolite1.dat blood0.dat 0.6 0.6 0 0 0 0 0 8 100 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force temp1.dat + temp2.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force tac1.dat tac1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub parent.dat blood.dat 0.12 0.04 0 0 0 0 0 8 100 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub metabolite1.dat blood0.dat 0.12 0.12 0 0 0 0 0 8 100 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force temp1.dat + temp2.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force tac2.dat tac2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr -sn temp3.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -sn temp3.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp3.dat frames.dat test1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test1.dat test1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg test1.svg test1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f test2.dat ] ; then
  printf "\n Simulating tissue data with metab2 input \n"
  p2t_v3c -nosub parent.dat blood.dat 0.2 0.1 0 0 0 0 0 8 100 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub metabolite2.dat blood0.dat 0.4 0.5 0 0 0 0 0 8 100 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force temp1.dat + temp2.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force tac1.dat tac1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub parent.dat blood.dat 0.1 0.2 0 0 0 0 0 8 100 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub metabolite2.dat blood0.dat 0.2 0.25 0 0 0 0 0 8 100 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc --force temp1.dat + temp2.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force tac2.dat tac2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr -sn temp4.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -sn temp4.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp4.dat frames.dat test2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test2.dat test2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg test2.svg test2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 0.1.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -stupidoption parent.dat metabolite1.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0 sim.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.1.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM parent.dat metabolite1.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0 sim.dat extra &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.1.2 \n"
printf " Test case: Missing argument(s). \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM  metabolite1.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0 sim.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM parent.dat  blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0 sim.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM parent.dat metabolite1.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0  &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM parent.dat metabolite1.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1 0.1  sim.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM   &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n passed. \n\n"


printf "\n-------------------------------------------------------------------\n"

printf "\n 0.2.0 \n"
printf " Test case: Correct input, no options. \n"
printf " Expected result 1: Values of variables are correct. \n"
printf " Expected result 2: Output file is created. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt pfile "parent_short.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt mfile "metabolite1_short.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt bfile "blood_short.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt sfile "output.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt f "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Vb "0.05"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fA "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt rate_constants "0.1 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt save_only_total "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

if [ ! -f output.dat ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.1 \n"
printf " Test case: Options -sub and -nosub. \n"
printf " Expected result: Options are correctly identified. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -sub parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt save_only_total "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -nosub parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt save_only_total "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.2 \n"
printf " Test case: Options -parallel and -series. \n"
printf " Expected result: Options are correctly identified. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -parallel parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -series parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -paral parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 -ser parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Input file \n"
printf "===================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case: Non-existing input file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dat
$PROGRAM nonexisting.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM parent_short.dat nonexisting.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM parent_short.dat metabolite1_short.dat nonexisting.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.1 \n"
printf " Test case: Wrong input file format. \n"
printf " Expected result: Error. \n\n"

printf "abc defg\nbubudum\nmeh\n" > bad.dat
rm -f stdout.txt output.dat
$PROGRAM bad.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'bad.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM parent_short.dat bad.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'bad.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM parent_short.dat metabolite1_short.dat bad.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'bad.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case: Input file contains more than one TAC. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dat
$PROGRAM multi.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "plasma data contains more than one curve."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM parent_short.dat multi.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "metabolite data contains more than one curve."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM parent_short.dat metabolite1_short.dat multi.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "blood data contains more than one curve."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.3 \n"
printf " Test case: Input file contains less than 3 samples. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dat
$PROGRAM parent_tooshort.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "too few samples in plasma data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM parent_short.dat metabolite1_short.dat blood_tooshort.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "too few samples in blood data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM parent_tooshort.dat metabolite1_tooshort.dat blood_tooshort.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "too few samples in plasma data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Rate constants \n"
printf "===================================================================\n"

printf "\n 1.1.0 \n"
printf " Test case 1: All rate constants can be set to zero (because two inputs). \n"
printf " Test case 2: Negative rate constants are set to zero. \n"
printf " Test case 3: Rate constant must be smaller than 1.0E+08. \n"
printf " Expected result 1: Error if first is zero or less. \n"
printf " Expected result 2: Error if any rate constant is negative. \n"
printf " Expected result 3: Error if any rate constant is too large. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 parent_short.dat metabolite1_short.dat blood_short.dat 0 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "rate_constants" "0 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM -d2 parent_short.dat metabolite1_short.dat blood_short.dat 0,20 0 0 0 0 0 0 0 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "rate_constants" "0.2 0 0 0 0 0 0 0 0.02 0.01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


rm -f stdout.txt output.dat
$PROGRAM -d2 parent_short.dat metabolite1_short.dat blood_short.dat -0.10 0.09 -0.08 0.07 -0.06 0.05 -0.04 0.03 0.02 -0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "rate_constants" "0 0.09 0 0.07 0 0.05 0 0.03 0.02 0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



rm -f stdout.txt output.dat
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 2.0E+08 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 0.10 2.0E+08 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 2.0E+08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 2.0E+08 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 2.0E+08 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 2.0E+08 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 2.0E+08 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 2.0E+08 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Output file \n"
printf "===================================================================\n"

printf "\n 1.2.0 \n"
printf " Test case: Output file exists. \n"
printf " Expected result: Backup is made from the existing one. \n\n"
rm -f stdout.txt output.dat output.dat.bak
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 0.10 0.09 0.08 0.07 0.06 0.05 0.04 0.03 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM parent_short.dat metabolite1_short.dat blood_short.dat 0.40 0.60 0.18 0.17 0.16 0.15 0.14 0.13 0.02 0.01 5 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -xunit=n -yunit=n output.dat output.dat.bak
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Simulation provides the same result as p2t_3c \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: Vb=0. \n"
printf " Test case 2: Parent simulated, rate constants for metabolite are 0. \n"
printf " Test case 3: Compartments in series. \n"
printf " Expected result: Same results as with p2t_3c. \n\n"

rm -f output1.dat output2.dat
$PROGRAM -nosub -series parent.dat metabolite1.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0 0 0 0 0 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
p2t_3c -nosub -series parent.dat 0.1 0.1 0.1 0.1 0.1 0.1 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y test_2_0_0.svg -s output2.dat -l output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n output2.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case 1: Vb=0. \n"
printf " Test case 2: Parent simulated, rate constants for metabolite are 0. \n"
printf " Test case 3: Compartments in parallel. \n"
printf " Expected result: Same results as with p2t_3c. \n\n"

rm -f output1.dat output2.dat
$PROGRAM -nosub -parallel parent.dat metabolite1.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0 0 0 0 0 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
p2t_3c -nosub -parallel parent.dat 0.1 0.1 0.1 0.1 0.1 0.1 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y test_2_0_1.svg -s output2.dat -l output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n output2.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.2 \n"
printf " Test case 1: Vb=0. \n"
printf " Test case 2: Metabolite simulated, rate constants for parent are 0. \n"
printf " Test case 3: Parent compartments in series or parallel does not affect. \n"
printf " Expected result: Same results as with p2t_3c. \n\n"

rm -f output1.dat output2.dat output3.dat
$PROGRAM -nosub -series parent.dat parent.dat blood.dat 0.0 0.0 0.0 0.0 0.0 0.0 0 0 0.1 0.1 0 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub -parallel parent.dat parent.dat blood.dat 0.0 0.0 0.0 0.0 0.0 0.0 0 0 0.1 0.1 0 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
p2t_3c -nosub -parallel parent.dat 0.1 0.1 0.0 0.0 0.0 0.0 output3.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y test_2_0_2.svg -s output3.dat -l output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n output3.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n output3.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.3 \n"
printf " Test case 1: Vb=0.0 \n"
printf " Test case 2: Parent and metabolite simulated, km=0. \n"
printf " Test case 3: 1TCM for parent and metabolite. \n"
printf " Expected result: Same results as with p2t_3c. \n\n"

rm -f output1.dat output2.dat output.dat
$PROGRAM -series parent.dat metabolite1.dat blood.dat 0.6 0.2 0 0 0 0 0 0 0.6 0.6 0 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -series parent.dat metabolite1.dat blood.dat 0.12 0.04 0 0 0 0 0 0 0.12 0.12 0 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd output1.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
simframe output1.dat frames.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y test_2_0_3.svg -s test0.dat -l output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n test0.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Simulation provides the same result as p2t_v3c \n"
printf "===================================================================\n"

printf "\n 2.1.0 \n"
printf " Test case 1: Vb=0.1, fA=0.4 \n"
printf " Test case 2: Parent simulated, rate constants for metabolite are 0. \n"
printf " Test case 3: Compartments in series. \n"
printf " Expected result: Same results as with p2t_v3c. \n\n"

rm -f output1.dat output2.dat
$PROGRAM -nosub -series -fA=40 -f=0.2 parent.dat metabolite1.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0 0 0 0 10 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
p2t_v3c -nosub -series parent.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0.2 10 40 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y test_2_1_0.svg -s output2.dat -l output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n output2.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.1.1 \n"
printf " Test case 1: Vb=0.1, fA=0.4 \n"
printf " Test case 2: Parent simulated, rate constants for metabolite are 0. \n"
printf " Test case 3: Compartments in parallel. \n"
printf " Expected result: Same results as with p2t_v3c. \n\n"

rm -f output1.dat output2.dat
$PROGRAM -nosub -paral -fA=40 -f=0.2 parent.dat metabolite1.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0 0 0 0 10 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
p2t_v3c -nosub -paral parent.dat blood.dat 0.1 0.1 0.1 0.1 0.1 0.1 0.2 10 40 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y test_2_1_1.svg -s output2.dat -l output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n output2.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.2 \n"
printf " Test case 1: Vb=0.08 \n"
printf " Test case 2: Parent and metabolite simulated, km=0. \n"
printf " Test case 3: 1TCM for parent and metabolite. \n"
printf " Expected result: Same results as with p2t_v3c. \n\n"

rm -f output1.dat output2.dat output.dat
$PROGRAM -series parent.dat metabolite1.dat blood.dat 0.6 0.2 0 0 0 0 0 0 0.6 0.6 8 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -series parent.dat metabolite1.dat blood.dat 0.12 0.04 0 0 0 0 0 0 0.12 0.12 8 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd output1.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
simframe output1.dat frames.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y test_2_1_2.svg -s test1.dat -l output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n test1.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 2.2.0 \n"
printf " Test case 1: Vb=0 \n"
printf " Test case 2: km>0. \n"
printf " Expected result 1: Metabolite TAC in tissue is correctly simulated. \n\n"
printf " Expected result 2: Total TAC in tissue is correctly simulated. \n\n"

rm -f output1.dat output2.dat
$PROGRAM -series -sub parent.dat metabolite1.dat blood.dat 0.6 0.2 0 0 0 0 0 0.1 0 0.05 0 output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
p2t_3c -sub -series parent.dat 0.6 0.2 0.1 0 0.05 0 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr mets.dat output1.dat 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr metc.dat output2.dat 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg test_2_2_0a.svg -l metc.dat mets.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n metc.dat mets.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacadd -ovr mets.dat output1.dat 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr temp1.dat output2.dat 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacadd -ovr temp2.dat output2.dat 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force temp1.dat + temp2.dat metc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg test_2_2_0b.svg -l metc.dat mets.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=n -xunit=n -yunit=n metc.dat mets.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Quantitativity as compared to computations in Excel. \n"
printf "===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case 1: K1-k7 >0, metabolite parameters = 0. \n"
printf " Test case 2: Compartments in series. \n"
printf " Expected result: Logan plot of simulated TAC gives correct Vt. \n\n"

rm -f output.dat outputf.dat logan.res logan.ift
$PROGRAM -series -nosub parent.dat metabolite1.dat blood.dat 0.8 1.0 0.5 0.8 0.3 0.6 0.2 0 0 0 0 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
simframe output.dat frames5.dat outputf.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
logan -c -sd=n -svg=test_3_0_0.svg outputf.dat parent.dat 100 120 logan.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
reslist -ift logan.res > logan.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 logan.ift DV 1.369863
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case 1: Full model, except Vb=0. \n"
printf " Test case 2: Compartments in series. \n"
printf " Note: Obligatory assumption in this test is that both inputs are the same. \n"
printf " Expected result: Logan plot of simulated TAC gives correct Vt. \n\n"

rm -f output.dat outputf.dat logan.res logan.ift
$PROGRAM -series -nosub parent.dat parent.dat blood.dat 0.8 1.0 0.5 0.8 0.3 0.6 0.2 0.5 0.4 0.2 0 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
simframe output.dat frames5.dat outputf.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
logan -c -sd=n -svg=test_3_0_1.svg outputf.dat parent.dat 100 120 logan.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
reslist -ift logan.res > logan.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 logan.ift DV 4.222222
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case 1: K1-k7 >0, metabolite parameters = 0. \n"
printf " Test case 2: Compartments in parallel. \n"
printf " Expected result: Logan plot of simulated TAC gives correct Vt. \n\n"

rm -f output.dat outputf.dat logan.res logan.ift
$PROGRAM -parallel -nosub parent.dat metabolite1.dat blood.dat 0.8 1.0 0.5 0.8 0.3 0.6 0.2 0 0 0 0 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
simframe output.dat frames5.dat outputf.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
logan -c -sd=n -svg=test_3_1_0.svg outputf.dat parent.dat 100 120 logan.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
reslist -ift logan.res > logan.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0002 logan.ift DV 1.488372
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.1 \n"
printf " Test case 1: Full model, except Vb=0. \n"
printf " Test case 2: Compartments in parallel. \n"
printf " Note: Obligatory assumption in this test is that both inputs are the same. \n"
printf " Expected result: Logan plot of simulated TAC gives correct Vt. \n\n"

rm -f output.dat outputf.dat logan.res logan.ift
$PROGRAM -paral -nosub parent.dat parent.dat blood.dat 0.8 1.0 0.5 0.8 0.3 0.6 0.2 0.5 0.4 0.2 0 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
simframe output.dat frames5.dat outputf.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
logan -c -sd=n -svg=test_3_1_1.svg outputf.dat parent.dat 100 120 logan.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
reslist -ift logan.res > logan.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 logan.ift DV 4.285714
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 3.2.0 \n"
printf " Test case 1: K1-k7 > 0, Vb=0. \n"
printf " Test case 2: Compartments in parallel. \n"
printf " Test case 3: Steady-state input at level 1. \n"
printf " Expected result: Compartment TACs of parent are at correct level at steady-state. \n\n"

rm -f output.dat outputf.dat ssl.dat ssl.ift
$PROGRAM -paral -sub ssinput.dat ssinput.dat ssinput.dat 0.8 1.0 0.5 0.8 0.3 0.6 0.2 0 0 0 0 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
simframe output.dat frames5.dat outputf.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y test_3_2_0.svg -l outputf.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\nExtract the end part of data \n"
simframe output.dat framel.dat ssl.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -nv -ift ssl.dat > ssl.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.002 ssl.ift value[1][1] 1.488372
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.002 ssl.ift value[2][1] 0.744186
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.002 ssl.ift value[3][1] 0.465116
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.002 ssl.ift value[4][1] 0.27907
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

