#!/bin/bash
#: Title      : test_dftsuv
#: Date       : 2019-08-07
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../dftsuv$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f dynamic.dft ] || [ ! -f dynamic2.dft ] || [ ! -f iframe.dft ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case: No command line options or arguments. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Usage"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "See also"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.1 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -stupidoption dynamic.dft 30 60 140 70 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case: Too many arguments. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM dynamic.dft 30 60 140 70 output.res extra &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.3 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM 30 60 140 70 output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM dynamic.dft 30 60 140 70 &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt
$PROGRAM dynamic.dft 30 60 140    output.res &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Basic tests \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: Plain TAC format without units; saved in TAC format. \n"
printf " Test case 2: SUV TAC is saved with option -curve. \n"
printf " Expected result 1: Results are correct. \n"
printf " Expected result 2: SUV TACs are correct. \n"
printf " Expected result 3: Warnings about assumed units. \n\n"

rm -f stdout.txt output.dat output.tac

$PROGRAM -curve=output.tac dynamic.dft 30 60 140 70 output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "unknown data time units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "assuming that data and time range are in same units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "for comparison, calculate SUV using interpol and taccalc (140/70=2)\n\n"
rm -f mean.dft compsuv.dft
interpol -a -F=iframe.dft dynamic.dft mean.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc mean.dft : 2 compsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacformat -mid -format=simple -hdr=no compsuv.dft compsuv.dat 
tacmatch -rel=0.1 -y=y -yunit=n -xunit=n compsuv.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "for comparison, calculate SUV TAC using taccalc (140/70=2)\n\n"
rm -f dynamicsuv.dft
taccalc dynamic.dft : 2 dynamicsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacformat -mid -format=simple -hdr=no dynamicsuv.dft dynamicsuv.dat 
tacmatch -rel=0.1 -y=y -yunit=n -xunit=n dynamicsuv.dat output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case 1: DFT TAC format with units; saved in DFT format. \n"
printf " Test case 2: SUV TAC is saved with option -curve. \n"
printf " Expected result 1: Results are correct. \n"
printf " Expected result 2: SUV TACs are correct. \n"
printf " Expected result 3: No warnings. \n\n"


rm -f stdout.txt output.tac output.dft

$PROGRAM -curve=output.tac dynamic2.dft 30 60 140 70 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n passed. \n\n"

printf "for comparison, calculate SUV using interpol and taccalc (140/70=2)\n\n"
rm -f mean.dft compsuv.dft
interpol -F=iframe.dft dynamic2.dft mean.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc mean.dft : 2 compsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yset=g/ml compsuv.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Note that results do not need to match perfectly because interpol
# does not consider the frame lengths in the original TAC
tacmatch -rel=3 -x=y -y=y -yunit=y -xunit=y compsuv.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "for comparison, calculate SUV TAC using taccalc (140/70=2)\n\n"
rm -f dynamicsuv.dft
taccalc dynamic2.dft : 2 dynamicsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yset=g/ml dynamicsuv.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -rel=0.1 -x=y -y=y -yunit=y -xunit=y dynamicsuv.dft output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.1.1 \n"
printf " Test case 1: Calculate PID/kg tissue; saved in DFT format. \n"
printf " Test case 2: PID/kg TAC is saved with option -curve. \n"
printf " Expected result 1: Results are correct. \n"
printf " Expected result 2: PID/kg TACs are correct. \n"
printf " Expected result 3: No warnings. \n\n"

rm -f stdout.txt output.tac output.dft

$PROGRAM -density=0.714285714 -curve=output.tac dynamic2.dft 30 60 140 0 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

# Density was set so that correct values should be the same as in original
# data ; only units must be set
cp -f dynamic2.dft dynamicpid.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yset="PID/kg" dynamicpid.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -rel=0.1 -x=y -y=y -yunit=y -xunit=y dynamicpid.dft output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

interpol -F=iframe.dft dynamic2.dft mean.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yset="PID/kg" mean.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -rel=3 -x=y -y=y -yunit=y -xunit=y mean.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.0 \n"
printf " Test case 1: SUV TAC is saved with option -curve. \n"
printf " Test case 2: Result file name not given. \n"
printf " Expected result 1: SUV TACs are correct. \n"
printf " Expected result 2: No error on missing result file. \n\n"

rm -f stdout.txt output.tac

$PROGRAM -curve=output.tac dynamic.dft 30 60 140 70 &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "unknown data time units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "assuming that data and time range are in same units."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "for comparison, calculate SUV TAC using taccalc (140/70=2)\n\n"
rm -f dynamicsuv.dft
taccalc dynamic.dft : 2 dynamicsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacformat -mid -format=simple -hdr=no dynamicsuv.dft dynamicsuv.dat 
tacmatch -rel=0.1 -y=y -yunit=n -xunit=n dynamicsuv.dat output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   Playing with times \n"
printf "===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: User-defined time range extends a bit longer than the TAC. \n"
printf " Expected result 1: Results are correct. \n"
printf " Expected result 2: No warnings. \n\n"

rm -f output.dft stdout.txt

$PROGRAM dynamic2.dft 0 70 140 70 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n for comparison, calculate SUV using dftinteg and taccalc (140/70=2) \n\n"
rm -f mean.dft compsuv.dft
dftinteg -avg dynamic2.dft 0 70 mean.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc mean.dft : 2 compsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yset="g/mL" compsuv.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -rel=0.2 -x=y -y=y -yunit=y -xunit=y compsuv.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case: User-defined time range extends much longer than the TAC. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft stdout.txt

$PROGRAM dynamic2.dft 0 90 140 70 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "time range 0-90 is outside measurement range 0-60."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n-----------------------------------------------------------------\n\n"



printf "\n 3.1.0 \n"
printf " Test case: Time range from 0 when PET data is slightly delayed. \n"
printf " Expected result 1: Results are correct. \n"
printf " Expected result 2: No warnings. \n\n"

rm -f temp.dft output.dft stdout.txt
tactime dynamic2.dft +0.03333333333 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp.dft 0 60 140 70 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n for comparison, calculate SUV using dftinteg and taccalc (140/70=2) \n\n"
rm -f mean.dft compsuv.dft
dftinteg -avg temp.dft 0 60 mean.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc mean.dft : 2 compsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yset="g/mL" compsuv.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -rel=0.2 -x=y -y=y -yunit=y -xunit=y compsuv.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.1 \n"
printf " Test case: Time range from 0 when TAC starts a lot later. \n"
printf " Expected result: Error. \n\n"

rm -f temp.dft output.dft stdout.txt
taccut dynamic2.dft 30 60 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp.dft 0 60 140 70 output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "integration time range oversteps data range."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"




printf "\n 3.2.0 \n"
printf " Test case 1: DFT TAC format with times in sec. \n"
printf " Test case 2: SUV TAC is saved with option -curve. \n"
printf " Expected result 1: Results are correct. \n"
printf " Expected result 2: SUV TACs are correct. \n"
printf " Expected result 3: No warnings. \n\n"


rm -f stdout.txt output.tac output.dft
cp -f dynamic2.dft temp.dft
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -curve=output.tac temp.dft 30 60 140 70 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "calculate comparison data (140/70=2)\n\n"
rm -f temp.dft dynamicsuv.dft compsuv.res
taccalc dynamic2.dft : 2 dynamicsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yset="g/mL" dynamicsuv.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg -avg dynamicsuv.dft 30 30 compsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec compsuv.dft dynamicsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

tacmatch -rel=1 -x=y -y=y -yunit=y -xunit=y compsuv.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -rel=0.1 -x=y -y=y -yunit=y -xunit=y dynamicsuv.dft output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.3.0 \n"
printf " Test case: SUV from static data, user gives zeroes as time range. \n"
printf " Expected result 1: Correct result. \n"
printf " Expected result 2: No warnings. \n\n"

rm -f temp.dft output.dft stdout.txt
dftinteg -avg dynamic2.dft 30 30 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp.dft 0 0 140 70 output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n calculate comparison data (140/70=2)\n\n"
rm -f compsuv.dft
taccalc temp.dft : 2 compsuv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yset="g/mL" compsuv.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

tacmatch -rel=0.1 -x=y -y=y -yunit=y -xunit=y compsuv.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Result data formats \n"
printf "===================================================================\n"

printf "\n 4.0.0 \n"
printf " Test case: SUV saved as RES file. \n"
printf " Expected result: Correct contents. \n\n"

rm -f output.res

$PROGRAM dynamic2.dft 40 60 140 70 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n calculate data to compare to (140/70=2) \n\n"
rm -f temp.dft compsuv.res
taccalc dynamic2.dft : 2 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -yset="g/mL" temp.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg -avg temp.dft 40 20 compsuv.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

resmatch -abs=0.1 compsuv.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

