/** @file addtimes.c
 *  @brief Add time column into TAC data originally without times (e.g. countrates).
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Add sample times into curve data originally without times,",
  "for example certain countrate files.",
  "Added times are 0.5, 1.5, ... sec by default.",
  " ",
  "Usage: @P [Options] tac_file(s)",
  " ",
  "Options:",
  " -min",
  "     Times are written in minutes (0.00833, 0.025, ...).",
  " -cols",
  "     Data may initially contain more than one column.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacsetx, tacframe, tacdecay, tactime, tocr, tac4frpl",
  " ",
  "Keywords: TAC, input, countrate, head-curve",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  main()
 */
int main(int argc, char *argv[])
{
  int ai, help=0, version=0, verbose=1;
  int ii, fileNr=0, ret, is_min=0, is_many_cols=0, tn, n;
  char *cptr, iftfile[FILENAME_MAX], tmp[2*FILENAME_MAX], *vptr, *newval;
  IFT ift;
  float tv;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftInit(&ift);
  iftfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "MINUTES", 1)==0) {
      is_min=1; continue;
    } else if(strncasecmp(cptr, "COLS", 1)==0) {
      is_many_cols=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* In verbose mode print options */
  if(verbose>1) {
    printf("is_min := %d\n", is_min);
    printf("is_many_cols := %d\n", is_many_cols);
  }

  /*
   *  Do one file at a time, starting from the first non-option
   *  command-line argument   
   */
  for(; ai<argc; ai++) {
    strcpy(iftfile, argv[ai]);
    if(verbose>0) fprintf(stdout, "%s:\n", iftfile);

    /* Read data */
    ret=iftRead(&ift, iftfile, 0, 0);
    if(ret) {
      fprintf(stderr, "Error (%d): %s\n", ret, ift.status);
      iftEmpty(&ift); return(2);
    }
    fileNr++;

    /* Go through data lines */
    for(ii=0, tn=0; ii<ift.keyNr; ii++) {
      if(verbose>2) iftWriteItem(&ift, ii, stdout, 0);
      /* do not change comment lines */
      if(ift.item[ii].type!=(char)0 && ift.item[ii].type!=' ') continue;
      /* do not change lines with key */
      if(strlen(ift.item[ii].key)>0) continue;
      /* check that there is a value to edit */
      if(ift.item[ii].value==NULL) continue;
      /* if required, check that there is only one data column */
      vptr=strdup(ift.item[ii].value);
      n=0; cptr=strtok(vptr, " \t");
      while(cptr!=NULL) {cptr=strtok(NULL, " \t"); n++;}
      free(vptr);
      if(is_many_cols==0 && n>1) continue;
      /* construct the new value with time */
      newval=malloc((strlen(ift.item[ii].value)+21)*sizeof(char));
      if(newval==NULL) continue;
      tv=0.5+(float)tn; if(is_min) tv/=60.0;
      sprintf(newval, "%f %s", tv, ift.item[ii].value);
      /* and put it in place */
      ret=iftReplaceNthValue(&ift, ii, newval, 0);
      if(ret) {
        fprintf(stderr, "Error (%d): %s\n", ret, ift.status);
        free(newval); iftEmpty(&ift); return(4);
      }
      tn++; free(newval);
    }
    if(tn==0) {
      fprintf(stderr, "Note: no times added in %s\n", iftfile);
      iftEmpty(&ift); continue;
    }
    fprintf(stdout, "  %d time(s) added.\n", tn);

    /* Backup original file */
    ret=backupExistingFile(iftfile, NULL, tmp);
    if(ret!=0) {
      fprintf(stderr, "Error: %s\n", tmp);
      iftEmpty(&ift); return(12);
    }

    /* Write edited file */
    ret=iftWrite(&ift, iftfile, 0);
    if(ret) {
      fprintf(stderr, "Error (%d): %s\n", ret, ift.status);
      iftEmpty(&ift); return(13);
    }
    iftEmpty(&ift);
  } /* next file */

  if(fileNr<1) {
    fprintf(stderr, "Error: no files processed.\n");
    return(1);
  } else if(fileNr>1 && verbose>0) {
    fprintf(stdout, "%d files processed.\n", fileNr);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
