/** @file tacdecay.c
 *  @brief Corrects or removes correction for physical decay in PET TAC data.
 *  @remark Previously dftdecay.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpccsv.h"
#include "tpcisotope.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Correction of PET TAC files for physical decay, or removing decay correction.",
  " ",
  "Usage: @P [Options] tacfile [outputfile]",
  " ",
  "Options:",
  " -decay=<on|off>",
  "     Correct the data for radioactive decay (on, default), or remove",
  "     decay correction (off).",
  " -i=<Isotope code>",
  "     Most TAC files do not contain the isotope code, thus it can be",
  "     specified with this option.",
  "     Accepted isotope codes are for example F-18, C-11, and O-15.",
  " -sec or -min",
  "     Sample times are known to be in seconds or minutes, but is not",
  "     specified or is wrong in TAC file.", 
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: remove the correction for physical decay from a F-18 study,",
  "           writing non-corrected data into a new file.",
  "  @P -decay=off -i=F-18 a123ap.bld a123ap_nodecay.bld",
  " ",
  "Example 2: correct the TAC data for radioactive decay, overwriting the data.",
  "  @P -decay=on -i=C-11 ia456dy1.tac",
  " ",
  "See also: tactime, tacframe, tacunit, tacformat, imgdecay, dcftime",
  " ",
  "Keywords: TAC, physical decay, halflife, isotopes, simulation",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  isotope  isot=ISOTOPE_UNKNOWN;
  int      correct=1; // 1=correct for decay, 0=remove decay correction
  unit     knownTimeunit=UNIT_UNKNOWN;
  char     tacfile1[FILENAME_MAX], tacfile2[FILENAME_MAX];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile1[0]=tacfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1;
    if(strncasecmp(cptr, "I=", 2)==0) {
      cptr+=2;
      if((isot=isotopeIdentify(cptr))==ISOTOPE_UNKNOWN) {
        fprintf(stderr, "Error: invalid isotope '%s'.\n", cptr);
        return(1);
      }
      continue;
    } else if(strncasecmp(cptr, "DECAY=", 6)==0) {
      if((correct=tpcYesNo(cptr+6))==-1) {
        fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
        return(1);
      }
      continue;
    } else if(strncasecmp(cptr, "MINUTES", 3)==0) {
      knownTimeunit=UNIT_MIN; continue;
    } else if(strncasecmp(cptr, "SECONDS", 3)==0) {
      knownTimeunit=UNIT_SEC; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'


  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(tacfile1, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(tacfile2, argv[ai++], FILENAME_MAX);
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!tacfile1[0]) {
    fprintf(stderr, "Error: missing file name; use option --help\n");
    return(1);
  }
  /* If output filename was not given, use input filename */
  if(!tacfile2[0]) strcpy(tacfile2, tacfile1);


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile1 := %s\n", tacfile1);
    printf("tacfile2 := %s\n", tacfile2);
    printf("correct := %d\n", correct);
    if(isot!=ISOTOPE_UNKNOWN) printf("isotope := %s\n", isotopeName(isot));
    if(knownTimeunit!=UNIT_UNKNOWN) printf("knownTimeunit := %s\n", unitName(knownTimeunit));
    fflush(stdout);
  }



  /*
   *  Read TAC data
   */
  if(verbose>1) {fprintf(stdout, "reading %s\n", tacfile1); fflush(stdout);}
  TAC tac; tacInit(&tac);
  if(tacRead(&tac, tacfile1, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), tacfile1);
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    if(tac.isframe) printf("frames := yes\n"); else printf("frames := no\n");
    fflush(stdout);
  }
  /* Correct the time unit, if given by user */
  if(knownTimeunit!=UNIT_UNKNOWN) {
    tac.tunit=knownTimeunit;
  } else if(!unitIsTime(tac.tunit)) {
    double xmin, xmax;
    if(tacXRange(&tac, &xmin, &xmax)) {
      fprintf(stderr, "Error: invalid sample times.\n");
      tacFree(&tac); return(2);
    }
    if(xmax>30.0 && lambdaFromIsotope(isot)>0.3) {
      tac.tunit=UNIT_SEC;
      fprintf(stderr, "Warning: assuming that sample times are in seconds.\n");
    } else {
      tac.tunit=UNIT_MIN;
      fprintf(stderr, "Warning: assuming that sample times are in minutes.\n");
    }
  }
  if(verbose>2) {
    printf("xunit := %s\n", unitName(tac.tunit));
    printf("yunit := %s\n", unitName(tac.cunit));
    fflush(stdout);
  }

  /* If datafile contains valid isotope, then check that it is the same as given by user */
  {
    isotope fisot=tacGetIsotope(&tac);
    if(verbose>3) {printf("tac.isotope :=  %s\n", isotopeName(fisot)); fflush(stdout);}
    if(isot==ISOTOPE_UNKNOWN && fisot==ISOTOPE_UNKNOWN) {
      fprintf(stderr, "Error: valid isotope not specified.\n");
      tacFree(&tac); return(3);
    }
    if(isot==ISOTOPE_UNKNOWN && fisot!=ISOTOPE_UNKNOWN) {
      isot=fisot;
      if(verbose>1) printf("isotope := %s\n", isotopeName(isot));
    } else if(isot!=ISOTOPE_UNKNOWN && fisot==ISOTOPE_UNKNOWN) {
      fisot=isot;
      tacSetIsotope(&tac, fisot);
      if(verbose>1) printf("isotope := %s\n", isotopeName(isot));
    }
    if(isot!=fisot) {
      fprintf(stderr, "Error: different isotope in %s\n", tacfile1);
      tacFree(&tac); return(3);
    }
  }

  /* If datafile contains information on decay correction, then check that to prevent 
     user from decay correcting data twice, or removing non-existing decay correction */
  decaycorrection fdc=tacGetHeaderDecayCorrection(&tac.h);
  if(fdc==DECAY_UNKNOWN) {
    if(verbose>1) printf("Note: status of current decay correction is not known.\n");
  } else if(fdc==DECAY_CORRECTED && correct==1) {
    fprintf(stderr, "Error: physical decay is already corrected in %s.\n", tacfile1);
    tacFree(&tac); return(4);
  } else if(fdc==DECAY_NOTCORRECTED && correct==0) {
    fprintf(stderr, "Error: physical decay is not corrected in %s.\n", tacfile1);
    tacFree(&tac); return(4);
  }


  /*
   *  Decay correction / removal
   */
  if(tacDecayCorrection(&tac, isot, correct, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), tacfile1);
    tacFree(&tac); return(5);
  }
  /* Set also header */
  if(correct==0) tacSetHeaderDecayCorrection(&tac.h, DECAY_NOTCORRECTED);
  else tacSetHeaderDecayCorrection(&tac.h, DECAY_CORRECTED);



  /*
   *  Save corrected file
   */
  if(verbose>1) printf("saving %s\n", tacfile2);
  {
    FILE *fp; fp=fopen(tacfile2, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing\n");
      tacFree(&tac); return(11);
    }
    int ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      tacFree(&tac); return(12);
    }
  
    /* Tell user what we did */
    if(verbose>=0) {
      if(correct) printf("  data was corrected for decay with half-life of");
      else printf("  decay correction was removed from data with half-life of");
      printf(" %s\n", isotopeName(isot));
      fflush(stdout);
    }
  }
  tacFree(&tac);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
