#!/bin/bash
#: Title      : test_tacdecay
#: Date       : 2019-09-02
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../tacdecay$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"


if [ ! -f c11_min.dft ] || [ ! -f correct_c11_min.dft ] || [ ! -f correct_c11_min_mid.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f c11_min.tac ] || [ ! -f correct_c11_min.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f f18_sec.dft ] || [ ! -f correct_f18_sec.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f o15blood.kbq ] || [ ! -f o15blood_min.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 0.1.0 \n"
printf " Test case: Decay correction option not given. \n"
printf " Expected result: Decay correction on by default. \n\n"

rm -f output.dft stdout.txt

$PROGRAM -d -i=O-15 o15blood.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt correct 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.1.1 \n"
printf " Test case: Decay correction option has invalid value. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft stdout.txt

$PROGRAM -d -i=O-15 -decay=1 o15blood.kbq output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-decay=1'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.1.2 \n"
printf " Test case: Decay correction option given. \n"
printf " Expected result: Option interpreted correctly. \n\n"

rm -f output.dft stdout.txt

$PROGRAM -d -i=O-15 -decay=off o15blood.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt correct 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dft stdout.txt

$PROGRAM -d -i=O-15 -decay=on o15blood.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt correct 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n\n===============================================================\n\n"

printf "\n 0.2.0 \n"
printf " Test case: Isotope code not given by user or in file. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft stdout.txt
cp -f o15blood.kbq temp.kbq

$PROGRAM -decay=on temp.kbq output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "valid isotope not specified."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.1 \n"
printf " Test case: Isotope code not given by user but is in file. \n"
printf " Expected result: Isotope code is correctly read from file. \n\n"

rm -f output.dft stdout.txt
cp -f o15blood.kbq temp.kbq
echo "# isotope := O-15" >> temp.kbq

$PROGRAM -d -decay=on temp.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt isotope "O-15"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dft stdout.txt
cp -f o15blood.kbq temp.kbq
echo "# isotope := C-11" >> temp.kbq

$PROGRAM -d -decay=on temp.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt isotope "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


rm -f output.dft stdout.txt
cp -f o15blood.kbq temp.kbq
printf "# isotope := F-18" >> temp.kbq

$PROGRAM -d -decay=on temp.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt isotope "F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.2 \n"
printf " Test case: Isotope code not given by user and is invalid in file. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft stdout.txt
cp -f o15blood.kbq temp.kbq
printf "# isotope := oxygen-25" >> temp.kbq

$PROGRAM -d -decay=on temp.kbq output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.3 \n"
printf " Test case: Isotope code given by user, not in file. \n"
printf " Expected result: Isotope code is correctly identified. \n\n"


rm -f output.dft stdout.txt

$PROGRAM -d -decay=on -i=O-15 o15blood.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt isotope "O-15"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


rm -f output.dft stdout.txt

$PROGRAM -d -decay=on -i=C-11 o15blood.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt isotope "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


rm -f output.dft stdout.txt

$PROGRAM -d -decay=on -i=F-18 o15blood.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt isotope "F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.4 \n"
printf " Test case: Isotope code given by user and is the same in file. \n"
printf " Expected result: Isotope code is correctly identified. \n\n"


rm -f output.dft stdout.txt
cp -f o15blood.kbq temp.kbq
printf "# isotope := O-15" >> temp.kbq

$PROGRAM -d -decay=on -i=O-15 temp.kbq output.dft &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt isotope "O-15"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.5 \n"
printf " Test case: Isotope code given by user and is different in file. \n"
printf " Expected result: Error. \n\n"


rm -f output.dft stdout.txt
cp -f o15blood.kbq temp.kbq
printf "# isotope := F-18" >> temp.kbq

$PROGRAM -d -decay=on -i=C-11 temp.kbq output.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "different isotope in temp.kbq"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n===============================================================\n\n"


printf "\n 1.0.0 \n"
printf " Test case: Input file contains one or more than one TAC. \n"
printf " Expected result: results are correct for all TACs. \n\n"

rm -f output.dat

$PROGRAM -decay=on -i=C-11 c11_min.dft output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 correct_c11_min.dft output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
tacadd -ovr temp.dft c11_min.dft 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr temp2.dft correct_c11_min.dft 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -decay=on -i=C-11 temp.dft output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 temp2.dft output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Input file contains one or more than one time frames. \n"
printf " Expected result: results are correct. \n\n"

rm -f output.dat
taccut c11_min.dft 0 21 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccut correct_c11_min.dft 0 21 temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -decay=on -i=C-11 temp.dft output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 temp2.dft output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case: Sample time unit unknown with isotope O-15. \n"
printf " Expected result: assumed to be in sec. \n\n"

rm -f output.kbq stdout.txt

$PROGRAM -d2 -decay=on -i=O-15 o15blood.kbq output.kbq &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt xunit "sec"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case: Sample time unit unknown with isotope C-11. \n"
printf " Expected result: assumed to be in min. \n\n"

rm -f output.kbq stdout.txt

$PROGRAM -d2 -decay=on -i=C-11 o15blood.kbq output.kbq &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt xunit "min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.2 \n"
printf " Test case: Sample time unit given in the file. \n"
printf " Expected result: min and sec are processed correctly. \n\n"

rm -f output1.dft stdout.txt

$PROGRAM -decay=on -i=C-11 c11_min.dft output1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 correct_c11_min.dft output1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit output1.dft > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt timeunit "min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output2.dft stdout.txt
cp -f c11_min.dft c11_sec.dft
tacunit -xconv=sec c11_sec.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -decay=on -i=C-11 c11_sec.dft output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit output2.dft > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt timeunit "sec"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -x=no -y=y -xunit=n -abs=0.01 -rel=0.1 output1.dft output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.3 \n"
printf " Test case: Sample time unit given by user. \n"
printf " Expected result: min and sec are processed correctly. \n\n"

rm -f output.kbq stdout.txt

$PROGRAM -d2 -decay=on -i=O-15 -sec o15blood.kbq output.kbq &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt xunit "sec"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.kbq stdout.txt

$PROGRAM -d2 -decay=on -i=O-15 -min o15blood.kbq output.kbq &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt xunit "min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.kbq stdout.txt

$PROGRAM -d2 -decay=on -i=C-11 -min o15blood.kbq output.kbq &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt xunit "min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.kbq stdout.txt

$PROGRAM -d2 -decay=on -i=C-11 -sec o15blood.kbq output.kbq &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt xunit "sec"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n\n===============================================================\n\n"


printf "\n 2.0.0 \n"
printf " Test case 1: Output filename not given. \n"
printf " Test case 2: Output file contains decay correction information. \n"
printf " Expected result: Input file is modified. \n\n"

rm -f stdout.txt datafile.dft

$PROGRAM -decay=on -i=F-18 f18_sec.dft datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 correct_f18_sec.dft datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval datafile.dft isotope "F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval datafile.dft decay_correction "yes"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -decay=off -i=F-18 datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 f18_sec.dft datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

iftisval datafile.dft isotope "F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval datafile.dft decay_correction "no"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n\n===============================================================\n\n"


printf "\n 3.0.0 \n"
printf " Test case: Decay correction based on frame mid times only. \n"
printf " Expected result: Correct values. \n\n"

rm -f temp1.dat datafile.dft
tacformat -format=simple -mid c11_min.dft temp1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -decay=on -i=C-11 temp1.dat datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 correct_c11_min_mid.dft datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -decay=off -i=C-11 datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -xunit=n -yunit=n -tacnames=n -abs=0.01 -rel=0.1 temp1.dat datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf " Same with PMOD format. \n"

rm -f temp1.bld datafile.bld

tacformat -format=simple -mid -hdr=no c11_min.tac temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -decay=on -i=C-11 temp1.bld datafile.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=no correct_c11_min_mid.dft datafile.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -decay=off -i=C-11 datafile.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -xunit=n -yunit=n -tacnames=y -abs=0.01 -rel=0.1 temp1.bld datafile.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.0.1 \n"
printf " Test case: Decay correction based on frame start and end times. \n"
printf " Expected result: Correct values. \n\n"

rm -f datafile.dft

$PROGRAM -decay=on -i=C-11 c11_min.dft datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 correct_c11_min.dft datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf " Same with PMOD format. \n"

rm -f datafile.tac

$PROGRAM -decay=on -i=C-11 c11_min.tac datafile.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 -tacnames=y correct_c11_min.tac datafile.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n===============================================================\n\n"


printf "\n 4.0.0 \n"
printf " Test case: Trying to do decay correction twice \n"
printf " Expected result: Error, if file contains decay correction info. \n\n"

rm -f datafile.dft stdout.txt

$PROGRAM -decay=on -i=C-11 c11_min.dft datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -decay=on -i=C-11 datafile.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "physical decay is already corrected in datafile.dft."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.1 \n"
printf " Test case: Trying to remove decay correction twice \n"
printf " Expected result: Error, if file contains decay correction info. \n\n"

rm -f stdout.txt

$PROGRAM -decay=off datafile.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -decay=off datafile.dft &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "physical decay is not corrected in datafile.dft."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

