/** @file iftadd.c
 *  @brief Inserts the items with specified key name from one
           Interfile-type file to another.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Inserts the items with specified key name from Interfile-type file2 to",
  "file1. If no key names are specified, then all items are copied.",
  "If file1 does not exist, it will be created.",
  "Notice that lines not containing key and value are deleted in file1.",
  " ",
  "Usage: @P [Options] filename1 filename2 key(s)",
  " ",
  "Options:",
  " -ovr",
  "     When key name exists, the key and and its value are replaced.",
  " -nodupl",
  "     When key name exists, the new key and its value are not inserted.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P patname.hdr iea345.hdr patient_name patient_id",
  " ",
  "See also: iftlist, iftdel, iftisval, iftvalc, iftren, nii_ehdr, e7emhdr",
  " ",
  "Keywords: header, IFT, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   ret, n=0, fki=0, keyNr;
  int   copy_mode=0; /* 0=always insert, 1=overwrite, 2=cancel if exists */
  char *cptr, iftfile1[FILENAME_MAX], iftfile2[FILENAME_MAX];
  IFT   ift1, ift2;
  FILE *fp;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftInit(&ift1); iftInit(&ift2);
  iftfile1[0]=iftfile2[0]=(char)0; 
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "OVR", 2)==0) {
      copy_mode=1; continue;
    } else if(strncasecmp(cptr, "NODUPL", 3)==0) {
      copy_mode=2; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // later arguments may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  keyNr=0;
  for(; ai<argc; ai++) {
    if(!iftfile1[0]) {strlcpy(iftfile1, argv[ai], FILENAME_MAX); continue;}
    if(!iftfile2[0]) {strlcpy(iftfile2, argv[ai], FILENAME_MAX); continue;}
    if(keyNr==0) fki=ai; 
    keyNr++;
  }

  /* Check that we got both filenames */
  if(!iftfile2[0]) {tpcPrintUsage(argv[0], info, stderr); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("iftfile1 := %s\n", iftfile1);
    printf("iftfile2 := %s\n", iftfile2);
    printf("keyNr := %d\n", keyNr);
    printf("copy_mode := %d\n", copy_mode);
  }


  /*
   *  Read the IFT files
   */

  if(verbose>1) printf("reading %s\n", iftfile1);
  fp=fopen(iftfile1, "r"); if(fp==NULL) {
    fprintf(stdout, "File %s did not exist and was created.\n", iftfile1);
  } else {
    ret=iftRead(&ift1, fp, 1, 1, &status); fclose(fp);
    if(ret) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      iftFree(&ift1); return(2);
    }
    if(verbose>2) printf("list1 size: %d item(s)\n", ift1.keyNr);
  }


  if(verbose>1) printf("reading %s\n", iftfile2);
  fp=fopen(iftfile2, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile2);
    iftFree(&ift1); return(2);
  }
  ret=iftRead(&ift2, fp, 1, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    iftFree(&ift1); iftFree(&ift2); return(2);
  }
  if(verbose>2) printf("list2 size: %d item(s)\n", ift2.keyNr);

  /* If no key names were specified by user, then select all */
  if(keyNr==0) for(int li=0; li<ift2.keyNr; li++) ift2.item[li].sw=1;

  /* If key name(s) were given, then select them */
  if(keyNr>0) {
    int n=0, i, si;
    for(ai=fki; ai<argc; ai++) {
      si=0;
      do {
        i=iftFindKey(&ift2, argv[ai], si);
        if(i>=0) {ift2.item[i].sw=1; n++; si=i+1;} 
        else if(si==0 && keyNr>1) 
          fprintf(stderr, "Warning: key '%s' was not found.\n", argv[ai]);
      } while(i>=0);
    }
    if(n==0) {
      if(keyNr==1) fprintf(stderr, "Error: key not found.\n");
      else fprintf(stderr, "Error: none of keys found.\n");
      iftFree(&ift1); iftFree(&ift2); return(3);
    }
    if(verbose>1) printf("%d key(s) found.\n", n);
  }


  /*
   *  Insert selected items
   */
  if(verbose>1) {
    if(keyNr==0) printf("inserting the contents\n");
    else printf("inserting selected contents\n");
  }
  n=0;
  for(int i=0; i<ift2.keyNr; i++) {
    if(verbose>4) printf("key := '%s'\n", ift2.item[i].key);
    if(ift2.item[i].sw==0) {
      if(verbose>4) printf("  not selected\n");
      continue;
    }
    if(verbose>4) printf("  selected\n");
    /* If required, check if key already exists */
    if(copy_mode>0) {
      if(iftFindNrOfKeys(&ift1, ift2.item[i].key)>0) {
        if(copy_mode==1) {
          if(verbose>0) printf("deleting existing key '%s'\n", ift2.item[i].key);
          iftDeleteKey(&ift1, ift2.item[i].key);
        } else {
          if(verbose>0) 
            fprintf(stderr, "Warning: existing key '%s' not added.\n", 
                    ift2.item[i].key);
          continue;
        }
      }
    }
    if(verbose>3) printf("adding item with key '%s'\n", ift2.item[i].key);
    ret=iftPut(&ift1, ift2.item[i].key, ift2.item[i].value, 
               ift2.item[i].comment, &status);
    if(ret!=TPCERROR_OK)
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    else n++;
  }
  if(n==0) {
    fprintf(stderr, "Error: no items could be inserted.\n");
    iftFree(&ift1); iftFree(&ift2);
    return(9);
  } else {
    if(verbose>0)
      fprintf(stdout, "  %d item(s) are inserted in %s\n", n, iftfile1);
  }
  iftFree(&ift2);


  /*
   *  Write the modified IFT contents.
   */
  if(verbose>1) printf("writing modified IFT in %s\n", iftfile1);
  fp=fopen(iftfile1, "w"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile1);
    iftFree(&ift1); return(11);
  }
  ret=iftWrite(&ift1, fp, &status); fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift1); return(12);
  }
  iftFree(&ift1);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
