/** @file iftlist.c
 *  @brief List ift file contents.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List on screen the specified or all key names and values in",
  "an Interfile-type (IFT) header file.",
  " ",
  "Usage: @P [options] filename [key1 [key2...]] [> outputfile]",
  " ",
  "Options:",
  " -a[ll]",
  "     List all values, also those with no key name",
  " -value",
  "     Print only the key value(s) without key name(s)",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P iea345.hdr patient_name",
  " ",
  "See also: iftadd, iftdel, iftisval, iftmatch, rmcmnts",
  " ",
  "Keywords: header, IFT, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, is_key_required=1, print_value=0;
  char *cptr, iftfile[FILENAME_MAX];
  IFT ift;
  char *key;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftInit(&ift);
  iftfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "ALL", 1)==0) {
      is_key_required=0; continue;
    } else if(strcasecmp(cptr, "VALUE")==0) {
      print_value=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // key name argument may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("is_key_required := %d\n", is_key_required);
    printf("print_value := %d\n", print_value);
  }
 
  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose;

  /* Process other arguments, starting from the first non-option */
  int n=0, m, li;
  for(; ai<argc; ai++) {
    if(!iftfile[0]) {
      strlcpy(iftfile, argv[ai], FILENAME_MAX);
      /* Read IFT file */
      if(verbose>1) printf("reading %s\n", iftfile);
      FILE *fp=fopen(iftfile, "r"); if(fp==NULL) {
        fprintf(stderr, "Error: cannot open file %s\n", iftfile);
        return(2);
      }
      ret=iftRead(&ift, fp, is_key_required, 1, &status); fclose(fp);
      if(ret) {
        fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
        iftFree(&ift); return(2);
      }
      if(verbose>1) printf("list size: %d item(s)\n", ift.keyNr);
    } else { /* Next parameters are key names; print them */
      key=strdup(argv[ai]); m=0;
      do {
        li=iftSearchKey(&ift, key, m);
        if(li<0) {
          if(m<1) fprintf(stderr, "Key \"%s\" was not found.\n", argv[ai]);
        } else {
          if(print_value==0) iftWriteItem(&ift, li, stdout, NULL);
          else fprintf(stdout, "%s\n", ift.item[li].value);
          m=li+1;
        }
      } while(li>=0);
      free(key);
      n++;
    }
  }
  /* Check that we got what we need */
  if(!iftfile[0]) {
    tpcPrintUsage(argv[0], info, stderr); iftFree(&ift); return(1);}

  /*
   *  If no key names were specified, then list all keys and their values
   */
  if(n<1) {
    if(verbose>0) printf("Parameters in %s :\n", iftfile);
    if(print_value==0) ret=iftWrite(&ift, stdout, NULL);
    else for(li=0; li<ift.keyNr; li++)
      fprintf(stdout, "%s\n", ift.item[li].value);
  }

  iftFree(&ift);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
