/** @file iftren.c
 *  @brief Renames specified key in Interfile-type file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Renames the specified key in Interfile-type file.",
  "Only the first occurrence of key name is renamed.",
  "If new key name is not given, then previous key name is deleted.",
  "Empty string can be entered as current key name to add key name for",
  "the first line that currently has no key name.",
  " ",
  "Usage: @P [Options] filename key [newkey]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P parameters.ift p1 start_time",
  " ",
  "See also: iftlist, iftedit, iftadd, iftdel, iftisval, iftvalc, iftmatch",
  " ",
  "Keywords: header, IFT, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   ret;
  char  iftfile[FILENAME_MAX], keyname1[FILENAME_MAX], keyname2[FILENAME_MAX];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftfile[0]=keyname1[0]=keyname2[0]=(char)0; 
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //char *cptr=argv[ai]+1;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // later arguments may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(iftfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(keyname1, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(keyname2, argv[ai++], FILENAME_MAX);
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing? */
  if(!keyname1[0] && !keyname2[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}
  /* Note that:
     - if keyname1 is empty, then user wants to add new key name
     - if keyname2 is empty or missing, then user wants to remove key name
     - both cannot be empty or missing
   */

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n"); fflush(stdout);
    printf("iftfile := %s\n", iftfile);
    printf("keyname1 := %s\n", keyname1);
    printf("keyname2 := %s\n", keyname2);
  }


  /*
   *  Read IFT file
   */
  IFT ift; iftInit(&ift);

  if(verbose>1) printf("reading %s\n", iftfile);
  FILE *fp=fopen(iftfile, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile);
    return(2);
  }
  ret=iftRead(&ift, fp, 0, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift); return(2);
  }
  if(verbose>2) printf("list size: %d item(s)\n", ift.keyNr);

  /*
   *  Find the key from IFT
   */
  int li;
  if(strlen(keyname1)==0) {
    /* Search for the first empty key name */
    li=-1;
    for(int i=0; i<ift.keyNr; i++) {
      if(ift.item[i].key==NULL || strlen(ift.item[i].key)==0) {li=i; break;}
    }
  } else {
    /* Search for the first occurrence of keyname */
    li=iftFindKey(&ift, keyname1, 0);
  }
  if(li<0) {
    fprintf(stderr, "Error: file contains no key '%s'\n", keyname1);
    iftFree(&ift); return(3);
  }
  if(verbose>3) printf("key_index := %d\n", li);

  /*
   *  Replace the key name
   */
  if(verbose>1) printf("renaming key '%s' -> '%s'\n", ift.item[li].key, keyname2);
  ret=iftReplaceKey(&ift, li, keyname2, &status);
  if(ret) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift); return(4);
  }


  /*
   *  Write the modified IFT contents.
   */
  if(verbose>1) printf("writing modified IFT in %s\n", iftfile);
  fp=fopen(iftfile, "w"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile);
    iftFree(&ift); return(11);
  }
  ret=iftWrite(&ift, fp, &status); fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift); return(12);
  }
  iftFree(&ift);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
