/** @file imgformat.c
 *  @brief View or set PET image file format. NOT FUNCTIONAL YET!
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpccsv.h"
#include "tpctac.h"
#include "tpcimage.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Convert PET image data in filename1 into filename2",
  "in specified file format.",
  "Data can be written only in a few formats (listed below), but more",
  "file formats can be read.",
  " ",
  "Usage: @P [options] filename1 [filename2]",
  " ",
  "Options:",
  " -f=<format-id>",
  "     Accepted format-id's:",
  "     FLAT      - Pixel values are written to binary flat file as 32-bit",
  "                 floating point values in order t,z,y,x.",
  "     NIFTI1    - NIfTI-1 single file format.",
  "     Without this option, only the format of the given image is shown.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacformat, img2flat, e63to7, ecat2ana, ecat2nii",
  " ",
  "Keywords: image, tool, format",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  int new_format=IMG_FORMAT_UNKNOWN;
  char imgfile1[FILENAME_MAX], imgfile2[FILENAME_MAX];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile1[0]=imgfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "F=", 2)==0) {
      new_format=imgFormatIdentify(cptr+2);
      if(new_format!=IMG_FORMAT_UNKNOWN) continue;
    } else if(strncasecmp(cptr, "FORMAT=", 7)==0) {
      new_format=imgFormatIdentify(cptr+7);
      if(new_format!=IMG_FORMAT_UNKNOWN) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(imgfile1, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(imgfile2, argv[ai++], FILENAME_MAX);
  if(ai<argc) {fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]); return(1);}
  /* Is something missing? */
  if(!imgfile1[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("new_format := %s\n", imgFormatDescr(new_format));
    printf("imgfile1 := %s\n", imgfile1);
    if(imgfile2[0]) printf("imgfile2 := %s\n", imgfile2);
    fflush(stdout);
  }

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /*
   *  Read image data
   */
  if(verbose>1) {printf("reading %s\n", imgfile1); fflush(stdout);}
  IMG img; imgInit(&img);
  ret=imgRead(&img, imgfile1, &status);
  if(ret) { // error
    imgFree(&img);
    /* If format conversion was requested, then this certainly is bad */
    if(new_format!=IMG_FORMAT_UNKNOWN) {
      fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), imgfile1); fflush(stderr);
      return(2);
    }
    /* User just wanted to know the current format; try from file name */
    ret=imgFormatFromFName(imgfile1, &status);
    if(ret==IMG_FORMAT_UNKNOWN) {
      fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), imgfile1); fflush(stderr);
      return(2);
    }
    fprintf(stdout, "%s\n", imgFormatDescr(ret)); fflush(stdout);
    return(0);
  }
  if(verbose>2) imgContents(&img, stdout);
  
  /* If new format was not specified, then just print the current format */
  if(new_format==IMG_FORMAT_UNKNOWN) {
    fprintf(stdout, "%s\n", imgFormatDescr(img.format)); fflush(stdout);
    imgFree(&img);
    return(0);
  }
  
  /* If output file name was not given by user, then make it here */
  if(!imgfile2[0]) {
    strcpy(imgfile2, imgfile1);
    filenameRmPath(imgfile2); filenameRmExtension(imgfile2);
    strcat(imgfile2, imgDefaultExtension(new_format));
    if(verbose>1) {
      printf("imgfile2 := %s\n", imgfile2);
    }
  }
  

  /* Try to save the data in the new format */
  if(verbose>1) {printf("writing %s\n", imgfile2); fflush(stdout);}
  img.oformat=new_format;
  ret=imgWrite(&img, imgfile2, &status);
  if(ret==TPCERROR_UNSUPPORTED) {
    fprintf(stderr, "Error: writing format %s is not supported.\n", tacFormattxt(new_format));
    return(6);
  }
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error)); fflush(stderr);
    return(7);
  }
  if(verbose>0) {printf("  %s written.\n", imgfile2); fflush(stdout);}


  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
