/** @file abss.c
 *  @brief Processing ABSS data stored in TAC struct.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcabss.h"
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the number of ABSS samples with positive counts.
   @sa tacRead, abssWrite, abssHigherCounts, abssCalculateRatio, abssFixChannel
   @return Returns <>0 in case of an error.
 */
int abssAboveZero(
  /** Pointer to TAC struct containing raw ABSS data. */
  TAC *abss,
  /** Number of positive samples in channel 1 is written in here;
      enter NULL if not needed. */
  int *n1,
  /** Number of positive samples in channel 2 is written in here;
      enter NULL if not needed. */
  int *n2,
  /** Number of samples with positive counts in either of channels;
      enter NULL if not needed. */
  int *n
) {
  if(n1!=NULL) *n1=0;
  if(n2!=NULL) *n2=0;
  if(n!=NULL)  *n=0;
  /* Check the data */
  if(abss==NULL || abss->tacNr<1 || abss->sampleNr<1) return(1);
  if(abss->format!=TAC_FORMAT_ABSS_ALLOGG && 
     abss->format!=TAC_FORMAT_ABSS_ALLOGG_OLD &&
     abss->format!=TAC_FORMAT_ABSS_GEMS &&
     abss->format!=TAC_FORMAT_ABSS_SCANDITRONICS)
  {
    return(2);
  }

  /* Set the data columns */
  int c1, c2;
  if(abss->format==TAC_FORMAT_ABSS_ALLOGG ||
     abss->format==TAC_FORMAT_ABSS_ALLOGG)
  {
    c1=0; c2=1;
  } else {
    c1=0; c2=3;
  }
  if(c2>=abss->tacNr) {return(3);}

  /* Count the samples with positive counts */
  int m1, m2, m;
  m1=m2=m=0;
  for(int i=0; i<abss->sampleNr; i++) {
    if(abss->c[c1].y[i]>0.0) m1++;
    if(abss->c[c2].y[i]>0.0) m2++;
    if(abss->c[c1].y[i]>0.0 || abss->c[c2].y[i]>0.0) m++;
  }
  if(n1!=NULL) *n1=m1;
  if(n2!=NULL) *n2=m2;
  if(n!=NULL)  *n=m;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the number of ABSS samples which are positive and higher than
    the counts from the other channel.
   @sa abssAboveZero, abssCalculateRatio, tacRead, abssWrite
   @return Returns <>0 in case of an error.
 */
int abssHigherCounts(
  /** Pointer to TAC struct containing raw ABSS data. */
  TAC *abss,
  /** Number of samples in channel 1 that are positive and higher than counts
      from channel 2. */
  int *n1,
  /** Number of samples in channel 2 that are positive and higher than counts
      from channel 1. */
  int *n2
) {
  if(n1!=NULL) *n1=0;
  if(n2!=NULL) *n2=0;
  /* Check the data */
  if(abss==NULL || abss->tacNr<1 || abss->sampleNr<1) return(1);
  if(abss->format!=TAC_FORMAT_ABSS_ALLOGG && 
     abss->format!=TAC_FORMAT_ABSS_ALLOGG_OLD &&
     abss->format!=TAC_FORMAT_ABSS_GEMS &&
     abss->format!=TAC_FORMAT_ABSS_SCANDITRONICS)
  {
    return(2);
  }

  /* Set the data columns */
  int c1, c2;
  if(abss->format==TAC_FORMAT_ABSS_ALLOGG ||
     abss->format==TAC_FORMAT_ABSS_ALLOGG)
  {
    c1=0; c2=1;
  } else {
    c1=0; c2=3;
  }
  if(c2>=abss->tacNr) {return(3);}

  /* Count the samples */
  int m1, m2, m;
  m1=m2=m=0;
  for(int i=0; i<abss->sampleNr; i++) {
    if(abss->c[c1].y[i]>abss->c[c2].y[i] && abss->c[c1].y[i]>0.0) m1++;
    if(abss->c[c2].y[i]>abss->c[c1].y[i] && abss->c[c2].y[i]>0.0) m2++;
  }
  if(n1!=NULL) *n1=m1;
  if(n2!=NULL) *n2=m2;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the number of ABSS samples with positive counts.
   @sa tacRead, abssWrite, abssFixChannel
   @return Returns <>0 in case of an error.
 */
int abssCalculateRatio(
  /** Pointer to TAC struct containing raw ABSS data. */
  TAC *abss,
  /** Channel1-to-channel2 is written in here. */
  double *ratio,
  /** Number of samples which could be used in calculation of the ratio;
      enter NULL if not needed. */
  int *n
) {
  if(ratio!=NULL) *ratio=nan("");
  if(n!=NULL)  *n=0;
  /* Check the data */
  if(abss==NULL || abss->tacNr<1 || abss->sampleNr<1) return(1);
  if(abss->format!=TAC_FORMAT_ABSS_ALLOGG && 
     abss->format!=TAC_FORMAT_ABSS_ALLOGG_OLD &&
     abss->format!=TAC_FORMAT_ABSS_GEMS &&
     abss->format!=TAC_FORMAT_ABSS_SCANDITRONICS)
  {
    return(2);
  }

  /* Set the data columns */
  int c1, c2;
  if(abss->format==TAC_FORMAT_ABSS_ALLOGG ||
     abss->format==TAC_FORMAT_ABSS_ALLOGG)
  {
    c1=0; c2=1;
  } else {
    c1=0; c2=3;
  }
  if(c2>=abss->tacNr) {return(3);}

  /* Count the cps sum for both channels */
  double s1, s2, v1, v2, fdur, r;
  s1=s2=0.0;
  int m=0;
  for(int i=0; i<abss->sampleNr; i++) {
    fdur=abss->x2[i]-abss->x1[i]; if(!(fdur>0.0)) continue;
    v1=abss->c[c1].y[i]/fdur; v2=abss->c[c2].y[i]/fdur;
    if(!isfinite(v1) || !isfinite(v2)) continue;
    s1+=v1; s2+=v2; m++;
  }
  if(m<1) {return(4);}

  /* Calculate the ratio of sums */
  r=s1/s2; if(!isfinite(r)) {return(5);}  
  if(ratio!=NULL) *ratio=r;
  if(n!=NULL) *n=m;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the number of ABSS samples with positive counts.
   @sa abssAboveZero, abssCalculateRatio, tacRead, abssWrite, abssCalculateRatio
   @return Returns <>0 in case of an error.
 */
int abssFixChannel(
  /** Pointer to TAC struct containing raw ABSS data. */
  TAC *abss,
  /** Channel to fix; either 1 or 2. */
  int channel,
  /** Correct channel1-to-channel2 ratio. */
  double ratio
) {
  /* Check the data */
  if(abss==NULL || abss->tacNr<1 || abss->sampleNr<1) return(1);
  if(abss->format!=TAC_FORMAT_ABSS_ALLOGG && 
     abss->format!=TAC_FORMAT_ABSS_ALLOGG_OLD &&
     abss->format!=TAC_FORMAT_ABSS_GEMS &&
     abss->format!=TAC_FORMAT_ABSS_SCANDITRONICS)
  {
    return(2);
  }
  if(channel!=1 && channel!=2) return(3);
  if(!isfinite(ratio) || ratio<=0.0) return(4);

  /* Set the data columns */
  int c1, c2;
  if(abss->format==TAC_FORMAT_ABSS_ALLOGG ||
     abss->format==TAC_FORMAT_ABSS_ALLOGG)
  {
    c1=0; c2=1;
  } else {
    c1=0; c2=3;
  }
  if(c2>=abss->tacNr) {return(5);}

  /* Fix the samples */
  double f;
  if(channel==1) {
    f=ratio;
    for(int i=0; i<abss->sampleNr; i++)
      abss->c[c1].y[i]=f*abss->c[c2].y[i];
  } else {
    f=1.0/ratio;
    for(int i=0; i<abss->sampleNr; i++)
      abss->c[c2].y[i]=f*abss->c[c1].y[i];
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Divide the ABSS coincident counts by frame duration.
   @sa tacRead, abssWrite, abssCalculateRatio, abssFixChannel
   @return Returns <>0 in case of an error.
 */
int abssCalculateCps(
  /** Pointer to TAC struct containing raw ABSS data. */
  TAC *abss
) {
  /* Check the data */
  if(abss==NULL || abss->tacNr<1 || abss->sampleNr<1) return(1);
  if(abss->format!=TAC_FORMAT_ABSS_ALLOGG && 
     abss->format!=TAC_FORMAT_ABSS_ALLOGG_OLD &&
     abss->format!=TAC_FORMAT_ABSS_GEMS &&
     abss->format!=TAC_FORMAT_ABSS_SCANDITRONICS)
  {
    return(2);
  }

  /* Set the data columns */
  int c1, c2;
  if(abss->format==TAC_FORMAT_ABSS_ALLOGG ||
     abss->format==TAC_FORMAT_ABSS_ALLOGG)
  {
    c1=0; c2=1;
  } else {
    c1=0; c2=3;
  }
  if(c2>=abss->tacNr) {return(3);}

  /* Count the cps for both channels */
  double fdur;
  for(int i=0; i<abss->sampleNr; i++) {
    fdur=abss->x2[i]-abss->x1[i]; if(!(fdur>0.0)) continue;
    abss->c[c1].y[i]/=fdur;
    abss->c[c2].y[i]/=fdur;
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Compute the mean of the two channels in Scanditronics or GEMS ABSS data
    stored in TAC struct. 

    If data is collected with Allogg, then column two
    is copied to 'mean' as such. 
    Data is not divided by frame duration in this function.

   @sa tacRead, abssCalculateCps, abssCalculateRatio
   @return Returns <>0 in case of an error.
 */
int abssChannelMean(
  /** Pointer to TAC struct containing raw ABSS data. */
  TAC *abss,
  /** Pointer to double array, at least of length abss->sampleNr. */
  double *mean
) {
  /* Check the data */
  if(abss==NULL || abss->tacNr<1 || abss->sampleNr<1) return(1);
  if(abss->format!=TAC_FORMAT_ABSS_ALLOGG && 
     abss->format!=TAC_FORMAT_ABSS_ALLOGG_OLD &&
     abss->format!=TAC_FORMAT_ABSS_GEMS &&
     abss->format!=TAC_FORMAT_ABSS_SCANDITRONICS)
  {
    return(2);
  }
  if(mean==NULL) return(3);

  /* Set the data columns */
  int c1, c2;
  if(abss->format==TAC_FORMAT_ABSS_ALLOGG ||
     abss->format==TAC_FORMAT_ABSS_ALLOGG)
  {
    c1=0; c2=1;
  } else {
    c1=0; c2=3;
  }
  if(c2>=abss->tacNr) {return(3);}

  /* If Allogg, the just copy, and then return */
  if(abss->format==TAC_FORMAT_ABSS_ALLOGG ||
     abss->format==TAC_FORMAT_ABSS_ALLOGG)
  {
    for(int i=0; i<abss->sampleNr; i++) mean[i]=abss->c[c2].y[i];
    return(0);
  }

  /* Otherwise, compute the average for each sample frame */
  for(int i=0; i<abss->sampleNr; i++) {
    mean[i]=0.5*(abss->c[c1].y[i]+abss->c[c2].y[i]);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
