/** @file bf_srtm.c
 *  @brief BFM for SRTM.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
#include "tpccm.h"
/*****************************************************************************/
#include "tpcbfm.h"
/*****************************************************************************/

/*****************************************************************************/
/** Calculate set of basis functions for SRTM.
 *  @todo Add tests.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int bfmSRTM(
  /** Pointer to array containing PET sample (frame middle) times. */
  double *t,
  /** Pointer to array containing integral of reference tissue input 
   *  concentration values at each sample time, AUC 0-t. AUC must have been 
   *  calculated using values that are NOT corrected for decay. */
  double *cri,
  /** Nr of samples (array lengths) */
  const int n,
  /** Nr of basis functions to calculate */
  const int bfNr,
  /** Minimum of theta3. */
  const double t3min,
  /** Maximum of theta3. */
  const double t3max,
  /** Pointer to output TAC, to be allocated and filled with basis functions 
   *  in here. */
  TAC *bf,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(t, cri, %d, %d, %g, %g, bf)\n", __func__,
                       n, bfNr, t3min, t3max);
  if(t==NULL || cri==NULL || bf==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(n<2 || bfNr<1) {
    if(verbose>1) printf("invalid sample or BF number\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_TOO_FEW);
    return TPCERROR_TOO_FEW;
  }
  if(t3min<1.0E-10 || t3min>=t3max) { // range cannot be computed otherwise
    if(verbose>1) printf("invalid theta3 range\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  int ret;

  /* Allocate memory for basis functions */
  if(verbose>1) printf("allocating memory for basis functions\n");
  tacFree(bf);
  ret=tacAllocate(bf, n, bfNr);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }
  /* Copy and set information fields */
  bf->tacNr=bfNr; bf->sampleNr=n;
  bf->format=TAC_FORMAT_TSV_UK;
  bf->isframe=0;
  for(int bi=0; bi<bf->tacNr; bi++) sprintf(bf->c[bi].name, "B%5.5d", bi+1);
  for(int fi=0; fi<bf->sampleNr; fi++) bf->x[fi]=t[fi];
  /* Compute theta3 values to size fields */
  {
    if(verbose>1) printf("computing theta3 values\n");
    double a, b, c;
    a=log10(t3min); b=log10(t3max); c=(b-a)/(double)(bfNr-1);
    for(int bi=0; bi<bf->tacNr; bi++) {
      bf->c[bi].size=pow(10.0, (double)bi*c+a);
    }
  }
  if(verbose>2) {
    printf("bf_t3_range := %g - %g\n", bf->c[0].size, bf->c[bf->tacNr-1].size);
    printf("bf_t3_gaps := %g - %g\n", bf->c[1].size-bf->c[0].size, 
           bf->c[bf->tacNr-1].size-bf->c[bf->tacNr-2].size);
  }

  /* Calculate the basis functions */
  if(verbose>1) printf("computing basis functions\n");
  ret=0;
  for(int bi=0; bi<bf->tacNr && !ret; bi++) {
    if(verbose>99) printf("   theta3=%g\n", bf->c[bi].size);
    ret=simC1_i(t, cri, n, 1.0, bf->c[bi].size, bf->c[bi].y);
  }
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
