/** @file sim2cm.c
 *  @brief Simulation of 2-tissue compartmental models.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpccm.h"
/*****************************************************************************/

/*****************************************************************************/
/** Simulate tissue TAC using two-tissue compartment model and plasma TAC, at plasma TAC times.
     
    @details
    Memory for ct must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated memory for cta and/or 
    ctb can be given; if compartmental TACs are not required, NULL pointer can be given instead.
  
    The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
   
    @return Function returns 0 when successful, else a value >= 1.
    @author Vesa Oikonen
    @sa simC2_i, simC1, simC1DI, simC2l, simC3s, simC3p, simSRTM, simSamples, simDispersion,
     parExampleTTACs, parExamplePerfectBolus
 */
int simC2(
  /** Array of time values. */
  double *t,
  /** Array of arterial activities. */
  double *ca,
  /** Number of values in TACs. */
  const int nr,
  /** Rate constant of the model. */
  const double k1,
  /** Rate constant of the model. */
  const double k2,
  /** Rate constant of the model. */
  const double k3,
  /** Rate constant of the model. */
  const double k4,
  /** Pointer for TAC array to be simulated; must be allocated. */
  double *ct,
  /** Pointer for 1st compartment TAC to be simulated, or NULL. */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL. */
  double *ctb
) {
  int i;
  double dt2, r, u, v;
  double cai, ca_last, t_last;
  double ct1, ct1_last, ct2, ct2_last;
  double ct1i, ct1i_last, ct2i, ct2i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(t==NULL || ca==NULL || ct==NULL) return 2;

  /* Check parameters */
  if(k1<0.0) return 3;

  /* Calculate curves */
  t_last=0.0; if(t[0]<t_last) t_last=t[0];
  cai=ca_last=0.0;
  ct1_last=ct2_last=ct1i_last=ct2i_last=0.0;
  ct1=ct2=ct1i=ct2i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* Calculate partial results */
      r=1.0+k4*dt2;
      u=ct1i_last+dt2*ct1_last;
      v=ct2i_last+dt2*ct2_last;
      /* 1st tissue compartment and its integral */
      ct1 = ( k1*cai - (k2 + (k3/r))*u + (k4/r)*v ) / ( 1.0 + dt2*(k2 + (k3/r)) );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - k4*v) / r;
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
    }
    /* copy values to argument arrays */
    ct[i]=ct1+ct2;
    if(cta!=NULL) cta[i]=ct1;
    if(ctb!=NULL) ctb[i]=ct2;
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulate tissue TAC using two-tissue compartment model and plasma TAC, at plasma TAC times.
     
    @details
    Memory for ct must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated memory for cta and/or ctb 
    can be given; if compartmental TACs are not required, NULL pointer can be given instead.
  
    The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
     
    This version uses integral of arterial TAC as input function.
    Only advantage over simC2() is that the calculation of integral can be
    fully controlled and possibly more precise in some situations.
   
    @return Function returns 0 when successful, else a value >= 1.
    @author Vesa Oikonen
    @sa simC2, simC1, simC1_i, simC3s, simC3p, simSamples, simDelay
 */
int simC2_i(
  /** Array of time values. */
  double *t,
  /** Array of AUC 0-t of arterial activities. */
  double *cai,
  /** Number of values in TACs. */
  const int nr,
  /** Rate constant of the model. */
  const double k1,
  /** Rate constant of the model. */
  const double k2,
  /** Rate constant of the model. */
  const double k3,
  /** Rate constant of the model. */
  const double k4,
  /** Pointer for TAC array to be simulated; must be allocated. */
  double *ct,
  /** Pointer for 1st compartment TAC to be simulated, or NULL. */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL. */
  double *ctb
) {
  int i;
  double dt2, r, u, v;
  double t_last;
  double ct1, ct1_last, ct2, ct2_last;
  double ct1i, ct1i_last, ct2i, ct2i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(t==NULL || cai==NULL || ct==NULL) return 2;

  /* Check parameters */
  if(k1<0.0) return 3;

  /* Calculate curves */
  t_last=0.0; if(t[0]<t_last) t_last=t[0];
  ct1_last=ct2_last=ct1i_last=ct2i_last=0.0;
  ct1=ct2=ct1i=ct2i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* Calculate partial results */
      r=1.0+k4*dt2;
      u=ct1i_last+dt2*ct1_last;
      v=ct2i_last+dt2*ct2_last;
      /* 1st tissue compartment and its integral */
      ct1 = ( k1*cai[i] - (k2 + (k3/r))*u + (k4/r)*v ) / ( 1.0 + dt2*(k2 + (k3/r)) );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - k4*v) / r;
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
    }
    /* copy values to argument arrays */
    ct[i]=ct1+ct2;
    if(cta!=NULL) cta[i]=ct1;
    if(ctb!=NULL) ctb[i]=ct2;
    /* prepare to the next loop */
    t_last=t[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
