/** @file tpccsv.h
 *  @brief Header file for library libtpccsv.
 *  @details CSV and TSV data processing and file i/o
 *  @author Vesa Oikonen
 *  @copyright (c) Turku PET Centre
 */
#ifndef _TPCCSV_H_
#define _TPCCSV_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** One field of CSV data. */
typedef struct {
  /** Position in CSV: row index */
  int row;
  /** Position in CSV: column index */
  int col;
  /** Pointer to field content */
  char *content;
} CSV_item;

/** Main CSV content.
   @sa CSV_ITEM, IFT, TAC, PAR, csvInit, csvFree
 */
typedef struct {
  /** List of CSV/TSV data fields. */
  CSV_item *c;
  /** Length of list allocated for data fields (do not use directly). */
  int _item_nr;
  /** Total nr of data fields. */
  int nr;
  /** Nr of CSV rows. */
  int row_nr;
  /** Max column number per row. */
  int col_nr;
  /** CSV or TSV field separator (',', ';', or '\\t'); tabulator by default;
      space may also be supported by some functions. */
  char separator;
} CSV;
/*****************************************************************************/

/*****************************************************************************/
/* csv.c */
void csvInit(CSV *csv);
void csvFree(CSV *csv);
int csvAllocate(CSV *csv, int nr);
int csvDuplicate(CSV *csv1, CSV *csv2);
int csvPutString(CSV *csv, const char *s, int newline);
int csvPutDouble(CSV *csv, double v, int newline, int tointl);
int csvPutInt(CSV *csv, int v, int newline);
int csvRowLength(CSV *csv, int row);
int csvSetDimensions(CSV *csv);
int csvIsRegular(CSV *csv);
int csvTrimRight(CSV *csv);
char* csvCell(CSV *csv, int row, int col);
int csvCellReplace(CSV *csv, int row, int col, const char *s);
int csvRemoveItem(CSV *csv, int i);
int csvRemoveEmptyRows(CSV *csv);
int csvRemoveComments(CSV *csv);
int csvReorg(CSV *d, TPCSTATUS *status);
int csvTranspose(CSV *d);
/*****************************************************************************/

/*****************************************************************************/
/* csvio.c */
int csvList(CSV *csv, FILE *fp);
int csvWrite(CSV *csv, int regular, FILE *fp, TPCSTATUS *status);
int csvRead(CSV *csv, FILE *fp, TPCSTATUS *status);
int csvPutLine(CSV *csv, const char *line, TPCSTATUS *status);
int csvPutLineWithSpaces(CSV *csv, const char *line, TPCSTATUS *status);
int csvCleanSpaces(CSV *csv);
/*****************************************************************************/

/*****************************************************************************/
/* csvfind.c */
int csvFindField(CSV *csv, const char *s, int start_index);
int csvSearchField(CSV *csv, const char *s, int start_index);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCCSV */
