/** @file dcmsop.c
    @brief DICOM Storage SOPs.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
/*! @cond PRIVATE */
/** One item for table of DICOM SOPs. */
typedef struct DCM_SOP_ITEM {
  /** Pointer to the NULL terminated SOP UID string. 
      See DICOM tag (0008,0016) for SOP Class UID. */
  char uid[64];
  /** SOP Name; NULL terminated string. */
  char name[64];
} DCM_SOP_ITEM;

/** Table of DICOM Storage SOPs. 
    @note Table can not be accessed directly outside the c file.
*/
static DCM_SOP_ITEM dcm_sop[]={
  {"invalid", "invalid SOP"}, // do not change this
  {"1.2.840.10008.5.1.4.1.1.1", "Computed Radiography Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.12.1", "X-Ray Angiographic Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.128", "Positron Emission Tomography Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.130", "Enhanced PET Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.128.1", "Legacy Converted Enhanced PET Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.2", "CT Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.20", "NM Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.30", "Parametric Map Storage"},
  {"1.2.840.10008.5.1.4.1.1.3.1", "Ultrasound Multiframe Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.4", "MR Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.4.1", "Enhanced MR Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.4.2", "MR Spectroscopy Storage"},
  {"1.2.840.10008.5.1.4.1.1.6.1", "Ultrasound Image Storage"},
  {"1.2.840.10008.5.1.4.1.1.66", "Raw Data Storage"},
  {"1.2.840.10008.5.1.4.1.1.66.1", "Spatial Registration Storage"},
  {"1.2.840.10008.5.1.4.1.1.66.2", "Spatial Fiducials Storage"},
  {"1.2.840.10008.5.1.4.1.1.66.3", "Deformable Spatial Registration Storage"}, 
  {"1.2.840.10008.5.1.4.1.1.66.4", "Segmentation Storage"},
  {"unknown", "unknown SOP"} // do not change this 
};
/*! @endcond */
/*****************************************************************************/

/*****************************************************************************/
/** Identify the DICOM SOP UID.
 *  @return Returns the SOP UID list index.
 *  @sa dcmSOPName, dcmSOPUID
 */
unsigned int dcmSOPIdentify(
  /** SOP UID string. */
  const char *s
) {
  if(s==NULL || strnlen(s, 3)<3) return(0);

  /* Identify the SOP UID */
  unsigned int i=0;
  while(strcmp(dcm_sop[i].uid, "unknown")!=0) {
    if(strcmp(dcm_sop[i].uid, s)==0) return(i);
    i++;
  }
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM SOP UID Name.
 *  @return Returns pointer to the SOP Name string.
 *  @sa dcmSOPIdentify, dcmSOPUID
 */
char *dcmSOPName(
  /** SOP UID list index. */ 
  unsigned int i
) {
  unsigned int j=0;
  while(strcmp(dcm_sop[j].uid, "unknown")!=0) {
    if(i==j) return(dcm_sop[j].name);
    j++;
  }
  return(dcm_sop[j].name);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM SOP UID.
 *  @return Returns pointer to the UID string.
 *  @sa dcmSOPIdentify, dcmSOPName
 */
char *dcmSOPUID(
  /** SOP UID list index. */ 
  unsigned int i
) {
  unsigned int j=0;
  while(strcmp(dcm_sop[j].uid, "unknown")!=0) {
    if(i==j) return(dcm_sop[j].uid);
    j++;
  }
  return(dcm_sop[j].uid);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the name of DICOM SOP UID.
 *  @return Returns pointer to the SOP UID name string.
 *  @sa dcmSOPName, dcmSOPUID
 */
char *dcmSOPUIDName(
  /** SOP UID string. */
  const char *s
) {
  if(s==NULL || strnlen(s, 3)<3) return(dcm_sop[0].name);

  /* Identify the SOP UID */
  unsigned int i=0;
  while(strcmp(dcm_sop[i].uid, "unknown")!=0) {
    if(strcmp(dcm_sop[i].uid, s)==0) return(dcm_sop[i].name);
    i++;
  }
  return(dcm_sop[i].name);
}
/*****************************************************************************/

/*****************************************************************************/
