/** @file ecatheader.c
    @brief Reading and writing ECAT headers.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/
#include "tpcecat.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write ECAT main header from IFT struct into data block.
    @remark Stub function.
    @sa ecatReadMainheader, ecatVerifyMagic
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int ecatWriteMainheader(
  /** Pointer to IFT struct containing main header fields; contents are not modified. */
  IFT *ift,
  /** Pointer to data block of size ECATBLKSIZE bytes. */
  unsigned char *buf,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}

  if(ift==NULL || buf==NULL)  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return(TPCERROR_FAIL);
  }
  if(ift->keyNr<1)  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_MISSING_DATA);
    return(TPCERROR_MISSING_DATA);
  }

  /* Is current platform little endian (1) or not (0) ? */
  int little=endianLittle();
  if(verbose>1) {
    if(little) printf("little endian platform\n");
    else printf("big endian platform\n");
  }

  /* Determine ECAT format version */
  int format=6;
  /* If ECAT 7 magic_number not found, then assume ECAT 6 */
  int ii=iftFindKey(ift, "magic_number", 0);
  if(ii>=0) {
    if(strncmp(ift->item[ii].value, "MATRIX72v", 9)==0 || 
       strncmp(ift->item[ii].value, "MATRIX7011", 10)==0) format=7;
  }
  if(verbose>1) {printf("format := %d\n", format); fflush(stdout);}

  /* Copy the header fields; byte swapping when necessary */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_MISSING_DATA);
  char tmp[512];

  if(format==7) {

    ii=iftFindKey(ift, "magic_number", 0);
    if(ii>=0) strlcpy(tmp, ift->item[ii].value, 15); else strcpy(tmp, "MATRIX72v");
    memcpy((char*)buf+0, tmp, 14);

    ii=iftFindKey(ift, "original_file_name", 0);
    if(ii>=0) strlcpy(tmp, ift->item[ii].value, 33); else strcpy(tmp, "");
    memcpy((char*)buf+14, tmp, 32);

  } else { // ECAT 6

    ii=iftFindKey(ift, "magic_number", 0);
    if(ii>=0) strlcpy(tmp, ift->item[ii].value, 15); else strcpy(tmp, "MATRIX6");
    memcpy((char*)buf+0, tmp, 14);

    ii=iftFindKey(ift, "original_file_name", 0);
    if(ii>=0) strlcpy(tmp, ift->item[ii].value, 21); else strcpy(tmp, "");
    memcpy((char*)buf+28, tmp, 20);

  }


  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read ECAT main header from data block into IFT struct.
    @remark Stub function.
    @sa ecatWriteMainheader, ecatVerifyMagic
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int ecatReadMainheader(
  /** Pointer to data block containing main header. */
  const unsigned char *buf,
  /** Pointer to initiated IFT struct; any previous contents are NOT deleted. */
  IFT *ift,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}

  if(buf==NULL || ift==NULL)  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return(TPCERROR_FAIL);
  }

  /* Is current platform little endian (1) or not (0) ? */
  int little=endianLittle();
  if(verbose>1) {
    if(little) printf("little endian platform\n");
    else printf("big endian platform\n");
  }

  /* Check if the start matches the ECAT 7 magic numbers */
  int format=6;
  if(strncmp((char*)buf, "MATRIX72v", 9)==0 || strncmp((char*)buf, "MATRIX7011", 10)==0) format=7;
  if(verbose>1) {printf("format := %d\n", format); fflush(stdout);}

  /* Copy the header fields to IFT; byte swapping when necessary */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
  char tmp[512];

  if(format==7) {

    strlcpy(tmp, (char*)buf+0, 15);
    if(iftPut(ift, "magic_number", tmp, 0, NULL)!=0) return(TPCERROR_INVALID_FORMAT);
    strlcpy(tmp, (char*)buf+14, 33);
    if(iftPut(ift, "original_file_name", tmp, 0, NULL)!=0) return(TPCERROR_INVALID_FORMAT);

  } else { // ECAT 6

    strlcpy(tmp, (char*)buf+0, 15);
    if(iftPut(ift, "magic_number", tmp, 0, NULL)!=0) return(TPCERROR_INVALID_FORMAT);
    strlcpy(tmp, (char*)buf+28, 21);
    if(iftPut(ift, "original_file_name", tmp, 0, NULL)!=0) return(TPCERROR_INVALID_FORMAT);

  }


  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
