/** @file floatutil.c
 *  @brief Working with floats.
 *  @author Vesa Oikonen
 */
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <math.h>
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Verifies that given two floats have the same value inside given limits.
    Values are considered to match also if both are NaNs.
    @return 1 if match is found, and 0 if not.
    @author Vesa Oikonen
    @sa doubleMatch, floatMatchRel, floatMachEps, atofCheck
 */
int floatMatch(
  /** First value */
  const float v1,
  /** Second value */
  const float v2,
  /** Limit for absolute difference (if <0 then test will fail every time) */
  const float lim
) {
  if(isnan(v1) && isnan(v2)) return 1;
  if(isnan(v1) || isnan(v2)) return 0;
  if(v1==v2) return 1;
  if(isnan(lim) || lim<0.0) return 0;
  if(fabsf(v1-v2)<=lim) return 1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Verifies that given two floats have the same value inside given relative
    limit |2*(v1-v2)/(v1+v2)|.

    Values are considered to match also if both are NaNs, but not if mean
    is zero (unless both are exactly zero).
    @return 1 if match is found, and 0 if not.
    @author Vesa Oikonen
    @sa doubleMatchRel, floatMatch, floatMachEps
 */
int floatMatchRel(
  /** First value. */
  const float v1,
  /** Second value. */
  const float v2,
  /** Limit for relative difference (if <0 then test will fail every time). */
  const float lim
) {
  if(isnan(v1) && isnan(v2)) return 1;
  if(isnan(v1) || isnan(v2)) return 0;
  if(v1==v2) return 1;
  if(isnan(lim)) return 0;
  float mean;
  mean=0.5*(v1+v2); if(!isnormal(mean)) return 0;
  if(fabsf((v1-v2)/mean)<=lim) return 1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Estimates the machine epsilon, the upper bound on the relative error due to rounding in 
    floating point arithmetic, within one order of magnitude of the true machine epsilon.

    Standard C library should also have FLT_EPSILON and DBL_EPSILON in float.h.
    @return Estimate of machine epsilon.
    @author Vesa Oikonen
    @sa doubleMachEps, floatMatchRel, floatMatch
 */
float floatMachEps()
{
  float macheps=1.0;
  do {macheps/=2.0;} while((1.0+macheps/2.0)!=1.0);
  return(macheps);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy double values from the 2nd array to the first.
    @sa doubleCopy
 */
void floatCopy(
  /** Target array. */
  float *t,
  /** Source array. */
  float *s,
  /** Length of arrays. */
  const unsigned int n
) {
  unsigned int i;
  if(t==NULL || s==NULL || n<1) return;
  for(i=0; i<n; i++) t[i]=s[i];
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the maximum value in given float array.
    @return The index [0..n-1] of maximum value in array; 0 is returned also in case of errors.
    @sa doubleMaxIndex, floatSpanPositives, floatCSpanPositives
 */
unsigned int floatMaxIndex(
  /** Pointer to float array. */
  float *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(0);
  unsigned int i, mi=0;
  float mv=nanf("");
  for(i=0; i<n; i++) if(isnan(mv) || a[i]>mv) {mv=a[i]; mi=i;}
  return(mi);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the sum of values in given float array.
    @return The sum of array values.
    @sa doubleSum, floatMean, floatMaxIndex
 */
float floatSum(
  /** Pointer to float array. */
  float *a,
  /** Length of array. */
  const unsigned int n
) {
  float s=0.0;
  if(a==NULL || n<1) return(s);
  for(unsigned int i=0; i<n; i++) if(!isnan(a[i])) s+=a[i];
  return(s);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the mean of values in given float array.
    @return The mean of array values, or NaN in case of an error.
    @sa doubleMean, floatSum, floatMaxIndex
 */
float floatMean(
  /** Pointer to float array. */
  float *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(nanf(""));
  float s=0.0;
  unsigned int i, ci=0;
  for(i=0; i<n; i++) if(!isnan(a[i])) {ci++; s+=a[i];}
  if(ci<1) return(nanf(""));
  return(s/(float)ci);
}
/*****************************************************************************/

/*****************************************************************************/
/** Reads numerical value and its unit from argument string.

    Both decimal point and comma are accepted.
    Optional result float value is set to NaN if string was not valid value.
    Optional unit is set to UNIT_UNKNOWN, if not valid or not found.
    @return 0 if numerical value successfully read, and 1 in case of an error.
    @sa doubleGetWithUnit, atofVerified, atofCheck
 */
int floatGetWithUnit(
  /** String from where float and unit is read; string must not contain any
      extra space characters. */
  const char *s,
  /** Pointer to the float; enter NULL, if not needed. */
  float *v,
  /** Pointer to int for unit code; enter NULL, if not needed. */
  int *u
) {
  if(v!=NULL) *v=nanf(""); 
  if(u!=NULL) *u=UNIT_UNKNOWN; 
  if(s==NULL) return 1;
  int n=strTokenNr(s, " \t"); if(n==0 || n>2) return 1;
  if(n==1) { // no unit
    double dv;
    if(atofCheck(s, &dv)) return 1;
    *v=(float)dv; return 0;
  }
  /* separate number and unit */
  char buf[64];
  n=strTokenNCpy(s, " \t", 1, buf, 64);
  double dv;
  if(atofCheck(buf, &dv)) return 1;
  *v=(float)dv;
  if(u==NULL) return 0;
  n=strTokenNCpy(s, " \t", 2, buf, 64);
  *u=unitIdentify(buf);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the length of array consisting of only positive (>0 and not NaN) values.
   @return Returns the index of first nonpositive value in the given array.
   @sa doubleSpanPositives, floatCSpanPositives, atofList, floatNonzeroes
 */
int floatSpanPositives(
  /** Pointer to the array. */
  float *a,
  /** Length of the array. */
  const int n
) {
  if(a==NULL || n<1) return(0);
  int i=0;
  for(i=0; i<n; i++) if(!(a[i]>0.0)) break;
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the length of array consisting of non-positive (<=0 and NaN) values.
   @return Returns the index of first positive value in the given array.
   @sa doubleCSpanPositives, floatSpanPositives, atofList
 */
int floatCSpanPositives(
  /** Pointer to the array. */
  float *a,
  /** Length of the array. */
  const int n
) {
  if(a==NULL || n<1) return(0);
  int i=0;
  for(i=0; i<n; i++) if(a[i]>0.0) break;
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the number of non-zero values in given float array.
    Negative values are counted, but NaNs are not.
    @return The number of non-zero values in array; 0 is returned also in case of errors.
    @sa doubleNonzeroes, floatSpanPositives
 */
unsigned int floatNonzeroes(
  /** Pointer to float array. */
  float *a,
  /** Length of array. */
  const unsigned int n
) {
  if(a==NULL || n<1) return(0);
  unsigned int i, m=0;
  for(i=0; i<n; i++) if(fabsf(a[i])>0.0) m++;
  return(m);
}
/*****************************************************************************/

/*****************************************************************************/
