/** @file iftget.c
 *  @brief Get processed contents from IFT.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
/** Read the value in IFT as a float from given item index. 

    Value in IFT must not contain decimal comma.

    @sa iftGetDouble, iftFindKey, iftGetInt, iftSearchKey
    @return 0 if successful, <>0 in case of an error.
    @author Vesa Oikonen
 */
int iftGetFloat(
  /** Pointer to IFT */
  IFT *ift,
  /** IFT item index */
  int index,
  /** Pointer to float which is read from item value. It will be set to NaN
   *  in case of an error. */
  float *v
) {
  if(v!=NULL) *v=nanf(""); else return -2;
  if(ift==NULL) return -2;
  if(index<0 || index>=ift->keyNr) return -2;
  if(ift->item[index].value==NULL) return -1;
  if(sscanf(ift->item[index].value, "%f", v)!=1 || isnan(*v)) return -1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read the value in IFT as a double from given item index. 

    Value in IFT must not contain decimal comma.

    @sa iftGetFloat, iftGetDoubleWithUnit, iftFindKey, iftPutDouble
    @return 0 if successful, <>0 in case of an error.
    @author Vesa Oikonen
 */
int iftGetDouble(
  /** Pointer to IFT */
  IFT *ift,
  /** IFT item index */
  int index,
  /** Pointer to double which is read from item value. It will be set to NaN
   *  in case of an error. */
  double *v
) {
  if(v!=NULL) *v=nan(""); else return -2;
  if(ift==NULL) return -2;
  if(index<0 || index>=ift->keyNr) return -2;
  if(ift->item[index].value==NULL) return -1;
  if(sscanf(ift->item[index].value, "%lf", v)!=1 || isnan(*v)) return -1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read the value in IFT as an integer from given item index. 
    @sa iftGetUInt, iftGetDouble, iftFindKey, iftGetUInt, iftPutInt
    @return 0 if successful, <>0 in case of an error.
    @author Vesa Oikonen
 */
int iftGetInt(
  /** Pointer to IFT */
  IFT *ift,
  /** IFT item index */
  int index,
  /** Pointer to int which is read from item value. It will be set to -9999
   *  in case of an error. */
  int *v
) {
  if(v!=NULL) *v=-9999; else return -2;
  if(ift==NULL) return -2;
  if(index<0 || index>=ift->keyNr) return -2;
  if(ift->item[index].value==NULL) return -1;
  if(sscanf(ift->item[index].value, "%d", v)!=1 || *v==-9999) return -1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read the value in IFT as an unsigned integer from given item index. 

    @sa iftGetDouble, iftFindKey, iftSearchKey, iftSearchValue, iftGetInt, iftPutUint
    @return 0 if successful, <>0 in case of an error.
    @author Vesa Oikonen
 */
int iftGetUInt(
  /** Pointer to IFT */
  IFT *ift,
  /** IFT item index */
  int index,
  /** Pointer to unsigned int which is read from item value.
   *  It will be set to 0 in case of an error. */
  unsigned int *v
) {
  if(v!=NULL) *v=0; else return -2;
  if(ift==NULL) return -2;
  if(index<0 || index>=ift->keyNr) return -2;
  if(ift->item[index].value==NULL) return -1;
  if(sscanf(ift->item[index].value, "%ud", v)!=1) return -1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read the value in IFT as a double with unit from given item index. 

    Value in IFT is allowed to be written with decimal comma, and units may
    be written in pars () or [].

    @sa iftGetDouble, iftFindKey, iftSearchKey, iftSearchValue
    @return 0 if at least the double was found, otherwise <>0.
    @author Vesa Oikonen
 */
int iftGetDoubleWithUnit(
  /** Pointer to IFT */
  IFT *ift,
  /** IFT item index */
  int index,
  /** Pointer to double which is read from item value. It will be set to NaN
   *  in case of an error. */
  double *v,
  /** Pointer for unit (enum unit) which is read from item value. It will be
   *  to 0 (enum UNIT_UNKNOWN) if not identified or not present. */
  int *unit
) {
  if(v!=NULL) *v=nan("");
  if(unit!=NULL) *unit=UNIT_UNKNOWN;
  if(ift==NULL) return(2);
  if(index<0 || index>=ift->keyNr) return(1);
  if(ift->item[index].value==NULL) return(2);
  char s[128];
  if(strTokenNCpy(ift->item[index].value, " \t", 1, s, 128)==0) return(2);
  if(v!=NULL) *v=atofVerified(s);
  if(unit==NULL) return(0);
  if(strTokenNCpy(ift->item[index].value, " \t", 2, s, 128)==0) return(0);
  strCleanPars(s); *unit=unitIdentify(s);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
