/** @file iftio.c
 *  @brief IFT file i/o functions.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write one item in IFT to the specified file pointer. 

    Use iftWrite() to write all IFT items.

    @sa iftWrite, iftInit, iftFree
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */
int iftWriteItem(
  /** Pointer to IFT. */
  IFT *ift,
  /** Index [0..keyNr-1] of key and value to print. */
  int item,
  /** Output file pointer. */
  FILE *fp,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
  if(ift==NULL) return TPCERROR_FAIL;
  int verbose=0; if(status!=NULL) verbose=status->verbose-1;
  if(verbose>10) printf("%s(*ift, %d, fp)\n", __func__, item);
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(item<0 || item>=ift->keyNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Set interfile-type key/value separator */
  char eq_sign[3];
  switch(ift->type) {
    case 1: strcpy(eq_sign, ":="); break;
    case 2: strcpy(eq_sign, "="); break;
    case 3: strcpy(eq_sign, ":"); break;
    case 4: strcpy(eq_sign, " "); break;
    case 5: strcpy(eq_sign, "\t"); break;
    case 6: strcpy(eq_sign, ","); break;
    case 7: strcpy(eq_sign, ";"); break;
    default: strcpy(eq_sign, ":="); break;
  }

  int n=5; if(ift->item[item].key!=NULL) n+=strlen(ift->item[item].key);
  if(ift->item[item].value!=NULL) n+=strlen(ift->item[item].value);
  if(ift->item[item].comment!=0) n+=2; 
  if(verbose>100) printf("  n := %d\n", n);
  char line[n], cmt[3];
  /* 'Print' comment character, if required */
  if(ift->item[item].comment!=0) strcpy(cmt, "# "); else strcpy(cmt, "");
  /* 'Print' key and/or value */
  if(ift->item[item].key==NULL || strlen(ift->item[item].key)<1) {
    sprintf(line, "%s%s", cmt, ift->item[item].value);
  } else {
    if((ift->space_before_eq==0 && ift->space_after_eq==0) ||
       ift->type==4 || ift->type==5 || ift->type==6 || ift->type==7)
      sprintf(line, "%s%s%s%s", cmt, ift->item[item].key, eq_sign, ift->item[item].value);
    else if(ift->space_before_eq==1 && ift->space_after_eq==0)
      sprintf(line, "%s%s %s%s", cmt, ift->item[item].key, eq_sign, ift->item[item].value);
    else if(ift->space_before_eq==0 && ift->space_after_eq==1)
      sprintf(line, "%s%s%s %s", cmt, ift->item[item].key, eq_sign, ift->item[item].value);
    else
      sprintf(line, "%s%s %s %s", cmt, ift->item[item].key, eq_sign, ift->item[item].value);
  }
  /* Write line into file */
  if(fprintf(fp, "%s\n", line) < (int)strlen(line)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/******************************************************************************/

/******************************************************************************/
/** Write the contents of IFT to the specified file pointer.
    @sa iftWriteItem, iftInit, iftFree
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */
int iftWrite(
  /** Pointer to IFT. */
  IFT *ift,
  /** Output file pointer. */
  FILE *fp,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
  if(ift==NULL || fp==NULL) return TPCERROR_FAIL;
  if(status!=NULL && status->verbose>100) printf("keyNr := %d\n", ift->keyNr);
  /* Write the contents */
  int li, ret;
  for(li=0, ret=0; li<ift->keyNr; li++) {
    ret=iftWriteItem(ift, li, fp, status); if(ret!=TPCERROR_OK) break;
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read IFT data into IFT structure. 

    Any previous contents of IFT are preserved.
    This function can read the initial ASCII part of files that contain also
    binary data in the end (from some Interfile images), but not the binary data itself.

    @sa iftInit, iftWrite, iftFree
    @return enum tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the IFT structure with iftInit().
    @author Vesa Oikonen
 */
int iftRead(
  /** Pointer to IFT. */
  IFT *ift,
  /** Input file pointer. */
  FILE *fp,
  /** Specifies whether key name is required or not. If not required, then line contents without 
      equals sign are assumed to represent a value string and key name in IFT struct is left empty.
      - 0 = key name is not required,
      - 1 = only lines with key and equals sign are read,
      - 2 = only lines with key and value separated by space are read. */
  int is_key_required,
  /** Specifies whether comment lines are processed or not.
      - 0 = comment lines are not read,
      - 1 = also comment lines are read,
      - 2 = only comment lines are read. */
  int is_comment_accepted,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
  if(ift==NULL || fp==NULL) return TPCERROR_FAIL;
  int verbose=0; if(status!=NULL) verbose=status->verbose-1;
  if(verbose>0) {
    printf("%s(*ift, fp, %d, %d)\n", __func__, is_key_required, is_comment_accepted);
    fflush(stdout);
  }
  int initial_key_nr=0; if(ift->keyNr>0) initial_key_nr=ift->keyNr;
  if(verbose>11) {
    printf("  initial_key_nr := %d\n", initial_key_nr);
    fflush(stdout);
  }
  
  /* Get the size of the ASCII part of the file */
  size_t fsize=asciiFileSize(fp, NULL);
  if(verbose>11) {printf("  ASCII size := %d\n", (int)fsize); fflush(stdout);}
  /* If ASCII part is too small, then lets consider that an error */
  if( fsize<1 || (is_key_required && fsize<3) ) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  /* If ASCII part is too large, then lets consider that an error */
  if(fsize>5000000) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_TOO_BIG);
    return TPCERROR_TOO_BIG;
  }
  /* Read that to a string */
  char *data;
  data=asciiFileRead(fp, NULL, fsize+1); rewind(fp);
  if(data==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  /* Read one line at a time from the string and fill IFT */
  int i=0, j, ret;
  char *cptr, *line;
  cptr=data;
  while((line=strTokenDup(cptr, "\n\r", &j))!=NULL) {
    if(verbose>15) {printf(" '%s'\n", line); fflush(stdout);}
    ret=iftPutFromString(ift, line, is_key_required, is_comment_accepted, status);
    if(verbose>1 && ret!=0) {
      fprintf(stderr, "Warning: cannot read line %d: '%s'.\n", i, line);
      fflush(stderr);
    }  
    free(line); cptr+=j; i++;
  }
#if(0) // this would mess up the functions that read other data formats as ift
  /* If none found, and key is required, we will try with space as separator */
  if(initial_key_nr==ift->keyNr && is_key_required) {
    cptr=data; i=0;
    while((line=strTokenDup(cptr, "\n\r", &j))!=NULL) {
      ret=iftPutFromStringWithSpaceSeparator(ift, line, is_comment_accepted, status);
      if(verbose>1 && ret!=0)
        fprintf(stderr, "Warning: cannot read line %d: '%s'.\n", i, line);  
      free(line); cptr+=j; i++;
    }
  }
#endif
  /* Ready */
  free(data);
  if(i==0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Process a given string to add key and value to IFT.

    Either key or value can be empty, but not both of them.

    @sa iftInit, iftDuplicate
    @return tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the IFT struct with iftInit().
    @author Vesa Oikonen
 */
int iftPutFromString(
  /** Pointer to initiated IFT; previous contents are not changed. */
  IFT *ift,
  /** Pointer to the string to be processed, e.g. "key := value"; extra space characters are
      excluded from key and value. */
  const char *line,
  /** Specifies whether key name is required or not. If not required, then line contents without
      equals sign are assumed to represent a value string and key name in IFT struct is left empty.  
      - 0 = key name is not required,
      - 1 = only lines with key and equals sign are read,
      - 2 = only lines with key and value separated by space are read. */
  int is_key_required,
  /** Specifies whether comment lines are processed or not.
      - 0 = comment lines are not read,
      - 1 = also comment lines are read,
      - 2 = only comment lines are read. */
  int is_comment_accepted,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  if(ift==NULL) return TPCERROR_FAIL;
  if(line==NULL || strlen(line)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  int verbose=0; if(status!=NULL) verbose=status->verbose-1;
  if(verbose>10) {
    printf("%s(ift, ", __func__);
    if(line!=NULL) printf("\"%s\", ", line); else printf("NULL, ");
    printf("%d, %d)\n", is_key_required, is_comment_accepted);
    fflush(stdout);
  }
  if(is_key_required==2) { // If space is used as separator
    return(iftPutFromStringWithSpaceSeparator(ift, line, is_comment_accepted, status));
  }
  
  /* Check if line is a comment line */
  char *lptr=(char*)line; int i; char cmt;
  if(asciiCommentLine(lptr, &i)) {cmt=(char)1; lptr+=i;} else cmt=(char)0;
  /* If comment line requirement is not met, then return here */
  if((is_comment_accepted==0 && cmt) || (is_comment_accepted==2 && !cmt)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return TPCERROR_OK;
  }

  /* Find the 'equals' sign, ':=', '=', or ':' */
  char *eq_ptr;
  eq_ptr=strstrNoQuotation(lptr, ":=");
  if(eq_ptr==NULL) eq_ptr=strstrNoQuotation(lptr, "=");
  /* If not yet found, try ':' but it must not be part of time or Windows path, for example 'C:\tmp' */
  char *cptr2=lptr;
  while(eq_ptr==NULL) {
    if(verbose>100) printf("cptr2 := '%s'\n", cptr2);
    eq_ptr=strstrNoQuotation(cptr2, ":"); if(eq_ptr==NULL) break;
    if(eq_ptr[1]=='\\') {
      /* ok ':' was part of windows path ... but search for later equals sign */
      cptr2=eq_ptr+2; eq_ptr=NULL; continue;
    }
    if(verbose>100) printf("eq_ptr := '%s'\n", eq_ptr);
    if(strlen(cptr2)-strlen(eq_ptr)<2) break; // cannot be time
    if(verbose>100) printf("is this time '%s'\n", eq_ptr-2);
    if(strTimeValid(eq_ptr-2)>0) break; // was not time
    /* ok ':' was part of time ... but search for later equals sign */
    if(verbose>100) printf("yes it was\n");
    cptr2=eq_ptr+6; eq_ptr=NULL;
  }

  /* We must have equals sign if key is required */
  if(is_key_required!=0 && eq_ptr==NULL) {
    if(verbose>50) printf("key required but equals sign not found\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_KEY);
    return TPCERROR_NO_KEY;
  }

  char *key=NULL, *value=NULL;
  if(eq_ptr==NULL) {
    /* No key */
    key=strndup("", 0);
    /* then value is the whole string, excluding comment */
    value=strndup(lptr, strlen(lptr)); strClean(value);
  } else {
    /* key starts after comment */
    key=strndup(lptr, strlen(lptr)-strlen(eq_ptr)); 
    if(verbose>100) printf("key before cleaning is '%s'\n", key);
    strClean(key);
    if(verbose>100) printf("key after cleaning is '%s'\n", key);
    /* Find the end of the 'equals' sign; that is the start of value */
    int j=strspn(eq_ptr, ":="); value=strdup(eq_ptr+j); 
    if(verbose>100) printf("value before cleaning is '%s'\n", value);
    strClean(value);
    if(verbose>100) printf("value after cleaning is '%s'\n", value);
  }

  int ret=iftPut(ift, key, value, cmt, status);
  free(key); free(value);
  if(ret==0) statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return ret;
}
/*****************************************************************************/

/*****************************************************************************/
/** Process a given string to add key and value to IFT.

    Key and value must be present, with space character as the field separator.

    @sa iftInit, iftDuplicate
    @return tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the IFT struct with iftInit().
    @author Vesa Oikonen
 */
int iftPutFromStringWithSpaceSeparator(
  /** Pointer to initiated IFT; previous contents are not changed. */
  IFT *ift,
  /** Pointer to the string to be processed, e.g. "key value"; extra space characters are
      excluded from key and value. */
  const char *line,
  /** Specifies whether comment lines are processed or not.
      - 0 = comment lines are not read,
      - 1 = also comment lines are read,
      - 2 = only comment lines are read. */
  int is_comment_accepted,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  if(ift==NULL) return TPCERROR_FAIL;
  if(line==NULL || strlen(line)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  int verbose=0; if(status!=NULL) verbose=status->verbose-1;
  if(verbose>10) {
    printf("%s(ift, ", __func__);
    if(line!=NULL) printf("\"%s\", ", line); else printf("NULL, ");
    printf("%d)\n", is_comment_accepted);
  }
  
  /* Check if line is a comment line */
  char *lptr=(char*)line; int i; char cmt;
  if(asciiCommentLine(lptr, &i)) {cmt=(char)1; lptr+=i;} else cmt=(char)0;
  /* If comment line requirement is not met, then return here */
  if((is_comment_accepted==0 && cmt) || (is_comment_accepted==2 && !cmt)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return TPCERROR_OK;
  }

  /* Find the 'equals' sign, ' ' or '\t' */
  char *eq_ptr;
  eq_ptr=strstrNoQuotation(lptr, " ");
  if(eq_ptr==NULL) eq_ptr=strstrNoQuotation(lptr, "\t");

  /* key is required */
  if(eq_ptr==NULL) {
    if(verbose>50) printf("key required but space not found\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_KEY);
    return TPCERROR_NO_KEY;
  }

  char *key=NULL, *value=NULL;
  /* key starts after comment */
  key=strndup(lptr, strlen(lptr)-strlen(eq_ptr)); 
  if(verbose>100) printf("key before cleaning is '%s'\n", key);
  strClean(key);
  if(verbose>100) printf("key after cleaning is '%s'\n", key);
  /* Find the end of the 'equals' sign; that is the start of value */
  int j=strspn(eq_ptr, " \t"); value=strdup(eq_ptr+j); 
  if(verbose>100) printf("value before cleaning is '%s'\n", value);
  strClean(value);
  if(verbose>100) printf("value after cleaning is '%s'\n", value);

  int ret=iftPut(ift, key, value, cmt, status);
  free(key); free(value);
  if(ret==0) statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return ret;
}
/*****************************************************************************/

/*****************************************************************************/
