/** @file decay.c
 *  @brief Factors for correcting or simulating physical decay.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcisotope.h"
/*****************************************************************************/

/*****************************************************************************/
/** String descriptions of the status of decay correction. */
static char *decay_correction[] = {
"unknown",
"not corrected",
"corrected",
"corrected to scan start time",
"corrected to administration time",
0};
/*****************************************************************************/

/*****************************************************************************/
/** Return pointer to string describing the status of decay correction.
    @sa isotopeName
    @return pointer to the name string.
 */
char *decayDescr(
  /** Enum #decaycorrection. */
  decaycorrection d
) {
  if(d>4) return(decay_correction[0]);
  return(decay_correction[d]);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate lambda based on specified halflife.
 *  @return Lambda (=ln(2)/halflife), or NaN if halflife is NaN or <=0.
 *  @sa isotopeIdentifyHalflife, isotopeName, lambdaFromIsotope
 *  @author Vesa Oikonen
 */
double lambdaFromHalflife(
  /** Halflife of the isotope; if halflife is in minutes, then the unit of lambda will be 1/min, 
      if seconds, then lambda will be in units 1/sec, etc. */
  double halflife
) {
  if(halflife>0.0) return(M_LN2/halflife);  // M_LN2=log(2.0)
  else return nan("");
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate lambda for specified isotope.
 *  @return Lambda (=ln(2)/halflife) in units 1/min, or NaN if isotope is not identified.
 *  @sa decayCorrectionFactorFromLambda, isotopeName, isotopeIdentify, decayCorrectionFactorFromIsotope
 *  @author Vesa Oikonen
 */
double lambdaFromIsotope(
  /** isotope_code as enum or the index of isotope in isotope table. */
  int isotope
) {
  double hl=isotopeHalflife(isotope); if(isnan(hl) || hl<=0.0) return nan("");
  return(M_LN2/hl);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the correction factor for physical decay from measurement time to time zero.
    @return Factor for decay correction (>1 when lambda>0) or for simulating decay 
     (<1 when lambda<0), or NaN if input is invalid.
    @sa lambdaFromIsotope, isotopeIdentify, decayCorrectionFactorFromIsotope
    @author Vesa Oikonen
 */
double decayCorrectionFactorFromLambda(
  /** Positive or negative lambda for the isotope. Note that time unit must be 1/timeunit of 
      the starttime and duration. */
  double lambda,
  /** Radioactivity measurement start time, or middle time, if measurement duration is not known. */
  double starttime,
  /** Radioactivity measurement duration; if not known, then set to zero,
      and put measurement middle time for starttime. */
  double duration
) {
  if(starttime<0.0) return nan("");
  if(duration<0.0) duration=0.0;
  if(fabs(lambda)<1.0E-100) return nan("");
  double cf=exp(lambda*starttime);
  if(duration>1.0E-10) {
    double ff=fabs(lambda)*duration/(1.0-exp(-fabs(lambda)*duration));
    if(lambda<0.0) cf/=ff; else cf*=ff;
  }
  return(cf);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the correction factor for physical decay from measurement time to time zero.
 *  @return Factor for decay correction (>1), or NaN if input is invalid.
    @sa lambdaFromIsotope, isotopeIdentify, decayCorrectionFactorFromLambda, isotopeName
 *  @author Vesa Oikonen
 */
double decayCorrectionFactorFromIsotope(
  /** isotope_code as enum or the index of isotope in isotope table. */
  int isotope,
  /** Radioactivity measurement start time, or middle time, if measurement duration is not known. 
      Unit must be min. */
  double starttime,
  /** Radioactivity measurement duration; if not known, then set to zero,
      and put measurement middle time for starttime. Unit must be min. */
  double duration
) {
  double lambda=lambdaFromIsotope(isotope); if(isnan(lambda)) return nan("");
  return decayCorrectionFactorFromLambda(lambda, starttime, duration);
}
/*****************************************************************************/

/*****************************************************************************/
