/** @file isotope.c
 *  @brief Isotope halflife functions.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcisotope.h"
/*****************************************************************************/

/*****************************************************************************/
/*! @cond PRIVATE */
/** One item for table of isotopes */
typedef struct TPC_ISOTOPE {
  /** Isotope name */
  char name[MAX_ISOTOPE_LEN];
  /** Half-life in minutes */
  double hl;
  /** Branching ratio */
  double br;
} TPC_ISOTOPE;

/** Table of isotopes; start with two-letter elements; note that branching fraction and 
    half-life for isotopes may be uncertain; half-lives are in minutes. */
static TPC_ISOTOPE tpc_isotope[]={
  {"unknown", 0.,      0.},
  {"Br-75", 98.0,      0.90},
  {"Br-76", 978.33,    0.62},
  {"Cu-62", 9.7,       1.0},
  {"Cu-64", 762.018,   0.174},
  {"Fe-52", 4980.0,    0.56},
  {"Ga-68", 68.0,      0.891},
  {"Ge-68", 396000.0,  0.891},
  {"Na-22", 1368000.0, 0.906},
  {"Rb-82", 1.25,      0.950},
  {"Sc-44", 238.2,     0.94},
  {"Tb-152",1050.0,    0.17},
  {"Zn-62", 548.0,     0.18},
  {"Zr-89", 4704.0,    0.22},
  {"C-11",  20.4,      0.998},
  {"F-18",  109.8,     0.967},
  {"I-124", 6013.44,   0.26},
  {"N-13",  10.0,      0.998},
  {"O-15",  2.05,      0.999},
  {"O-14",  1.1818,    1.0},
  {"", 0., 0.}
};
/*! @endcond */
/*****************************************************************************/

/*****************************************************************************/
/** Get the half-life (in minutes) of an isotope specified with its isotope_code.
    @sa isotopeBranching, isotopeName, lambdaFromIsotope, decayCorrectionFactorFromIsotope
    @return halflife of the isotope in minutes, or NaN if not identified.
    @author Vesa Oikonen
 */
double isotopeHalflife(
  /** isotope_code as enum or the index of isotope in isotope table. */
  int isotope_code
) {
  int n=0;
  
  while(strlen(tpc_isotope[n].name)>1) n++;
  if(isotope_code<1 || isotope_code>n-1) return nan("");
  else return(tpc_isotope[isotope_code].hl);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the branching ratio (fraction) of an isotope spcecified with its isotope_code.
 
    @sa isotopeHalflife, isotopeName, lambdaFromIsotope, decayCorrectionFactorFromIsotope
    @return branching ratio of the isotope, or NaN if not identified.
    @author Vesa Oikonen
 */
double isotopeBranching(
  /** isotope_code as enum or the index of isotope in isotope table */
  int isotope_code
) {
  int n=0;
  
  while(strlen(tpc_isotope[n].name)>1) n++;
  if(isotope_code<1 || isotope_code>n-1) return nan("");
  else if(tpc_isotope[isotope_code].br<1.0E-10) return nan("");
  else return(tpc_isotope[isotope_code].br);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the string representation of an isotope specified with its isotope_code.
  
    @sa isotopeBranching, isotopeHalflife, elementName, elementSymbol
    @return pointer to string representation of the isotope, "unknown" if not identified.
    @author Vesa Oikonen
 */
char *isotopeName(
  /** isotope_code as enum or the index of isotope in isotope table. */
  int isotope_code
) {
  int n=0;
  
  while(strlen(tpc_isotope[n].name)>1) n++;
  if(isotope_code<1 || isotope_code>n-1)
    return tpc_isotope[ISOTOPE_UNKNOWN].name;
  else
    return(tpc_isotope[isotope_code].name);
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify the isotope based on halflife.
    @return Isotope code, or 0 (enum ISOTOPE_UNKNOWN) if not identified.
    @sa isotopeName, lambdaFromIsotope, decayCorrectionFactorFromIsotope
    @author Vesa Oikonen
 */
int isotopeIdentifyHalflife(
  /** Halflife in minutes */
  double halflife
) {
  if(halflife<=0.01) return ISOTOPE_UNKNOWN;
  int i, n=0;
  while(strlen(tpc_isotope[n].name)>1) n++;
  /* Check, if halflife can be found in the table */
  for(i=1; i<n; i++) {
    if(fabs(halflife/tpc_isotope[i].hl-1.0)<0.05 ) return i;
  }
  return ISOTOPE_UNKNOWN;
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify the given string representation of isotope, whether it is in format like 'C-11', 
    '11C', '^11^C', or '^11^Carbon'. Even a one-letter symbol like 'C' may be accepted for the 
    most common PET isotopes.

   @sa isotopeHalflife, isotopeName, lambdaFromIsotope, decayCorrectionFactorFromIsotope, elementIdentify
   @return Isotope code, or 0 (enum ISOTOPE_UNKNOWN) if not identified.
   @author Vesa Oikonen
 */
int isotopeIdentify(
  /** Name of isotope to identify. */
  const char *isotope
) {
  if(isotope==NULL || strnlen(isotope, 5)<1) return ISOTOPE_UNKNOWN; 

  unsigned short int i, n=0;
  while(strlen(tpc_isotope[n].name)>1) n++;

  /* Check, if isotope can be found directly in the table */
  for(i=0; i<n; i++) {
    if(strcasecmp(tpc_isotope[i].name, isotope)==0) return(i);
  }

  /* Ok it was not that easy... try to figure out what it is */
  char *cptr;
  int mass_nr=0, ic_mass_nr=0;
  /* Find the element */
  unsigned short int z=elementIdentify(isotope);
  if(z==0) return(ISOTOPE_UNKNOWN);
  /* Now find the mass number */
  cptr=strpbrk(isotope, "123456789"); // mass cannot start with zero
  if(cptr!=NULL) {
    mass_nr=atoi(cptr);
  }
  //printf("fixed isotope: %s-%d\n", elementSymbol(z), mass_nr);

  /* Check if element and mass matches any of the isotopes in our table */
  for(i=1; i<n; i++) {
    /* check the element */
    if(elementIdentify(tpc_isotope[i].name)!=z) continue;
    /* check the mass, if found */
    cptr=strchr(tpc_isotope[i].name, '-'); if(cptr==NULL) continue;
    ic_mass_nr=atoi(cptr+1);
    if(mass_nr>0 && ic_mass_nr!=mass_nr) continue;
    /* Match was found! */
    return(i);
  }

  return(ISOTOPE_UNKNOWN);
}
/*****************************************************************************/

/*****************************************************************************/
