/** @file ranoise.c
 *  @brief PET noise estimation.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcisotope.h"
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the standard deviation (SD) of noise for PET radioactivity concentration.

    SD of noise can be used in simulations.
    Reference: Varga & Szabo. J Cereb Blood Flow Metab 2002;22(2):240-244.

    @sa isotopeIdentify, drandGaussian, simC3s, simC3p, simRTCM

    @return Returns the SD, or NaN in case of an error.
 */
double noiseSD4Frame(
  /** Sample radioactivity concentration (decay corrected to zero time). */
  double y,
  /** Radioactivity measurement (frame) start time in minutes. */
  double t1,
  /** Radioactivity measurement (frame) duration in minutes. */
  double dt,
  /** Isotope code as enum or index of isotope in isotope table; enter ISOTOPE_UNKNOWN
      if not to be considered. */
  int isotope,
  /** Proportionality factor. Note that it should be in relation to the unit of y. */
  double a
) {
  /* Frame start time must be >=0 and frame duration must be >0 */
  if(isnan(y) || !(t1>=0.0) || !(dt>0.0) || !(a>=0.0)) return(nan(""));
  /* If concentration is not positive, then its SD is zero */
  if(!(y>0.0)) return(0.0);

  /* Calculate decay factor (<=1) */
  double df;
  if(isotope==ISOTOPE_UNKNOWN) {
    df=1.0;
  } else {
    df=1.0/decayCorrectionFactorFromIsotope(isotope, t1, dt);
    if(!(df>0.0 && df<=1.0)) return(nan(""));
  }

  /* Calculate SD */
  return(sqrt((a*y)/(dt*df)));
}
/*****************************************************************************/

/*****************************************************************************/
