/** @file modell.c
 *  @brief Model and function codes and descriptions for other libraries.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcmodels.h"
/*****************************************************************************/

/*****************************************************************************/
/*! @cond PRIVATE */
/** One item for table of models and mathematical functions */
typedef struct TPC_MODEL {
  /** Human-readable model code name. Use this in new files instead of id number in future. */
  char *code;
  /** Deprecated ID number of the function; only needed for compatibility with 
      file formats from tpcclib v1 and before that. */
  unsigned int id;
  /** Model type: 0=unknown, 1=CM, 2=function, 3=pharmacokinetic CM */
  unsigned int type; 
  /** Parameter number. */
  unsigned int parNr;
  /** Model/function description for humans. */
  char *desc;
} TPC_MODEL;
/*****************************************************************************/

/*****************************************************************************/
/** Table of models and mathematical functions.
  
   Code name is stored in parameter files, but the table index (line) number
   is used instead in the PAR structure. Code name for each model must be unique.
 
   Do not change the code name or parameter nr of existing lines, because existing 
   parameter files may depend on them. Instead, you can add new models and change 
   the order of lines freely, if all dependent libraries and applications are recompiled. 
   The first one, index 0, must represent unknown (invalid) model.
   Description field is only used as information to users, and it can be freely edited.
   
   Not all parameter numbers are yet set correctly; check before using.
   Neither are all (most) functions yet supported.
 */
static TPC_MODEL tpc_model[]={
  {"unknown",    0,    0, 0,  "unknown"},
  /* mathematical functions */
  {"level",       100,  2, 1,  "f(x)=A"},
  {"line",        101,  2, 2,  "line"},
  {"pol2",        102,  2, 3,  "2nd degree polynomial"},
  {"pol3",        103,  2, 4,  "3rd degree polynomial"},
  {"pol4",        104,  2, 5,  "4th degree polynomial"},
  {"pol5",        105,  2, 6,  "5th degree polynomial"},
  {"pol6",        106,  2, 7,  "6th degree polynomial"},
  {"pol7",        107,  2, 8,  "7th degree polynomial"},
  {"pol8",        108,  2, 9,  "8th degree polynomial"},
  {"pol9",        109,  2, 10, "9th degree polynomial"},
  {"ratf11",      211,  2, 0,  "1st/1st order rational function"},
  {"ratf21",      221,  2, 0,  "2nd/1st order rational function"},
  {"ratf22",      222,  2, 0,  "2nd/2nd order rational function"},
  {"ratf32",      232,  2, 0,  "3rd/2nd order rational function"},
  {"ratf33",      233,  2, 0,  "3rd/3rd order rational function"},
  {"1exp",        301,  2, 0,  "single exponential"},
  {"2exp",        302,  2, 0,  "sum of 2 exponentials"},
  {"3exp",        303,  2, 0,  "sum of 3 exponentials"},
  {"4exp",        304,  2, 0,  "sum of 4 exponentials"},
  {"5exp",        305,  2, 0,  "sum of 5 exponentials"},
  {"lundq1",      321,  2, 0,  "Lundqvist function"},
  {"lundq2",      322,  2, 0,  "sum of 2 Lundqvist functions"},
  {"lundq3",      323,  2, 0,  "sum of 3 Lundqvist functions"},
  {"ebolinf",     331,  2, 0,  "exponential bolus infusion function"},
  {"ebolinfrw",   332,  2, 0,  "Kudomi's exponential bolus infusion function for radiowater"},
  {"ebolinfaz",   334,  2, 0,  "exponential bolus function approaching zero"},
  {"ppfpk11195",  351,  2, 0,  "exp function for PK11195 plasma parent fraction"},
  {"mpfhill",     841,  2, 0,  "Hill function"},
  {"ppfhill",     842,  2, 0,  "Hill function (1-f(x))"},
  {"ppfhille",    843,  2, 0,  "Hill function (1-f(x)) with ascending or descending end"},
  {"mpfb",        844,  2, 0,  "Hill function with background"},
  {"ppfb",        845,  2, 0,  "Hill function (A-f(x))"},
  {"ppfehill1",   846,  2, 0,  "Extended Hill function for plasma parent fraction"},
  {"mpfehill1",   847,  2, 0,  "Extended Hill function for plasma metabolite fraction"},
  {"ppfefill2",   848,  2, 0,  "Extended Hill function #2 for plasma parent fraction"},
  {"mpfehill2",   849,  2, 0,  "Extended Hill function #2 for plasma metabolite fraction"},
  {"mpfmamede",   851,  2, 0,  "Mamede function"},
  {"ppfmamede",   852,  2, 0,  "Mamede function (1-f(x)"},
  {"ppfmeyer",    861,  2, 0,  "Meyer function for plasma parent fraction"},
  {"mpfmeyer",    862,  2, 0,  "Meyer function for plasma metabolite fraction"},
  {"ppfemeyer",   863,  2, 0,  "extended Meyer function for plasma parent fraction"},
  {"mpfemeyer",   864,  2, 0,  "extended Meyer function for plasma metabolite fraction"},
  {"ppf3hill",    871,  2, 0,  "1-3 metabolite Hill function for plasma parent fraction"},
  {"mpf3hill1",   872,  2, 0,  "1-3 metabolite Hill function for plasma metab1 fraction"},
  {"mpf3hill2",   873,  2, 0,  "1-3 metabolite Hill function for plasma metab2 fraction"},
  {"mpf3hill3",   874,  2, 0,  "1-3 metabolite Hill function for plasma metab3 fraction"},
  {"ppf3pow",     881,  2, 0,  "1-3 metabolite power function for plasma parent fraction"},
  {"mpf3pow1",    882,  2, 0,  "1-3 metabolite power function for plasma metab1 fraction"},
  {"mpf3pow2",    883,  2, 0,  "1-3 metabolite power function for plasma metab2 fraction"},
  {"mpf3pow3",    884,  2, 0,  "1-3 metabolite power function for plasma metab3 fraction"},
  {"fengm2s",     1312, 2, 5,  "PTAC function with two exponentials"},
  {"fengm2",      1313, 2, 7,  "PTAC function with three exponentials"},
  {"fengm2e",     1314, 2, 9,  "PTAC function with four exponentials"},
  {"gammav",      1401, 2, 4,  "Gamma variate"},
  {"gammavb",     1402, 2, 0,  "Gamma variate with background"},
  {"weibullcdfd", 1421, 2, 4,  "Weibull cdf with delay"},
  {"weibullcdfdd",1423, 2, 5,  "Weibull cdf and pdf sum with delay"},
  {"surge",       1431, 2, 2,  "Surge function with AUC as parameter"},
  {"tradsurge",   1432, 2, 2,  "Traditional surge function"},
  {"surgerecirc", 1433, 2, 3,  "Surge function with recirculation"},
  {"p2bsrc",      1434, 2, 4,  "Surge function with recirculation for plasma-to-blood ratio"},
  {"surgefdgaif", 1435, 2, 5,  "Surge function for late FDG AIF with delay"},
  {"erlangpdf",   1441, 2, 3,  "Probability density function of Erlang distribution"},
  {"hilld",       1801, 2, 0,  "Hill function with delay"},
  {"hilldd",      1811, 2, 0,  "derivative of Hill function with delay"},
  {"hillssd",     1821, 2, 0,  "sum of Hill function and derivative with delay"},
  {"imgprofile",  2111, 2, 0,  "image profile"},
  /* compartmental models */
  {"FRTM",                 0, 1, 4,  "full reference tissue model"},
  {"SRTM",                 0, 1, 3,  "simplified reference tissue model"},
  {"RRTM",                 0, 1, 3,  "reduced reference tissue model"},
  {"TRTM",                 0, 1, 3,  "transport-limited reference tissue model"},
  {"1TCM",                 0, 1, 5,  "one-tissue compartmental model"},
  {"2TCM",                 0, 1, 7,  "two-tissue compartmental model"},
  {"PAR3TCM",              0, 1, 9,  "parallel three-tissue compartmental model"},
  {"SER3TCM",              0, 1, 9,  "serial three-tissue compartmental model"},
  {"TTM",                  0, 1, 2,  "transit-time model"},
  {"radiowater",           0, 1, 4,  "radiowater perfusion model"},
  {"radiowater2",          0, 1, 6,  "radiowater perfusion model 2"},
  {"radiowater-lung",      0, 1, 5,  "radiowater perfusion model for lungs"},
  {"radiowater-liver",     0, 1, 7,  "radiowater perfusion model for liver"},
  {"radiowater-liver-TTM", 0, 1, 6,  "radiowater perfusion TTM model for liver"},
  {"dispdelay",            0, 1, 2,  "dispersion and delay in radiowater studies"},
  /* pharmacokinetic compartment models */
  {"O1CM",        0,    3, 3,  "one-compartment model with first-order absorption and elimination"},
  {"O2CM",        0,    3, 5,  "two-compartment model with first-order absorption and elimination"},
  /* combinations of functions and models */
  {"ebolinfdd",   3331,12, 8,  "exponential bolus infusion function plus delay and dispersion"},
  {"",            0,    0, 0,  ""}
};
/*! @endcond */
/*****************************************************************************/

/*****************************************************************************/
/** Number of listed models.
    @return Nr of models.
 */
unsigned int modelNr()
{
  unsigned int i=0;
  while(tpc_model[i].code[0]!='\0') i++;
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get string representation of a model code.
    @return pointer to string representation of the code, or NULL if not identified.
    @author Vesa Oikonen
    @sa modelDesc, modelCodeIndex, modelNr
 */
char *modelCode(
  /** model list index. */
  const unsigned int i
) {
  if(i>=modelNr()) return((char*)NULL);
  return(tpc_model[i].code);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the deprecated function id for working with older file formats.
    @return function id, or 0 if not identified.
    @author Vesa Oikonen
    @sa modelOld2New, modelCode, modelCodeIndex, modelNr
 */
unsigned int modelOldId(
  /** model list index. */
  const unsigned int i
) {
  return(tpc_model[i].id);
}
/*****************************************************************************/

/*****************************************************************************/
/** Convert the deprecated function id to the model list index.
    @return model/function list index, or 0 if not identified.
    @author Vesa Oikonen
    @sa modelOldId, modelCode, modelCodeIndex, modelNr
 */
unsigned int modelOld2New(
  /** Deprecated function id. */
  const unsigned int i
) {
  unsigned int n=modelNr();
  if(i==0) return(0);
  for(unsigned int j=1; j<n; j++) if(tpc_model[j].id==i) return(j);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get description of a model.
    @return pointer to string, or NULL if not identified.
    @author Vesa Oikonen
    @sa modelCode
 */
char *modelDesc(
  /** model list index. */
  const unsigned int i
) {
  if(i>=modelNr()) return((char*)NULL);
  return(tpc_model[i].desc);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get model index for the string representation of model code.
    @return index number, 0, if unknown model code.
    @author Vesa Oikonen
    @sa modelCode, modelDesc
 */
unsigned int modelCodeIndex(
  /** model code as a string; code is case-insensitive, but otherwise exact match is required. */
  const char *s
) {
  if(s==NULL || *s=='\0') return(0);
  unsigned int i=0;
  while(tpc_model[i].code[0]!='\0') {
    if(!strcasecmp(tpc_model[i].code, s)) return(i);
    i++;
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get model parameter number.
    @return parameter number, or 0 if invalid model code.
    @author Vesa Oikonen
 */
unsigned int modelParNr(
  /** model code; can be retrieved using modelCodeIndex(). */
  const unsigned int code
) {
  if(code<1 || code>=modelNr()) return(0);
  return(tpc_model[code].parNr);
}
/*****************************************************************************/

/*****************************************************************************/
