/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpcnlopt.h"
/*****************************************************************************/

/*****************************************************************************/
int test_nloptSimplex(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>10) {
    char logfile[256];
    sprintf(logfile, "%s.log", __func__);
    status->fp=fopen(logfile, "w");
  } else status->fp=stdout;

  int ret, parNr;
  NLOPT nlo; nloptInit(&nlo);
  double e;

  if(verbose>1) fprintf(status->fp, "\ncalling function with empty input\n");
  ret=nloptSimplex(&nlo, 0, status);
  if(ret==TPCERROR_OK) return(1);
  if(verbose>2) fprintf(status->fp, "  -> error returned, which is good\n");


  if(verbose>1) fprintf(status->fp, "\n Function should use 1D method when N=1 \n");
  parNr=1;
  ret=nloptAllocate(&nlo, parNr); if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(11);}
  /* set parameter limits and initial values */
  nlo.totalNr=1;
  nlo.xfull[0]=1.0;  nlo.xlower[0]=0.0;  nlo.xupper[0]=255.0;
  nlo.xtol[0]=1.E-04; nlo.xdelta[0]=1.0;
  /* set object function */
  nlo._fun=nloptX2;
  /* Optimize */
  nlo.maxFunCalls=500;
  ret=nloptSimplex(&nlo, 0, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(12);}
  if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    e=fabs(nlo.xfull[i]-50.5);
    if(e>1.0E-04) {ret++; if(verbose>2) fprintf(status->fp, "  |x[%d]-true|=%g\n", i, e);}
  }
  if(ret>0) {nloptFree(&nlo); return(13);}

  if(verbose>1) fprintf(status->fp, "\n Function should return error if even that one parameter is fixed \n");
  nlo.funCalls=0;
  nlo.xfull[0]=10.0;  nlo.xlower[0]=10.0;  nlo.xupper[0]=10.0; nlo.xdelta[0]=0.0;
  ret=nloptSimplex(&nlo, 0, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(ret==TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(14);
  }


  if(verbose>1) fprintf(status->fp, "\n Rosenbrock's valley with N=2 \n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  parNr=2;
  ret=nloptAllocate(&nlo, parNr);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(21);}
  /* set initial parameters */
  nlo.xfull[0]=0.9;
  nlo.xfull[1]=1.1;
  for(int i=0; i<2; i++) {
    nlo.xlower[i]=-100.0; nlo.xupper[i]=100.0; 
    nlo.xtol[i]=1.0E-08; nlo.xdelta[i]=1.0E-02;
  }
  /* set object function */
  nlo._fun=nloptDejong2;
  /* Optimize */
  ret=nloptSimplex(&nlo, 0, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(22);}
  if(verbose>2)
    for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    e=fabs(nlo.xfull[i]-1.0);
    if(e>1.0E-03) {
      ret++; if(verbose>2) fprintf(status->fp, "  |x-true|=%g\n", e);
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(23);
  }




  if(verbose>1) fprintf(status->fp, "\n Rosenbrock's valley with N=2 when one parameter is fixed to correct result \n");
  nlo.funCalls=0;
  nlo.xfull[1]=nlo.xlower[1]=nlo.xupper[1]=1.0; nlo.xtol[1]=0.0; nlo.xdelta[1]=0.0;
  ret=nloptSimplex(&nlo, 0, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(24);}
  if(verbose>2)
    for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    e=fabs(nlo.xfull[i]-1.0);
    if(e>1.0E-03) {
      ret++; if(verbose>2) fprintf(status->fp, "  |x-true|=%g\n", e);
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(25);
  }
  nloptFree(&nlo);


  if(verbose>1) fprintf(status->fp, "\n Rosenbrock's valley with N=6 \n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  parNr=6;
  ret=nloptAllocate(&nlo, parNr);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(31);}
  /* set initial parameters */
  nlo.xfull[0]=0.91;
  nlo.xfull[1]=1.11;
  nlo.xfull[2]=0.95;
  nlo.xfull[3]=1.05;
  nlo.xfull[4]=0.99;
  nlo.xfull[5]=1.15;
  for(int i=0; i<6; i++) {
    nlo.xlower[i]=-100.0; nlo.xupper[i]=100.0; 
    nlo.xtol[i]=1.0E-08; nlo.xdelta[i]=5.0E-02;
  }
  /* set object function */
  nlo._fun=nloptDejong2;
  /* Optimize */
  ret=nloptSimplex(&nlo, 0, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(32);}
  if(verbose>2)
    for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    e=fabs(nlo.xfull[i]-1.0);
    if(e>1.0E-03) {
      ret++; if(verbose>2) fprintf(status->fp, "  |x-true|=%g\n", e);
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(33);
  }


  if(verbose>1) fprintf(status->fp, "\n Same, but some parameters fixed to correct values \n");
  nlo.funCalls=0;
  nlo.xfull[2]=nlo.xlower[2]=nlo.xupper[2]=1.0; nlo.xtol[2]=0.0; nlo.xdelta[2]=0.0;
  nlo.xfull[3]=nlo.xlower[3]=nlo.xupper[3]=1.0; nlo.xtol[3]=0.0; nlo.xdelta[3]=0.0;
  nlo.xfull[5]=nlo.xlower[5]=nlo.xupper[5]=1.0; nlo.xtol[5]=0.0; nlo.xdelta[5]=0.0;
  /* Optimize */
  ret=nloptSimplex(&nlo, 0, status);
  if(verbose>2) {fprintf(status->fp, "  -> ret := %d\n", ret); fflush(status->fp);}
  if(verbose>3) {fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls); fflush(status->fp);}
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(34);}
  if(verbose>2) {
    for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
    fflush(status->fp);
  }
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    e=fabs(nlo.xfull[i]-1.0);
    if(e>1.0E-03) {
      ret++; if(verbose>2) {fprintf(status->fp, "  |x-true|=%g\n", e); fflush(status->fp);}
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__); fflush(stderr);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(35);
  }


  if(verbose>1) fprintf(status->fp, "\n Still Rosenbrock's valley with N=6, but initial values are correct \n");
  nlo.funCalls=0;
  nlo.xfull[0]=1.0;
  nlo.xfull[1]=1.0;
  nlo.xfull[2]=1.0;
  nlo.xfull[3]=1.0;
  nlo.xfull[4]=1.0;
  nlo.xfull[5]=1.0;
  for(int i=0; i<6; i++) {
    nlo.xlower[i]=-100.0; nlo.xupper[i]=100.0; 
    nlo.xtol[i]=1.0E-08; nlo.xdelta[i]=5.0E-01;
  }
  /* Optimize */
  ret=nloptSimplex(&nlo, 0, status);
  if(verbose>2) {fprintf(status->fp, "  -> ret := %d\n", ret); fflush(status->fp);}
  if(verbose>3) {fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls); fflush(status->fp);}
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(36);}
  if(verbose>2) {
    for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
    fflush(status->fp);
  }
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    e=fabs(nlo.xfull[i]-1.0);
    if(e>1.0E-06) {
      ret++; if(verbose>2) {fprintf(status->fp, "  |x-true|=%g\n", e); fflush(status->fp);}
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__); fflush(stderr);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(37);
  }
  nloptFree(&nlo);


  if(verbose>1) {fprintf(status->fp, "\n Test bi-exponential curve fitting \n"); fflush(status->fp);}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  {
    NLOPT_DATA data; nloptdataInit(&data);
    data.n=20;
    double buf[4*data.n]; data.x=buf; data.y=buf+data.n; data.w=buf+2*data.n; data.sy=buf+3*data.n;
    int parNr=4; double p[parNr]; p[0]=80.0; p[1]=0.2; p[2]=20.0; p[3]=0.02;
    for(unsigned int i=0; i<data.n; i++) {
      data.x[i]=(double)(1+i); 
      data.y[i]=p[0]*exp(-p[1]*data.x[i]) + p[2]*exp(-p[3]*data.x[i]);
      data.w[i]=1.0;
    }

    NLOPT nlo; nloptInit(&nlo);
    if(nloptAllocate(&nlo, parNr)!=TPCERROR_OK) {nloptFree(&nlo); return(100);}

    if(verbose>1) {
      fprintf(status->fp, "\n Level: easy; limits force into single optimum \n");
      fflush(status->fp);
    }
    /* set initial parameters */
    nlo.xfull[0]=91.2;  nlo.xlower[0]=50.0; nlo.xupper[0]=200.0; nlo.xdelta[0]=10.0; nlo.xtol[0]=0.01;
    nlo.xfull[1]=0.35;  nlo.xlower[1]=0.01; nlo.xupper[1]=1.0;   nlo.xdelta[1]=0.10; nlo.xtol[1]=0.001;
    nlo.xfull[2]=11.9;  nlo.xlower[2]=0.0;  nlo.xupper[2]=50.0;  nlo.xdelta[2]=10.0; nlo.xtol[2]=0.01;
    nlo.xfull[3]=0.006; nlo.xlower[3]=0.0;  nlo.xupper[3]=0.1;   nlo.xdelta[3]=0.01; nlo.xtol[3]=0.00001;
    nlo._fun=nloptBiexp; nlo.fundata=&data;
    if(verbose>100) {data.verbose=verbose-100; data.fp=stdout;}
    if(verbose>4) {
      fprintf(status->fp, "  correct_wss := %g\n", nlo._fun(parNr, p, nlo.fundata));
      fprintf(status->fp, "  initial_point_wss := %g\n", nlo._fun(parNr, nlo.xfull, nlo.fundata));
      fflush(status->fp);
    }
    ret=nloptSimplex(&nlo, 0, status);
    if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
    if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
    if(ret!=TPCERROR_OK) {
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout; nloptFree(&nlo); return(101);
    }
    if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %e\n", i, nlo.xfull[i]);
    /* Check optimized parameters */
    ret=0;
    for(int i=0; i<parNr; i++) {
      double e=fabs(nlo.xfull[i]-p[i]);
      if(e>nlo.xtol[i]) {
        ret++; 
        if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
      }
    }
    if(ret>0) {
      fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout;
      nloptFree(&nlo); return(102);
    }


    if(verbose>1) {
      fprintf(status->fp, "\n Level: moderate; limits allow dual optimum but initial values and\n");
      fprintf(status->fp, " deltas prefer one of the optima\n");
      fflush(status->fp);
    }
    /* set initial parameters */
    nlo.xfull[0]=70.0; nlo.xlower[0]=0.0; nlo.xupper[0]=200.0; nlo.xdelta[0]=10.0; nlo.xtol[0]=0.01;
    nlo.xfull[1]=0.50; nlo.xlower[1]=0.0; nlo.xupper[1]=1.0;   nlo.xdelta[1]=0.10; nlo.xtol[1]=0.0001;
    nlo.xfull[2]=11.0; nlo.xlower[2]=0.0; nlo.xupper[2]=200.0; nlo.xdelta[2]=10.0; nlo.xtol[2]=0.01;
    nlo.xfull[3]=0.01; nlo.xlower[3]=0.0; nlo.xupper[3]=1.0;   nlo.xdelta[3]=0.005; nlo.xtol[3]=0.0001;
    if(verbose>4) {
      fprintf(status->fp, "  correct_wss := %g\n", nlo._fun(parNr, p, nlo.fundata));
      fprintf(status->fp, "  initial_point_wss := %g\n", nlo._fun(parNr, nlo.xfull, nlo.fundata));
      fflush(status->fp);
    }
    ret=nloptSimplex(&nlo, 0, status);
    if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
    if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
    if(ret!=TPCERROR_OK) {
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout; nloptFree(&nlo); return(111);
    }
    if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %e\n", i, nlo.xfull[i]);
    /* Check optimized parameters */
    ret=0;
    if(nlo.xfull[0]>nlo.xfull[2]) {
      for(int i=0; i<parNr; i++) {
        e=fabs(nlo.xfull[i]-p[i]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
    } else {
      int i;
      for(i=0; i<2; i++) {
        e=fabs(nlo.xfull[i]-p[i+2]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
      for(; i<parNr; i++) {
        e=fabs(nlo.xfull[i]-p[i-2]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
    }
    if(ret>0) {
      fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout;
      nloptFree(&nlo); return(112);
    }


    if(verbose>1) {
      fprintf(status->fp, "\n Level: difficult; limits allow dual optimum, and initial values and\n");
      fprintf(status->fp, " deltas do not clearly prefer either of the optima\n");
      fflush(status->fp);
    }
    /* set initial parameters */
    nlo.xfull[0]=50.0; nlo.xlower[0]=0.0; nlo.xupper[0]=200.0; nlo.xdelta[0]=10.0; nlo.xtol[0]=0.01;
    nlo.xfull[1]=0.08; nlo.xlower[1]=0.0; nlo.xupper[1]=1.0;   nlo.xdelta[1]=0.05; nlo.xtol[1]=0.0001;
    nlo.xfull[2]=50.0; nlo.xlower[2]=0.0; nlo.xupper[2]=200.0; nlo.xdelta[2]=10.0; nlo.xtol[2]=0.01;
    nlo.xfull[3]=0.08; nlo.xlower[3]=0.0; nlo.xupper[3]=1.0;   nlo.xdelta[3]=0.05; nlo.xtol[3]=0.0001;
    if(verbose>4) {
      fprintf(status->fp, "  correct_wss := %g\n", nlo._fun(parNr, p, nlo.fundata));
      fprintf(status->fp, "  initial_point_wss := %g\n", nlo._fun(parNr, nlo.xfull, nlo.fundata));
      fflush(status->fp);
    }
    ret=nloptSimplex(&nlo, 0, status);
    if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
    if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
    if(ret!=TPCERROR_OK) {
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout; nloptFree(&nlo); return(121);
    }
    if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %e\n", i, nlo.xfull[i]);
    /* Check optimized parameters */
    ret=0;
    if(nlo.xfull[0]>nlo.xfull[2]) {
      for(int i=0; i<parNr; i++) {
        e=fabs(nlo.xfull[i]-p[i]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
    } else {
      int i;
      for(i=0; i<2; i++) {
        e=fabs(nlo.xfull[i]-p[i+2]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
      for(; i<parNr; i++) {
        e=fabs(nlo.xfull[i]-p[i-2]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
    }
    if(ret>0) {
      fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout;
      nloptFree(&nlo); return(122);
    }


    if(verbose>1) {
      fprintf(status->fp, "\n Level: impossible; trying to fit biexponential to monoexponential\n");
      fflush(status->fp);
    }
    p[0]=50.0; p[1]=0.1; p[2]=50.0; p[3]=0.1;
    for(unsigned int i=0; i<data.n; i++) {
      data.x[i]=(double)(1+i); 
      data.y[i]=p[0]*exp(-p[1]*data.x[i]) + p[2]*exp(-p[3]*data.x[i]);
      data.w[i]=1.0;
    }

    /* set initial parameters */
    nlo.xfull[0]=70.0; nlo.xlower[0]=0.0; nlo.xupper[0]=200.0; nlo.xdelta[0]=10.0; nlo.xtol[0]=0.01;
    nlo.xfull[1]=0.08; nlo.xlower[1]=0.0; nlo.xupper[1]=1.0;   nlo.xdelta[1]=0.05; nlo.xtol[1]=0.0001;
    nlo.xfull[2]=40.0; nlo.xlower[2]=0.0; nlo.xupper[2]=200.0; nlo.xdelta[2]=10.0; nlo.xtol[2]=0.01;
    nlo.xfull[3]=0.08; nlo.xlower[3]=0.0; nlo.xupper[3]=1.0;   nlo.xdelta[3]=0.05; nlo.xtol[3]=0.0001;
    if(verbose>4) {
      fprintf(status->fp, "  correct_wss := %g\n", nlo._fun(parNr, p, nlo.fundata));
      fprintf(status->fp, "  initial_point_wss := %g\n", nlo._fun(parNr, nlo.xfull, nlo.fundata));
      fflush(status->fp);
    }
    ret=nloptSimplex(&nlo, 0, status);
    if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
    if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
    if(ret!=TPCERROR_OK) {
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout; nloptFree(&nlo); return(131);
    }
    if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %e\n", i, nlo.xfull[i]);
    /* Check optimized parameters */
    ret=0;
    if(nlo.funval>1.0E-06) { // WSS should still be good
      ret++; if(verbose>2) {fprintf(status->fp, "  wss=%e\n", nlo.funval); fflush(status->fp);}
    }
    e=(nlo.xfull[0]+nlo.xfull[2])-(p[0]+p[2]); // Sum of p[0] and p[1] should be correct
    if(e>nlo.xtol[0]) {
      ret++; if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
    }
    if(nlo.xfull[0]>nlo.xfull[2]) { // The more important exponential should have correct decay
      e=fabs(nlo.xfull[1]-p[1]);
      if(e>nlo.xtol[1]) {
        ret++; 
        if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
      }
    } else {
      e=fabs(nlo.xfull[3]-p[3]);
      if(e>nlo.xtol[3]) {
        ret++; 
        if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
      }
    }
    if(ret>0) {
      fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout;
      nloptFree(&nlo); return(132);
    }


    nloptFree(&nlo);
  }

  if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
  status->fp=stdout;

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_nloptSimplexARRS(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>3) {
    char logfile[256];
    sprintf(logfile, "%s.log", __func__);
    status->fp=fopen(logfile, "w");
  }


  NLOPT nlo; nloptInit(&nlo);

  if(verbose>1) fprintf(status->fp, "\ncalling function with empty input\n");
  if(nloptSimplexARRS(NULL, 0, status)==TPCERROR_OK) return(1);
  if(nloptSimplexARRS(&nlo, 0, status)==TPCERROR_OK) return(2);

  int parNr, ret;

  if(verbose>1) fprintf(status->fp, "\n============\n Rosenbrock's valley with N=2, with good initial guess \n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  parNr=2;
  ret=nloptAllocate(&nlo, parNr);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(100);}
  /* set limits and tolerances */
  for(int i=0; i<parNr; i++) {
    nlo.xlower[i]=0.0; nlo.xupper[i]=5.0; nlo.xtol[i]=1.0E-05;
  }
  /* set initial guess */
  nlo.xfull[0]=0.95;
  nlo.xfull[1]=1.04;
  /* set object function */
  nlo._fun=nloptDejong2; nlo.maxFunCalls=50000;
  /* Optimize */
  ret=nloptSimplexARRS(&nlo, 4, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(101);}
  if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    double e=fabs(nlo.xfull[i]-1.0);
    if(e>nlo.xtol[i]) {
      ret++; if(verbose>2) fprintf(status->fp, "  |x[%d]-true|=%g\n", i, e);
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(102);
  }
  nloptFree(&nlo);

#if(0)
  if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
  status->fp=stdout;
  return(0);
#endif

  if(verbose>1) fprintf(status->fp, "\n============\n Rosenbrock's valley with N=2, without initial guess \n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  parNr=2;
  ret=nloptAllocate(&nlo, parNr);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(110);}
  /* set limits and tolerances */
  for(int i=0; i<parNr; i++) {
    nlo.xlower[i]=-5.0; nlo.xupper[i]=5.0; nlo.xtol[i]=1.0E-05;
  }
  /* do not set initial guess */
  for(int i=0; i<parNr; i++) nlo.xfull[i]=nan("");
  /* set object function */
  nlo._fun=nloptDejong2; nlo.maxFunCalls=50000;
  /* Optimize */
  ret=nloptSimplexARRS(&nlo, 4, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(111);}
  if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    double e=fabs(nlo.xfull[i]-1.0);
    if(e>nlo.xtol[i]) {
      ret++; if(verbose>2) fprintf(status->fp, "  |x[%d]-true|=%g\n", i, e);
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(112);
  }
  nloptFree(&nlo);

#if(0)
  if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
  status->fp=stdout;
  return(0);
#endif




  if(verbose>1) fprintf(status->fp, "\n============\n Rosenbrock's valley with N=3, without initial guesses \n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  parNr=3;
  ret=nloptAllocate(&nlo, parNr);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(200);}
  /* set limits and tolerances */
  for(int i=0; i<parNr; i++) {
    nlo.xlower[i]=-2.0; nlo.xupper[i]=4.0; nlo.xtol[i]=1.0E-05;
  }
  /* do not set initial guess */
  for(int i=0; i<parNr; i++) nlo.xfull[i]=nan("");
  /* set object function */
  nlo._fun=nloptDejong2; nlo.maxFunCalls=50000;
  /* Optimize */
  ret=nloptSimplexARRS(&nlo, 5, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(201);}
  if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    double e=fabs(nlo.xfull[i]-1.0);
    if(e>nlo.xtol[i]) {
      ret++; if(verbose>2) fprintf(status->fp, "  |x[%d]-true|=%g\n", i, e);
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(202);
  }
  nloptFree(&nlo);


  if(verbose>1) fprintf(status->fp, "\n===============\n Rosenbrock's valley with N=4, with good initial guesses \n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  parNr=4;
  ret=nloptAllocate(&nlo, parNr);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(300);}
  /* set limits and tolerances */
  for(int i=0; i<parNr; i++) {
    nlo.xlower[i]=-2.0; nlo.xupper[i]=4.0; nlo.xtol[i]=1.0E-05;
  }
  /* Since there is local AND global minimum, we'll set good initial guess */
  nlo.xfull[0]=1.05;
  nlo.xfull[1]=0.96;
  nlo.xfull[2]=0.98;
  nlo.xfull[3]=1.03;
  /* set object function */
  nlo._fun=nloptDejong2; nlo.maxFunCalls=50000;
  /* Optimize */
  ret=nloptSimplexARRS(&nlo, 0, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(301);}
  if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    double e=fabs(nlo.xfull[i]-1.0);
    if(e>nlo.xtol[i]) {
      ret++; if(verbose>2) fprintf(status->fp, "  |x[%d]-true|=%g\n", i, e);
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(302);
  }
  nloptFree(&nlo);


  if(verbose>1) fprintf(status->fp, "\n===============\n Rosenbrock's valley with N=4, two of which fixed \n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  parNr=4;
  ret=nloptAllocate(&nlo, parNr);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(310);}
  /* set limits and tolerances */
  for(int i=0; i<parNr; i++) {
    nlo.xlower[i]=-5.0; nlo.xupper[i]=5.0; nlo.xtol[i]=1.0E-05;
  }
  /* Since there is local AND global minimum, we'll set good initial guess */
  nlo.xfull[0]=1.0; nlo.xlower[0]=nlo.xupper[0]=nlo.xfull[0]; nlo.xtol[0]=0.0;
  nlo.xfull[1]=-2.06;
  nlo.xfull[2]=1.0; nlo.xlower[2]=nlo.xupper[2]=nlo.xfull[2]; nlo.xtol[2]=0.0;
  nlo.xfull[3]=5.0;
  /* set object function */
  nlo._fun=nloptDejong2; nlo.maxFunCalls=50000;
  /* Optimize */
  ret=nloptSimplexARRS(&nlo, 0, status);
  if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
  if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
  if(ret!=TPCERROR_OK) {nloptFree(&nlo); return(311);}
  if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %g\n", i, nlo.xfull[i]);
  /* Check optimized parameters */
  ret=0;
  for(int i=0; i<parNr; i++) {
    double e=fabs(nlo.xfull[i]-1.0);
    if(e>nlo.xtol[i]) {
      ret++; if(verbose>2) fprintf(status->fp, "  |x[%d]-true|=%g\n", i, e);
    }
  }
  if(ret>0) {
    fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
    status->fp=stdout;
    nloptFree(&nlo); return(312);
  }
  nloptFree(&nlo);






  if(verbose>1) {
    fprintf(status->fp, "\n ===============================================\n");
    fprintf(status->fp, "\n Test bi-exponential curve fitting \n"); fflush(status->fp);
  }
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  {
    NLOPT_DATA data; nloptdataInit(&data);
    data.n=20;
    double buf[4*data.n]; data.x=buf; data.y=buf+data.n; data.w=buf+2*data.n; data.sy=buf+3*data.n;
    int parNr=4; double p[parNr]; p[0]=80.0; p[1]=0.2; p[2]=20.0; p[3]=0.02;
    for(unsigned int i=0; i<data.n; i++) {
      data.x[i]=(double)(1+i); 
      data.y[i]=p[0]*exp(-p[1]*data.x[i]) + p[2]*exp(-p[3]*data.x[i]);
      data.w[i]=1.0;
    }

    NLOPT nlo; nloptInit(&nlo);
    if(nloptAllocate(&nlo, parNr)!=TPCERROR_OK) {nloptFree(&nlo); return(1000);}

    if(verbose>1) {
      fprintf(status->fp, "\n Level: easy; limits force into single optimum \n");
      fflush(status->fp);
    }
    /* set initial parameters */
    nlo.xfull[0]=91.2;  nlo.xlower[0]=50.0; nlo.xupper[0]=200.0; nlo.xtol[0]=0.01;
    nlo.xfull[1]=0.35;  nlo.xlower[1]=0.01; nlo.xupper[1]=1.0;   nlo.xtol[1]=0.001;
    nlo.xfull[2]=11.9;  nlo.xlower[2]=0.0;  nlo.xupper[2]=50.0;  nlo.xtol[2]=0.01;
    nlo.xfull[3]=0.006; nlo.xlower[3]=0.0;  nlo.xupper[3]=0.1;   nlo.xtol[3]=0.00001;
    nlo._fun=nloptBiexp; nlo.fundata=&data; nlo.maxFunCalls=10000;
    if(verbose>100) {data.verbose=verbose-100; data.fp=stdout;}
    if(verbose>4) {
      fprintf(status->fp, "  correct_wss := %g\n", nlo._fun(parNr, p, nlo.fundata));
      fprintf(status->fp, "  initial_point_wss := %g\n", nlo._fun(parNr, nlo.xfull, nlo.fundata));
      fflush(status->fp);
    }
    ret=nloptSimplexARRS(&nlo, 0, status);
    if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
    if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
    if(ret!=TPCERROR_OK) {
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout; nloptFree(&nlo); return(1101);
    }
    if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %e\n", i, nlo.xfull[i]);
    /* Check optimized parameters */
    ret=0;
    for(int i=0; i<parNr; i++) {
      double e=fabs(nlo.xfull[i]-p[i]);
      if(e>nlo.xtol[i]) {
        ret++; 
        if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
      }
    }
    if(ret>0) {
      fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout;
      nloptFree(&nlo); return(1102);
    }


    if(verbose>1) {
      fprintf(status->fp, "\n ---------------------------------------------------------------\n");
      fprintf(status->fp, "\n Level: moderate; limits allow dual optimum but initial values\n");
      fprintf(status->fp, " prefer one of the optima\n");
      fflush(status->fp);
    }
    /* set initial parameters */
    nlo.xfull[0]=78.0; nlo.xlower[0]=0.0; nlo.xupper[0]=200.0; nlo.xtol[0]=0.01;
    nlo.xfull[1]=0.26; nlo.xlower[1]=0.0; nlo.xupper[1]=1.0;   nlo.xtol[1]=0.0001;
    nlo.xfull[2]=11.0; nlo.xlower[2]=0.0; nlo.xupper[2]=200.0; nlo.xtol[2]=0.01;
    nlo.xfull[3]=0.03; nlo.xlower[3]=0.0; nlo.xupper[3]=1.0;   nlo.xtol[3]=0.0001;
    if(verbose>4) {
      fprintf(status->fp, "  correct_wss := %g\n", nlo._fun(parNr, p, nlo.fundata));
      fprintf(status->fp, "  initial_point_wss := %g\n", nlo._fun(parNr, nlo.xfull, nlo.fundata));
      fflush(status->fp);
    }
    nlo.maxFunCalls=10000;
    ret=nloptSimplexARRS(&nlo, 0, status);
    if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
    if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
    if(ret!=TPCERROR_OK) {
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout; nloptFree(&nlo); return(1111);
    }
    if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %e\n", i, nlo.xfull[i]);
    /* Check optimized parameters */
    ret=0;
    if(nlo.xfull[0]>nlo.xfull[2]) {
      for(int i=0; i<parNr; i++) {
        double e=fabs(nlo.xfull[i]-p[i]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
    } else {
      int i;
      for(i=0; i<2; i++) {
        double e=fabs(nlo.xfull[i]-p[i+2]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
      for(; i<parNr; i++) {
        double e=fabs(nlo.xfull[i]-p[i-2]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
    }
    if(ret>0) {
      fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout;
      nloptFree(&nlo); return(1112);
    }


    if(verbose>1) {
      fprintf(status->fp, "\n ---------------------------------------------------------------\n");
      fprintf(status->fp, "\n Level: difficult; limits allow dual optimum, and initial values\n");
      fprintf(status->fp, " do not clearly prefer either of the optima\n");
      fflush(status->fp);
    }
    /* set initial parameters */
    nlo.xfull[0]=50.0; nlo.xlower[0]=0.0; nlo.xupper[0]=100.0; nlo.xtol[0]=0.01;
    nlo.xfull[1]=0.08; nlo.xlower[1]=0.0; nlo.xupper[1]=1.0;   nlo.xtol[1]=0.0001;
    nlo.xfull[2]=50.0; nlo.xlower[2]=0.0; nlo.xupper[2]=100.0; nlo.xtol[2]=0.01;
    nlo.xfull[3]=0.08; nlo.xlower[3]=0.0; nlo.xupper[3]=1.0;   nlo.xtol[3]=0.0001;
    if(verbose>4) {
      fprintf(status->fp, "  correct_wss := %g\n", nlo._fun(parNr, p, nlo.fundata));
      fprintf(status->fp, "  initial_point_wss := %g\n", nlo._fun(parNr, nlo.xfull, nlo.fundata));
      fflush(status->fp);
    }
    nlo.maxFunCalls=10000;
    ret=nloptSimplexARRS(&nlo, 0, status);
    if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
    if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
    if(ret!=TPCERROR_OK) {
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout; nloptFree(&nlo); return(1121);
    }
    if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %e\n", i, nlo.xfull[i]);
    /* Check optimized parameters */
    ret=0;
    if(nlo.xfull[0]>nlo.xfull[2]) {
      for(int i=0; i<parNr; i++) {
        double e=fabs(nlo.xfull[i]-p[i]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
    } else {
      int i;
      for(i=0; i<2; i++) {
        double e=fabs(nlo.xfull[i]-p[i+2]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
      for(; i<parNr; i++) {
        double e=fabs(nlo.xfull[i]-p[i-2]);
        if(e>nlo.xtol[i]) {
          ret++; 
          if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
        }
      }
    }
    if(ret>0) {
      fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout;
      nloptFree(&nlo); return(1122);
    }


    if(verbose>1) {
      fprintf(status->fp, "\n ---------------------------------------------------------------\n");
      fprintf(status->fp, "\n Level: impossible; trying to fit biexponential to monoexponential\n");
      fflush(status->fp);
    }
    p[0]=50.0; p[1]=0.1; p[2]=50.0; p[3]=0.1;
    for(unsigned int i=0; i<data.n; i++) {
      data.x[i]=(double)(1+i); 
      data.y[i]=p[0]*exp(-p[1]*data.x[i]) + p[2]*exp(-p[3]*data.x[i]);
      data.w[i]=1.0;
    }

    /* set initial parameters */
    nlo.xfull[0]=70.0; nlo.xlower[0]=0.0; nlo.xupper[0]=200.0; nlo.xtol[0]=0.01;
    nlo.xfull[1]=0.08; nlo.xlower[1]=0.0; nlo.xupper[1]=1.0;   nlo.xtol[1]=0.0001;
    nlo.xfull[2]=40.0; nlo.xlower[2]=0.0; nlo.xupper[2]=200.0; nlo.xtol[2]=0.01;
    nlo.xfull[3]=0.08; nlo.xlower[3]=0.0; nlo.xupper[3]=1.0;   nlo.xtol[3]=0.0001;
    if(verbose>4) {
      fprintf(status->fp, "  correct_wss := %g\n", nlo._fun(parNr, p, nlo.fundata));
      fprintf(status->fp, "  initial_point_wss := %g\n", nlo._fun(parNr, nlo.xfull, nlo.fundata));
      fflush(status->fp);
    }
    nlo.maxFunCalls=100000;
    ret=nloptSimplexARRS(&nlo, 0, status);
    if(verbose>2) fprintf(status->fp, "  -> ret := %d\n", ret);
    if(verbose>3) fprintf(status->fp, "  -> function_calls := %d\n", nlo.funCalls);
    if(ret!=TPCERROR_OK) {
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout; nloptFree(&nlo); return(1131);
    }
    if(verbose>2) for(int i=0; i<parNr; i++) fprintf(status->fp, "  x[%d] := %e\n", i, nlo.xfull[i]);
    /* Check optimized parameters */
    ret=0;
    if(nlo.funval>1.0E-06) { // WSS should still be good
      ret++; if(verbose>2) {fprintf(status->fp, "  wss=%e\n", nlo.funval); fflush(status->fp);}
    }
    double e=(nlo.xfull[0]+nlo.xfull[2])-(p[0]+p[2]); // Sum of p[0] and p[1] should be correct
    if(e>nlo.xtol[0]) {
      ret++; if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
    }
    if(nlo.xfull[0]>nlo.xfull[2]) { // The more important exponential should have correct decay
      e=fabs(nlo.xfull[1]-p[1]);
      if(e>nlo.xtol[1]) {
        ret++; 
        if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
      }
    } else {
      e=fabs(nlo.xfull[3]-p[3]);
      if(e>nlo.xtol[3]) {
        ret++; 
        if(verbose>2) {fprintf(status->fp, "  |x-true|=%e\n", e); fflush(status->fp);}
      }
    }
    if(ret>0) {
      fprintf(stderr, "%s did not converge to correct minimum.\n", __func__);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
      if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
      status->fp=stdout;
      nloptFree(&nlo); return(1132);
    }


    nloptFree(&nlo);
  }



  if(status->fp!=NULL && status->fp!=stdout && status->fp!=stderr) fclose(status->fp);
  status->fp=stdout;

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
