/** @file par.c
 *  @brief PAR struct processing.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcmodels.h"
#include "tpcift.h"
/*****************************************************************************/
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the PAR struct before any use.
    @author Vesa Oikonen
    @sa parrInit, parnInit, parFree, parAllocate, parRead
 */
void parInit(
  /** Pointer to PAR */
  PAR *par
) {
  if(par==NULL) return;
  par->format=PAR_FORMAT_UNKNOWN;
  par->tacNr=par->_tacNr=0;
  par->parNr=par->_parNr=0;
  par->n=NULL;
  par->r=NULL;
  iftInit(&par->h);
}
/** Initiate the PARR struct before any use.
    @author Vesa Oikonen
    @sa parInit, parnInit
 */
void parrInit(
  /** Pointer to PARR */
  PARR *parr
) {
  if(parr==NULL) return;
  parr->model=0;
  parr->name[0]='\0';
  parr->start=parr->end=nan("");
  parr->fitNr=0;
  parr->dataNr=0;
  parr->p=parr->sd=parr->cl1=parr->cl2=NULL;
  parr->wss=nan("");
  parr->sw=0;
}
/** Initiate the PARN struct before any use.
    @author Vesa Oikonen
    @sa parInit, parrInit
 */
void parnInit(
  /** Pointer to PARN */
  PARN *parn
) {
  if(parn==NULL) return;
  parn->name[0]='\0';
  parn->unit=UNIT_UNKNOWN;
  parn->lim1=parn->lim2=parn->tol=nan("");
  parn->sw=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for PAR. All data is cleared.
   @sa parInit, parAllocate, parRead
 */
void parFree(
  /** Pointer to PAR struct */
  PAR *par
) {
  if(par==NULL) return;
  for(int i=0; i<par->_tacNr; i++) parrFree(par->r+i);
  free(par->r);
  free(par->n);
  iftFree(&par->h);
  parInit(par);
}
/** Free memory allocated for PARR. All data is cleared. */
void parrFree(
  /** Pointer to PARR struct */
  PARR *parr
) {
  if(parr==NULL) return;
  free(parr->p);
  free(parr->sd);
  free(parr->cl1);
  free(parr->cl2);
  parrInit(parr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for PAR data.

    Any previous contents are deleted.

    @return Returns TPCERROR status.
    @sa parInit, parFree, parAllocateMore, parAllocateWithTAC
 */
int parAllocate(
  /*! Pointer to initiated PAR struct data; any old contents are deleted.
   *  tacNr and parNr inside the struct are set to or kept at zero. */
  PAR *par,
  /*! Nr of parameters to allocate */
  int parNr,
  /*! Nr of parameter sets (TACs) to allocate */
  int tacNr
) {
  if(par==NULL) return TPCERROR_FAIL;
  /* Delete any previous contents */
  parFree(par);
  /* If no memory is requested, then just return */
  if(parNr<1 && tacNr<1) return TPCERROR_OK;
  if(parNr<1 || tacNr<1) return TPCERROR_FAIL;

  /* Allocate memory for PARR data */
  int ret;
  par->r=(PARR*)malloc(tacNr*sizeof(PARR));
  if(par->r==NULL) return TPCERROR_OUT_OF_MEMORY;
  for(int i=0; i<tacNr; i++) {
    parrInit(&par->r[i]);
    ret=parrAllocate(par->r+i, parNr); if(ret!=TPCERROR_OK) return ret;
  }
  par->_tacNr=tacNr;

  /* Allocate memory for parameter names and units */
  par->n=(PARN*)malloc(parNr*sizeof(PARN));
  if(par->n==NULL) return TPCERROR_OUT_OF_MEMORY;
  for(int i=0; i<parNr; i++) parnInit(&par->n[i]);
  par->_parNr=parNr;

  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for PARR data.
    Any previous contents are deleted.
    @sa parrInit, parrFree, parAllocate
    @return Returns TPCERROR status.
 */
int parrAllocate(
  /*! Pointer to initiated PARR struct data; any old contents are deleted. */
  PARR *parr,
  /*! Nr of parameters to allocate. */
  int parNr
) {
  if(parr==NULL) return TPCERROR_FAIL;
  /* Delete any previous contents */
  parrFree(parr);
  /* If no memory is requested, then just return */
  if(parNr<1) return TPCERROR_OK;

  /* Allocate memory for PARR data */
  parr->p=malloc(parNr*sizeof(double));
  parr->sd=malloc(parNr*sizeof(double));
  parr->cl1=malloc(parNr*sizeof(double));
  parr->cl2=malloc(parNr*sizeof(double));
  if(parr->p==NULL || parr->sd==NULL || parr->cl1==NULL || parr->cl2==NULL)
    return TPCERROR_OUT_OF_MEMORY;
  /* Initiate values to NaNs */
  for(int i=0; i<parNr; i++)
    parr->p[i]=parr->sd[i]=parr->cl1[i]=parr->cl2[i]=nan("");

  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate more memory for PAR data.
    Previous contents are preserved.
    @return Returns TPCERROR status.
    @sa parAllocate, parInit, parRead, parFree
 */
int parAllocateMore(
  /*! Pointer to initiated and previously allocated PAR struct data; any old 
      contents are preserved, including tacNr and parNr. */
  PAR *par,
  /*! Nr of additional parameters to allocate; if struct contains unused space 
      for requested parameters already, then nothing is done. */
  int parNr,
  /*! Nr of additional parameter sets (TACs) to allocate; if struct contains 
      unused space for requested parameters already, then nothing is done. */
  int tacNr
) {
  if(par==NULL) return TPCERROR_FAIL;
  if(par->_parNr<1 || par->_tacNr<1) return TPCERROR_FAIL;
  if(parNr<0 || tacNr<0) return TPCERROR_FAIL;
  /* Check if there is enough space already */
  int newTacNr, addTacNr, newParNr, addParNr;
  newTacNr=par->tacNr+tacNr; addTacNr=newTacNr-par->_tacNr;
  newParNr=par->parNr+parNr; addParNr=newParNr-par->_parNr;
  if(addTacNr<=0 && addParNr<=0) return TPCERROR_OK;

  /* Reallocate memory for parameter names and units, if necessary */
  if(addParNr>0) {
    PARN *parnPtr;
    parnPtr=realloc(par->n, sizeof(PARN)*newParNr);
    if(parnPtr==NULL) return TPCERROR_OUT_OF_MEMORY;
    par->n=parnPtr;
    parnPtr+=par->_parNr; for(int i=0; i<addParNr; i++) parnInit(parnPtr++);
  }

  /* Reallocate memory for parameters of previous TACs, if necessary */
  if(addParNr>0) {
    int i, j;
    for(i=0; i<par->tacNr; i++) {
      par->r[i].p=realloc(par->r[i].p, sizeof(double)*newParNr);
      par->r[i].sd=realloc(par->r[i].sd, sizeof(double)*newParNr);
      par->r[i].cl1=realloc(par->r[i].cl1, sizeof(double)*newParNr);
      par->r[i].cl2=realloc(par->r[i].cl2, sizeof(double)*newParNr);
      if(par->r[i].p==NULL || par->r[i].sd==NULL || par->r[i].cl1==NULL || par->r[i].cl2==NULL) 
        return TPCERROR_OUT_OF_MEMORY;
      for(j=par->parNr; j<newParNr; j++)
        par->r[i].p[j]=par->r[i].sd[j]=par->r[i].cl1[j]=par->r[i].cl2[j]=nan("");
    }
  }

  /* Allocate memory for more TACs, if necessary */
  if(addTacNr>0) {
    PARR *parrPtr;
    parrPtr=realloc(par->r, sizeof(PARR)*newTacNr);
    if(parrPtr==NULL) return TPCERROR_OUT_OF_MEMORY;
    par->r=parrPtr;
    parrPtr+=par->_tacNr; 
    for(int i=0; i<addTacNr; i++, parrPtr++) {
      parrInit(parrPtr);
      if(parrAllocate(parrPtr, newParNr)!=TPCERROR_OK)
        return TPCERROR_OUT_OF_MEMORY;
    }
  }
  
  par->_parNr=newParNr;
  par->_tacNr=newTacNr;
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has WSS.
    @sa parIsFitNr, parIsDataNr, parCLWithPar, parSDWithPar, parIsOptcrit
    @return Returns 1 if WSS exists, and 0 if not (or in case of an error).
 */ 
int parIsWSS(
  /** Pointer to parameter data. */
  PAR *par
) {
  if(par==NULL || par->tacNr<1) return(0);
  for(int ri=0; ri<par->tacNr; ri++) if(!isnan(par->r[ri].wss)) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has fit range specified.
    @sa parIsFitNr, parIsDataNr, parIsWSS
    @return Returns 0 if not, 1 if the same in each set, and 2 if variable.
 */ 
int parIsFitRange(
  /** Pointer to parameter data. */
  PAR *par
) {
  if(par==NULL || par->tacNr<1) return(0);
  int ri, ret=0;
  for(ri=0; ri<par->tacNr; ri++) {
    if(!isnan(par->r[ri].start) || !isnan(par->r[ri].end)) ret=1;
    if(ri==0 || ret==0) continue;
    if(!doubleMatch(par->r[ri].start, par->r[0].start, 0.01)) {ret=2; break;}
    if(!doubleMatch(par->r[ri].end,   par->r[0].end,   0.01)) {ret=2; break;}
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has model id specified.
    @sa parIsFitNr, parIsWSS, parRead, parIsOptcrit
    @return Returns 0 if not, 1 if the same in each set, and 2 if variable.
 */ 
int parIsModel(
  /** Pointer to parameter data. */
  PAR *par
) {
  if(par==NULL || par->tacNr<1) return(0);
  int ri, ret=0;
  for(ri=0; ri<par->tacNr; ri++) {
    if(ret==0 && par->r[ri].model>0) ret=1;
    if(ri>0 && par->r[ri].model!=par->r[ri-1].model) {ret=2; break;}
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if specified parameter has SD in at least one of the regions.
    @sa parCLWithPar, parIsWSS
    @return Returns 1 if SD exists, and 0 if not (or in case of an error).
 */ 
int parSDWithPar(
  /** Pointer to parameter data. */
  PAR *par,
  /** Index of parameter [0..parNr-1] to check. */
  int pi
) {
  //printf("%s(%d)\n", __func__, pi); fflush(stdout);
  if(par==NULL || par->tacNr<1 || pi<0 || pi>=par->parNr) return(0);
  for(int i=0; i<par->tacNr; i++) if(!isnan(par->r[i].sd[pi])) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if specified parameter has confidence limits in at least one of the regions.
    @sa parSDWithPar, parIsWSS
    @return Returns 1 if CLs exist, and 0 if not (or in case of an error).
 */ 
int parCLWithPar(
  /** Pointer to parameter data. */
  PAR *par,
  /** Index of parameter [0..parNr-1] to check. */
  int pi
) {
  if(par==NULL || par->tacNr<1 || pi<0 || pi>=par->parNr) return(0);
  for(int i=0; i<par->tacNr; i++)
    if(!isnan(par->r[i].cl1[pi]) && !isnan(par->r[i].cl2[pi])) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has fitNr specified.
    @sa parIsDataNr, parIsWSS
    @return Returns 0 if not, 1 if the same in each set, and 2 if variable.
 */ 
int parIsFitNr(
  /** Pointer to parameter data. */
  PAR *par
) {
  if(par==NULL || par->tacNr<1) return(0);
  int ri, ret=0;
  for(ri=0; ri<par->tacNr; ri++) {
    if(ret==0 && par->r[ri].fitNr>0) ret=1;
    if(ri>0 && par->r[ri].fitNr!=par->r[ri-1].fitNr) {ret=2; break;}
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if any of parameter sets has dataNr specified.
    @sa parIsFitNr, parIsWSS
    @return Returns 0 if not, 1 if the same in each set, and 2 if variable.
 */ 
int parIsDataNr(
  /** Pointer to parameter data */
  PAR *par
) {
  if(par==NULL || par->tacNr<1) return(0);
  int ri, ret=0;
  for(ri=0; ri<par->tacNr; ri++) {
    if(ret==0 && par->r[ri].dataNr>0) ret=1;
    if(ri>0 && par->r[ri].dataNr!=par->r[ri-1].dataNr) {ret=2; break;}
  }
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if PAR data contains optimality criterion.
    @sa parIsModel, parIsWSS, studynrFromFilename
    @return Returns pointer to optimization criterion name, or NULL if not found.
 */ 
char *parIsOptcrit(
  /** Pointer to parameter data. */
  PAR *par
) {
  if(par==NULL || par->h.keyNr<1) return((char*)NULL);
  int li=iftFindKey(&par->h, "optimality_criterion", 0);
  if(li<0) li=iftFindKey(&par->h, "optcrit", 0);
  if(li<0) return((char*)NULL);
  if(par->h.item[li].value==NULL || strlen(par->h.item[li].value)<1) return((char*)NULL);
  return(par->h.item[li].value);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if PAR data contains study number.
    @sa parSetStudyNr, parIsModel, parIsWSS, studynrFromFilename, parEnsureNames
    @return Returns pointer to study number, or NULL if not found.
 */ 
char *parIsStudyNr(
  /** Pointer to parameter data. */
  PAR *par
) {
  if(par==NULL || par->h.keyNr<1) return((char*)NULL);
  int li=iftFindKey(&par->h, "studynr", 0);
  if(li<0) li=iftFindKey(&par->h, "study_number", 0);
  if(li<0) return((char*)NULL);
  if(par->h.item[li].value==NULL || strlen(par->h.item[li].value)<1) return((char*)NULL);
  return(par->h.item[li].value);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set study number in PAR data, overwriting any previous study number.
    @sa parIsStudyNr, tacSetHeaderStudynr, tacGetHeaderStudynr, studynrFromFilename
    @return enum tpcerror (TPCERROR_OK when successful).
 */ 
int parSetStudyNr(
  /** Pointer to parameter data. */
  PAR *par,
  /** Pointer to string containing the study number;
      enter NULL or "" if you only want to delete studynr in header. */
  const char *s
) {
  if(par==NULL) return TPCERROR_FAIL;

  /* Find and delete previous studynr(s) */
  int i=0, start=0;
  while(1) {
    i=iftFindKey(&par->h, "studynr", start);
    if(i<0) i=iftFindKey(&par->h, "study_number", start);
    if(i<0) i=iftFindKey(&par->h, "study number", start);
    if(i<0) break;
    iftDelete(&par->h, i); if(i>start) start=i;
  }

  /* If new studynr is empty, then just return */
  if(s==NULL || *s=='\0') return(TPCERROR_OK);
  
  /* Otherwise create new key and value */
  return(iftPut(&par->h, "studynr", s, (char)1, NULL));
}
/*****************************************************************************/

/*****************************************************************************/
/** Ensure that PAR struct contains a name for each TAC and parameters.
    If not available, then index+1 is written as name.
    @sa parSetStudyNr, parSortByName
 */ 
void parEnsureNames(
  /** Pointer to parameter data. */
  PAR *d
) {
  if(d==NULL || d->tacNr<1 || d->parNr<1) return;
  /* Ensure TAC names */
  int u, n;
  u=d->tacNr; n=1; while((u/=10)>=1) n++;
  if(n>MAX_TACNAME_LEN) n=MAX_TACNAME_LEN;
  for(int i=0; i<d->tacNr; i++) {
    if(strlen(d->r[i].name)<1 || strcmp(d->r[i].name, ".")==0)
      sprintf(d->r[i].name, "%0*d", n, 1+i);
  }
  /* Ensure parameter names */
  u=d->parNr; n=1; while((u/=10)>=1) n++;
  if(n>MAX_PARNAME_LEN-1) n=MAX_PARNAME_LEN-1;
  for(int i=0; i<d->parNr; i++) {
    if(strlen(d->n[i].name)<1 || strcmp(d->n[i].name, ".")==0)
      sprintf(d->n[i].name, "P%0*d", n, 1+i);
  }
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Deletes the specified parameter (column) from data.
    @return Returns TPCERROR status.
    @sa parDeleteTAC
 */ 
int parDeletePar(
  /** Pointer to parameter data. */
  PAR *par,
  /** Index of parameter [0..parNr-1] to remove. */
  int pi
) {
  if(par==NULL || par->tacNr<1 || pi<0 || pi>=par->parNr) return(TPCERROR_FAIL);
  if(pi==par->parNr-1) {par->parNr--; return(TPCERROR_OK);}

  int i, j;

  /* Parameter name and unit */
  PARN parn;
  memcpy(&parn, &par->n[pi], sizeof(PARN));
  for(i=pi+1; i<par->parNr; i++)
    memcpy(&par->n[i-1], &par->n[i], sizeof(PARN));
  memcpy(&par->n[par->parNr-1], &parn, sizeof(PARN));

  /* Parameter values */
  double p, sd, cl1, cl2;
  for(j=0; j<par->tacNr; j++) {
    p=par->r[j].p[pi]; sd=par->r[j].sd[pi];
    cl1=par->r[j].cl1[pi]; cl2=par->r[j].cl2[pi];
    for(i=pi+1; i<par->parNr; i++) {
      par->r[j].p[i-1]=par->r[j].p[i]; par->r[j].sd[i-1]=par->r[j].sd[i];
      par->r[j].cl1[i-1]=par->r[j].cl1[i]; par->r[j].cl2[i-1]=par->r[j].cl2[i];
    }
    par->r[j].p[par->parNr-1]=p; par->r[j].sd[par->parNr-1]=sd;
    par->r[j].cl1[par->parNr-1]=cl1; par->r[j].cl2[par->parNr-1]=cl2;
  }

  par->parNr--;
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Deletes the specified TAC (row) from data.
    @return Returns TPCERROR status.
    @sa parDeletePar, parSortByName
 */ 
int parDeleteTAC(
  /** Pointer to parameter data. */
  PAR *par,
  /** Index of TAC [0..tacNr-1] to remove. */
  int ti
) {
  if(par==NULL || par->tacNr<1 || ti<0 || ti>=par->tacNr) return(TPCERROR_FAIL);
  /* Free the memory allocated for the TAC */
  parrFree(par->r+ti);
  /* If the last TAC is to be deleted, then just reduce the tacNr */
  if(ti==par->tacNr-1) {par->tacNr--; return(TPCERROR_OK);}
  /* Otherwise move the following TACs in its place */
  for(int i=ti+1; i<par->tacNr; i++) par->r[i-1]=par->r[i];
  par->tacNr--; parrInit(par->r+par->tacNr);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
