/** @file tpcpar.h
 *  @brief Header file for libtpcpar.
 *  @details Header file for model parameter library.
 *  @author Vesa Oikonen
 *  @copyright (c) Turku PET Centre
 */
#ifndef _TPCPAR_H_
#define _TPCPAR_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpccsv.h"
#include "tpcift.h"
#include "tpcmodels.h"
/*****************************************************************************/

/*****************************************************************************/
/** @public PAR file format codes; @sa par_format */
typedef enum {
  PAR_FORMAT_UNKNOWN,    ///< Unknown format
  PAR_FORMAT_RES,        ///< Model result format of Turku PET Centre
  PAR_FORMAT_FIT,        ///< Function fit format of Turku PET Centre
  PAR_FORMAT_IFT,        ///< Interfile-type data (supported for writing)
  PAR_FORMAT_CSV_INT,    ///< International CSV
  PAR_FORMAT_CSV_UK,     ///< UK CSV
  PAR_FORMAT_TSV_INT,    ///< International TSV (comma as decimal separator)
  PAR_FORMAT_TSV_UK,     ///< UK TSV (point as decimal separator)
  PAR_FORMAT_XML,        ///< XML format (currently not supported)
  PAR_FORMAT_HTML,       ///< HTML table format (currently not supported)
  PAR_FORMAT_LAST        ///< End of list
} parformat;
/*****************************************************************************/

/*****************************************************************************/
/** Parameters for one TAC.
   @sa PAR, PARN, TAC, TACC, parrInit, parrFree
 */
typedef struct PARR {
  /** Model or function id number */
  unsigned int model;
  /** Full name string, using '_' to separate any name tokens when necessary */
  char         name[MAX_TACNAME_LEN+1];
  /** Fit start time (min) */
  double       start;
  /** Fit end time (min) */
  double       end;
  /** The number of fitted parameters (excluding fixed parameters);
      must be < MAX_PETPAR_NR; needed for calculating AIC etc.
      @sa nloptFixedNr
   */
  int          fitNr;
  /** Number of fitted data points.
      @sa tacWSampleNr
   */
  int          dataNr;
  /** Array of parameters */
  double      *p;
  /** Parameter SD's   */
  double      *sd;
  /** Lower 95% confidence intervals */
  double      *cl1;
  /** Upper 95% confidence intervals */
  double      *cl2;
  /** (weighted) sum-of-squares */
  double       wss;
  /** Temporary switch */
  char         sw;
} PARR;
/*****************************************************************************/
/** Parameter name and unit.
   @sa PAR, PARR, TAC, TACC, parnInit, parnFree
 */
typedef struct PARN {
  /** Parameter name. */
  char         name[MAX_PARNAME_LEN+1];
  /** Parameter unit.
     @sa unit
   */
  int          unit;
  /** Parameter lower limit. */
  double       lim1;
  /** Parameter upper limit. */
  double       lim2;
  /** Parameter tolerance. */
  double       tol;
  /** Temporary switch. */
  char         sw;
} PARN;
/*****************************************************************************/
/** Set of model or function parameters.
   @sa PARR, PARN, TAC, IFT, CSV, parInit, parFree
 */
typedef struct {
  /** File format: PAR_FORMAT_* */
  int       format;
  /** Number of stored TAC parameter sets */
  int       tacNr;
  /** Number of allocated TAC parameter sets */
  int       _tacNr;
  /** Nr of parameter names and units */
  int       parNr;
  /** Nr of allocated parameter names and units; set also regional parNr */
  int       _parNr;
  /** List of parameter names and units */
  PARN     *n;
  /** Pointers to individual TAC parameters */
  PARR     *r;
  /** @brief Optional (but often useful) header information.
      @details Reserved keys:
       - studynr and study_number
       - timeunit and time_unit
       - unit and calibration_unit
       - radiopharmaceutical
       - isotope
       - injection_time and injection time
       - scan_start_time, scan start time, scan_start, and scan start
       - analysis_time
       - program
       - datafile
       - reffile
       - refname
       - plasmafile
       - plasmafile2
       - bloodfile
       - fitmethod
       - weighting
       - density
       - LC
       - beta
       - concentration
       - Vb
       - fA
       - E
       - model
       - dataNr
       - fitNr
       - fit_start
       - fit_end
   */
  IFT       h;
} PAR;
/*****************************************************************************/

/*****************************************************************************/
/* par */
extern void parInit(PAR *par);
extern void parrInit(PARR *parr);
extern void parnInit(PARN *parn);
extern void parFree(PAR *par);
extern void parrFree(PARR *parr);
extern int parAllocate(PAR *par, int parNr, int tacNr);
extern int parAllocateMore(PAR *par, int parNr, int tacNr);
extern int parrAllocate(PARR *parr, int parNr);
extern char *parIsStudyNr(PAR *par);
extern int parSetStudyNr(PAR *par, const char *s);
extern void parEnsureNames(PAR *d);
extern int parIsWSS(PAR *par);
extern int parIsFitRange(PAR *par);
extern int parIsModel(PAR *par);
extern int parSDWithPar(PAR *par, int pi);
extern int parCLWithPar(PAR *par, int pi);
extern int parIsFitNr(PAR *par);
extern int parIsDataNr(PAR *par);
extern int parDeletePar(PAR *par, int pi);
extern int parDeleteTAC(PAR *par, int ti);
extern char *parIsOptcrit(PAR *par);
/*****************************************************************************/

/*****************************************************************************/
/* pario */
extern char *parFormattxt(parformat c);
extern int parFormatIdentify(const char *s);
extern int parFormatFromExtension(const char *s);
extern char *parDefaultExtension(parformat c);
extern int parWrite(PAR *par, FILE *fp, parformat format, int extra, TPCSTATUS *status);
extern int parRead(PAR *par, const char *fname, TPCSTATUS *status);
extern void parListLimits(PAR *par, FILE *fp);
extern int parWriteLimits(PAR *par, const char *fname, const int verbose);
extern int parReadLimits(PAR *par, const char *fname, const int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* parcsv */
extern int parWriteCSV(PAR *par, FILE *fp, parformat format, int extra, TPCSTATUS *status);
extern int parReadCSV(PAR *par, CSV *csv, IFT *hdr, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* parift */
extern int parToIFT(PAR *par, IFT *ift, TPCSTATUS *status);
extern int parWriteIFT(PAR *par, FILE *fp, TPCSTATUS *status);
extern int parFromIFT(PAR *par, IFT *ift, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* parres */
extern int parWriteRES(PAR *par, FILE *fp, TPCSTATUS *status);
extern int parReadRES(PAR *par, CSV *csv, IFT *ift, TPCSTATUS *status);
extern int parPrintType(PAR *par, int parIndex);
/*****************************************************************************/

/*****************************************************************************/
/* parfit */
extern int parWriteFIT(PAR *par, FILE *fp, TPCSTATUS *status);
extern int parReadFIT(PAR *par, CSV *csv, IFT *ift, TPCSTATUS *status);
/*****************************************************************************/
/* parxmlio */
extern int parWriteXML(PAR *par, FILE *fp, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* parcomb */
int parSelectByAnother(PAR *d1, PAR *d2, int *pn, int *pr, TPCSTATUS *status);
int parCombineTACs(PAR *d1, PAR *d2, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* parcomp */
extern int parCompareParameterNames(
  PAR *d1, PAR *d2, const int i, const int casens, TPCSTATUS *status
);
extern int parCompareParameterUnits(PAR *d1, PAR *d2, const int i, TPCSTATUS *status);
extern int parCompareTacNames(
  PAR *d1, PAR *d2, const int i, const int casens, TPCSTATUS *status
);
extern int parCompareParameters(
  PAR *d1, PAR *d2, const int pi, const int ti,
  int checkpar, int checksd, int checkcl,
  const double test_abs, const double test_rel, TPCSTATUS *status
);
extern int parCompareWSS(
  PAR *d1, PAR *d2, const int ti,
  const double test_abs, const double test_rel, TPCSTATUS *status
);
/*****************************************************************************/

/*****************************************************************************/
/* parorder */
extern int parSortByName(PAR *d, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* parselect */
extern int parSelectTACs(PAR *d, const char *region_name, int reset, TPCSTATUS *status);
extern int parSelectedTACs(PAR *d);
extern int parSelectParameters(PAR *d, const char *par_name, int reset, TPCSTATUS *status);
extern int parSelectedParameters(PAR *d);
extern int parFindParameter(PAR *d, const char *par_name);
extern double parGetParameter(PAR *d, const char *par_name, const int ti);
extern int parGetParameterUnit(PAR *d, const char *par_name);
extern void parValueRange(PAR *d, const int i, double *pmin, double *pmax);
/*****************************************************************************/

/*****************************************************************************/
/* parexample */
extern int parExampleTTACs(PAR *d, int type, TPCSTATUS *status);
extern int parExamplePerfectBolus(PAR *d, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCPAR */
