/** @file shuffle.c
 *  @brief Random shuffle and related functions.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcrand.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Random shuffle

    Arrange the n elements of integer array in random order.
    Only effective if N is much smaller than UINT64_MAX.
    @pre Prepare the Mersenne Twister, for example using code
    MERTWI mt; mertwiInit(&mt); mertwiInitWithSeed64(&mt, mertwiSeed64());
    @sa randomShuffleUnsigned
 */
void randomShuffle(
  /** Integer array to be shuffled. */
  int *array, 
  /** Array length. */
  unsigned int n,
  /** Pointer to data structure for Mersenne Twister MT19937 pseudo-random number generator. */
  MERTWI *mt
) {
  if(n<=1 || array==NULL) return;
  unsigned int i, j;
  int tmp;
  for(i=0; i<n-1; i++) {
    j=i+mertwiRandomInt64(mt)/(UINT64_MAX/(n-i)+1);
    tmp=array[j]; array[j]=array[i]; array[i]=tmp;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Random shuffle
  
    Arrange the n elements of unsigned integer array in random order.
    Only effective if N is much smaller than UINT64_MAX.
    @pre Prepare the Mersenne Twister, for example using code
     MERTWI mt; mertwiInit(&mt); mertwiInitWithSeed64(&mt, mertwiSeed64());
    @sa randomShuffle, randomPermutation  
 */
void randomShuffleUnsigned(
  /** Integer array to be shuffled. */
  unsigned int *array, 
  /** Array length. */
  unsigned int n,
  /** Pointer to data structure for Mersenne Twister MT19937 pseudo-random number generator. */
  MERTWI *mt
) {
  if(n<=1 || array==NULL) return;
  unsigned int i, j, tmp;
  for(i=0; i<n-1; i++) {
    j=i+mertwiRandomInt64(mt)/(UINT64_MAX/(n-i)+1);
    tmp=array[j]; array[j]=array[i]; array[i]=tmp;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Random permutation for an array of signed integers.
 
    Given (allocated) array of length n is filled with random permutation of numbers
    (signed integers) in the range [a:n+a-1]; that is, each number once in random order.
    @pre Prepare the Mersenne Twister, for example using code
     MERTWI mt; mertwiInit(&mt); mertwiInitWithSeed64(&mt, mertwiSeed64());
    @sa randomPermutationUnsigned, randomShuffle
 */
void randomPermutation(
  /** Allocated array to be filled. */
  int *array, 
  /** Length of array. */
  unsigned int n,
  /** Smallest integer value. */ 
  int a,
  /** Pointer to data structure for Mersenne Twister MT19937 pseudo-random number generator. */
  MERTWI *mt
) {
  if(n<1 || array==NULL) return;
  int i;
  for(i=0; i<(int)n; i++) array[i]=i+a;
  randomShuffle(array, n, mt);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Random permutation for an array of unsigned integers.
 
    Given (allocated) array of length n is filled with random permutation of numbers
    (unsigned integers) in the range [a:n+a-1]; that is, each number once in random order.
    @pre Prepare the Mersenne Twister, for example using code
     MERTWI mt; mertwiInit(&mt); mertwiInitWithSeed64(&mt, mertwiSeed64());
    @sa randomPermutation, randomShuffleUnsigned
 */
void randomPermutationUnsigned(
  /** Allocated array to be filled. */
  unsigned int *array, 
  /** Length of array. */
  unsigned int n,
  /** Smallest integer value. */ 
  unsigned int a,
  /** Pointer to data structure for Mersenne Twister MT19937 pseudo-random number generator. */
  MERTWI *mt
) {
  if(n<1 || array==NULL) return;
  unsigned int i;
  for(i=0; i<n; i++) array[i]=i+a;
  randomShuffleUnsigned(array, n, mt);
}
/*****************************************************************************/

/*****************************************************************************/
