/** @file tpcrand.h
 *  @brief Header file for libtpcrand.
 *  @details Header file for library libtpcrand.
 *  @author Vesa Oikonen
 *  @copyright (c) Turku PET Centre
 */
#ifndef _TPCRAND_H_
#define _TPCRAND_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>
#include <stdint.h>
#include <inttypes.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Mersenne Twister state vector length */
#define TPCCLIB_MERTWI_NN 312
/** Mersenne Twister required constant */
#define TPCCLIB_MERTWI_A UINT64_C(0xB5026F5AA96619E9)
/** Structure needed in functions for Mersenne Twister MT19937.
    Contents are initiated in seed functions. */
typedef struct MERTWI {
  /** Constant N, set to by seed function. */
  unsigned int n;
  /** Constant M, set by seed function. */
  unsigned int m;
  /** Constant MATRIX_A, set by seed function. */
  uint64_t a;
  /** Constant UM, most significant 33 bits, set by seed function. */
  uint64_t um;
  /** Constant LM, least significant 31 bits, set by seed function. */
  uint64_t lm;
  /** The array for the state vector. */
  uint64_t mt[TPCCLIB_MERTWI_NN]; 
  /** Index of mt array; mti==NN+1 means mt[NN] is not initialized. */
  uint64_t mti;
} MERTWI;
/*****************************************************************************/

/*****************************************************************************/
/* mertwi */
extern uint32_t mertwiSeed32(void);
extern uint64_t mertwiSeed64(void);
extern void mertwiInit(MERTWI *mt);
extern void mertwiInitWithSeed64(MERTWI *mt, uint64_t seed);
extern void mertwiInitByArray64(MERTWI *mt, uint64_t init_key[], uint64_t key_length);
extern uint64_t mertwiRandomInt64(MERTWI *mt);
extern int64_t mertwiRandomInt63(MERTWI *mt);
extern double mertwiRandomDouble1(MERTWI *mt);
extern double mertwiRandomDouble2(MERTWI *mt);
extern double mertwiRandomDouble3(MERTWI *mt);
extern double mertwiRandomGaussian(MERTWI *mt);
extern double mertwiRandomExponential(MERTWI *mt, double mean);
extern int mertwiRandomBetween(MERTWI *mt, unsigned int nr, double *d, double low, double up, int type);
/*****************************************************************************/

/*****************************************************************************/
/* gaussdev */
extern unsigned int drandSeed(short int seed);
extern double drand();
extern int drandRange(unsigned int nr, double *d, double low, double up, int type);
extern double drandGaussian();
extern double drandExponential(double mean);
/*****************************************************************************/

/*****************************************************************************/
/* shuffle */
extern void randomShuffle(int *array, unsigned int n, MERTWI *mt);
extern void randomShuffleUnsigned(unsigned int *array, unsigned int n, MERTWI *mt);
extern void randomPermutation(int *array, unsigned int n, int a, MERTWI *mt);
extern void randomPermutationUnsigned(unsigned int *array, unsigned int n, unsigned int a, MERTWI *mt);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCRAND */
