/** @file pmodio.c
 *  @brief I/O functions for PMOD TAC file format.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file pointer in PMOD.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacReadPMOD, tacWrite
 */
int tacWritePMOD(
  /** Pointer to TAC structure, contents of which are to be written. */
  TAC *tac,
  /** File pointer to write to. */
  FILE *fp,
  /** Write (1) or do not write (0) also extra header fields found in IFT;
      note that these are not supported by other than TPC SW. */
  int extra,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(tac==NULL || tac->tacNr<1 || tac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  char *cptr, tunit[128], cunit[128];
  int n, prec=6;
  double v;

  /* Set units to format accepted by PMOD */
  if(tac->tunit==UNIT_SEC) strcpy(tunit, "seconds");
  else if(tac->tunit==UNIT_MIN) strcpy(tunit, "minutes");
  else strcpy(tunit, unitName(tac->tunit));
  if(tac->cunit==UNIT_UNITLESS) strcpy(cunit, "1/1");
  else strcpy(cunit, unitName(tac->cunit));
  if(strcasestr(cunit, "dL")==NULL) {
    /* Replace mL by cc, but only if unit does not contain 'dL' */
    cptr=strcasestr(cunit, "mL"); 
    if(cptr!=NULL) {*cptr='c'; cptr++; *cptr='c';}
  }

  /* Make sure that TAC names are available */
  if(verbose>2) printf("constructing TAC names\n");
  tacEnsureNames(tac);

  /* Write obligatory header line */
  if(verbose>2) printf("writing obligatory title lines\n");
  if(tac->isframe==0) n=fprintf(fp, "time[%s]", tunit);
  else n=fprintf(fp, "start[%s]\tend[%s]", tunit, cunit);
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  for(int ri=0; ri<tac->tacNr; ri++) {
    /* write TAC names, must not include space characters */
    strReplaceChar(tac->c[ri].name, ' ', '-');
    strReplaceChar(tac->c[ri].name, '\t', '-');
    n=fprintf(fp, "\t%s", tac->c[ri].name);
    if(n<1) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return TPCERROR_CANNOT_WRITE;
    }
    /* write calibration unit when necessary */
    if(ri==0 && tac->isframe==0) fprintf(fp, "[%s]", cunit);
  }
  if(tacIsWeighted(tac)) fprintf(fp, "\t%s", "weight");
  n=fprintf(fp, "\n"); if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Write data */
  if(verbose>2) printf("writing data table\n");
  for(int fi=0, n=0; fi<tac->sampleNr; fi++) {
    /* Time(s) (x, x1, x2) */
    if(tac->isframe==0) v=tac->x[fi]; else v=tac->x1[fi];
    if(isnan(v)) n=fprintf(fp, "."); else n=fprintf(fp, "%.5f", v);
    if(n<1) break;
    if(tac->isframe!=0) {
      v=tac->x2[fi];
      if(isnan(v)) n=fprintf(fp, "\t."); else n=fprintf(fp, "\t%.5f", v);
      if(n<1) break;
    }
    /* Concentrations (y values) */
    for(int ri=0; ri<tac->tacNr; ri++) {
      if(isnan(tac->c[ri].y[fi])) n=fprintf(fp, "\t.");
      else fprintf(fp, "\t%.*e", prec, tac->c[ri].y[fi]);
      if(n<1) break;
    }
    /* Weight */
    if(tacIsWeighted(tac)) {
      if(isnan(tac->w[fi])) n=fprintf(fp, "\t.");
      else n=fprintf(fp, "\t%.*e", prec, tac->w[fi]);
      if(n<1) break;
    }
    if(n>0) n=fprintf(fp, "\n");
    if(n<1) break;
  }
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  
  /* Write extra header, if requested */
  if(extra) {
    if(tac->h.keyNr>0) {
      int ret=iftWrite(&tac->h, fp, status);
      if(ret!=TPCERROR_OK) return ret;
    }
  }
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read PMOD format from CSV struct into TAC struct.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacWritePMOD, tacRead
 */
int tacReadPMOD(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to CSV from which data is read; if it contains only one column,
      then it is assumed to represent the first y column and x column is not filled. */
  CSV *csv,
  /** Pointer to possible header data, which, if available, is processed and copied to TAC too;
      enter NULL if not available. */
  IFT *hdr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) printf("%s()\n", __func__);

  tacFree(tac);

  if(csv==NULL || csv->row_nr<2 || csv->col_nr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  

  /* Check from the first field that data indeed is PMOD data */
  int isframe=-1;
  if(strncasecmp(csv->c[0].content, "Time", 4)==0) isframe=0;
  else if(strncasecmp(csv->c[0].content, "Start", 5)==0 &&
          strncasecmp(csv->c[1].content, "End", 3)==0) isframe=1;
  if(isframe<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Allocate memory for TAC data */
  int n, ret;
  n=csv->col_nr-1; if(isframe==1) n--;
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  ret=tacAllocate(tac, csv->row_nr-1, n);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return ret;
  tac->tacNr=n; tac->sampleNr=csv->row_nr-1;
  tac->isframe=isframe;

  /* Copy header to TAC struct */
  iftDuplicate(hdr, &tac->h);

  /* Get time unit from first field, inside parens */
  char *cptr;
  cptr=strchr(csv->c[0].content, '[');
  if(cptr==NULL) {
    tac->tunit=UNIT_UNKNOWN;
  } else {
    char *tmp=strdup(cptr+1);
    cptr=strrchr(tmp, ']'); if(cptr!=NULL) *cptr='\0';
    tac->tunit=unitIdentify(tmp);
    free(tmp);
  }

  /* Get concentration unit from 2nd field, inside parens */
  cptr=strchr(csv->c[1].content, '[');
  if(cptr==NULL) {
    tac->cunit=UNIT_UNKNOWN;
  } else {
    char *tmp=strdup(cptr+1);
    cptr=strrchr(tmp, ']'); if(cptr!=NULL) *cptr='\0';
    tac->cunit=unitIdentify(tmp);
    free(tmp);
  }
  
  /* Read TAC names from the title line */
  int i;
  n=0; if(isframe==0) i=1; else i=2;
  for(; i<csv->nr; i++) {
    if(csv->c[i].row!=0 || n>=tac->tacNr) break;
    /* Copy TAC name, removing unit if necessary */
    if(strchr(csv->c[i].content, '[')) {
      char *tmp=strdup(csv->c[i].content);
      cptr=strchr(tmp, '['); *cptr='\0';
      strlcpy(tac->c[n].name, tmp, MAX_TACNAME_LEN+1);
      free(tmp);
    } else {
      strlcpy(tac->c[n].name, csv->c[i].content, MAX_TACNAME_LEN+1);
    }
    if(strcmp(tac->c[n].name, ".")==0) strcpy(tac->c[n].name, "");
    n++;
  }
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  tac->tacNr=n;
  if(verbose>2) printf("  tacNr=%d\n  sampleNr=%d\n", tac->tacNr, tac->sampleNr);

  /* Copy x and y data from CSV into TAC struct */
  int fi=0, ri=0, oknr=0;
  ret=0;
  double v;
  for(i=0; i<csv->nr; i++) if(csv->c[i].row==1) break;
  for(; i<csv->nr; i++) {
    if(verbose>10) printf("i=%d\trow=%d\tcol=%d\n", i, csv->c[i].row, csv->c[i].col);
    fi=csv->c[i].row-1; if(fi<0 || fi>=tac->sampleNr) {ret++; continue;}
    if(csv->c[i].col<0) {ret++; continue;}
    ri=csv->c[i].col-1; if(tac->isframe) ri--; 
    if(ri>=tac->tacNr) {ret++; continue;}
    v=atofVerified(csv->c[i].content); if(!isnan(v)) oknr++;
    if(verbose>10) printf("   -> fi=%d\tri=%d\t=\t%g\n", fi, ri, v);
    if(ri<0) {
      if(tac->isframe) {
        if(ri==-2) tac->x1[fi]=v; else if(ri==-1) tac->x2[fi]=v;
      } else tac->x[fi]=v;
    } else tac->c[ri].y[fi]=v;
  }
  if(verbose>0 && ret>0)
    printf("%d error(s) in reading PMOD file format.\n", ret);
  if(oknr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  /* Set middle sample times when necessary */
  if(tac->isframe)
    for(fi=0; fi<tac->sampleNr; fi++)
      tac->x[fi]=0.5*(tac->x1[fi]+tac->x2[fi]);

  /* Delete any empty columns (no name, no values) from the end */
  int ci=tac->tacNr-1;
  while(ci>0) {
    if(strlen(tac->c[ci].name)>0) break;
    int i;
    for(i=0; i<tac->sampleNr; i++) if(!isnan(tac->c[ci].y[i])) break;
    if(i<tac->sampleNr) break;
    tac->tacNr--; ci--;
  }

  /* If units are not known, try to read units from header */
  if(tac->cunit==UNIT_UNKNOWN) tacGetHeaderUnit(tac, NULL);
  else tacSetHeaderUnit(&tac->h, UNIT_UNKNOWN);
  if(tac->tunit==UNIT_UNKNOWN) tacGetHeaderTimeunit(tac, NULL);
  else tacSetHeaderTimeunit(&tac->h, UNIT_UNKNOWN);

  /* Move column containing weights to its correct place in the struct */
  if(tac->tacNr>1) { // except if file contains only one TAC
    ret=tacWMove(tac, 1, status);
    if(ret!=TPCERROR_OK && ret!=TPCERROR_NO_WEIGHTS) {
      statusSet(status, __func__, __FILE__, __LINE__, ret);
      return(ret);
    }
  }
  if(tac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  tac->format=TAC_FORMAT_PMOD;
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
