/** @file sifio.c
 *  @brief I/O functions for Scan Information Files.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write SIF data stored in TAC struct in SIF format.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacReadSIF, tacWrite, TAC
 */
int tacWriteSIF(
  /** Pointer to TAC struct, contents of which are to be written;
      tacNr can be set to 0 to write only frame times. */
  TAC *tac,
  /** File pointer. */
  FILE *fp,
  /** Write (1) or do not write (0) also extra header fields found in tac->h;
      set this to 0, because extra header lines do not follow SIF format. */
  int extra,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(tac==NULL || tac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(verbose>0) printf("%s()\n", __func__);

  int fi, ret, n;
  int m2s=0; // 1=time units converted from min to sec
  
  /* Scan start time */
  char scan_start_time[20];
  ret=tacGetHeaderScanstarttime(&tac->h, scan_start_time, NULL);
  if(ret==TPCERROR_OK) {
    int yy, mm, dd, h, m, s;
    n=sscanf(scan_start_time, "%d-%d-%d %d:%d:%d", &yy, &mm, &dd, &h, &m, &s);
    if(n==6) 
      sprintf(scan_start_time, "%d/%d/%04d %02d:%02d:%02d", dd, mm, yy, h, m, s);
    else strcpy(scan_start_time, "1/1/1970 00:00:00");
  } else { // use default date & time
    strcpy(scan_start_time, "1/1/1970 00:00:00");
  }
  /* Column nr */
  int colNr=2; // frame start and end times
  colNr+=tac->tacNr;
  /* Study number */
  char studynr[MAX_STUDYNR_LEN+1];
  ret=tacGetHeaderStudynr(&tac->h, studynr, NULL);
  if(ret!=TPCERROR_OK) strcpy(studynr, ".");
  /* Isotope */
  char isotope[MAX_ISOTOPE_LEN+1];
  ret=tacGetHeaderIsotope(&tac->h, isotope, NULL);
  if(ret!=TPCERROR_OK) strcpy(studynr, "");
  /* Write header line */
  n=fprintf(fp, "%s %d %d 1 %s %s\n", 
            scan_start_time, tac->sampleNr, colNr, studynr, isotope);
  if(n<7) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Convert frame times to seconds because always sec in SIF */
  if(tac->tunit==UNIT_MIN) {m2s=1; tacXUnitConvert(tac, UNIT_SEC, status);}

  /* Check if frame times need to printed with decimals */
  int req_decimals=0;
  for(fi=1; fi<tac->sampleNr; fi++) {
    if(round(tac->x1[fi])==round(tac->x1[fi-1])) {req_decimals=1; break;}
    if(round(tac->x2[fi])==round(tac->x2[fi-1])) {req_decimals=1; break;}
  }

  /* Write data lines */
  ret=0;
  for(fi=0; fi<tac->sampleNr; fi++) {
    if(req_decimals==0) n=fprintf(fp, "%.0f %.0f", tac->x1[fi], tac->x2[fi]);
    else n=fprintf(fp, "%.1f %.1f", tac->x1[fi], tac->x2[fi]);
    if(n<3) {ret++; break;}
    for(int ci=0; ci<tac->tacNr; ci++) {
      n=fprintf(fp, " %.0f", tac->c[ci].y[fi]);
      if(n<2) {ret++; break;}
    }
    if(ret==0) fprintf(fp, "\n"); else break;
  }
  if(ret!=0) {
    if(m2s!=0) tacXUnitConvert(tac, UNIT_MIN, status);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
 
  /* Convert frame times back to min if necessary */
  if(m2s!=0) tacXUnitConvert(tac, UNIT_MIN, status);
  
  /* Write extra information */
  if(extra!=0) {
    /* do nothing at least for now */
  }
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read SIF from CSV structure into TAC structure.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacWriteSIF, tacRead, csvRead, TAC, CSV, tacWCopy, tacSetWeights
 */
int tacReadSIF(
  /** Pointer to TAC structure, contents of which are to be filled here. 
      tacNr will always be set to 2, even if count data is not available. 
      Trues will be saved or computed into a non-standard 3rd column, which is not included
      in tacNr, even if column is found in CSV.
   */
  TAC *tac,
  /** Pointer to CSV from which SIF data is read. */
  CSV *csv,
  /** Pointer to possible header data, which, if available, if processed and copied to TAC too; 
      enter NULL if not available. */
  IFT *hdr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  tacFree(tac);
  if(csv==NULL || csv->row_nr<1 || csv->col_nr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  
  if(verbose>0) printf("%s()\n", __func__);

  /* Check whether CSV file even can contain SIF data */
  int title_item_nr;
  title_item_nr=csvRowLength(csv, 0);
  if(csv->row_nr<2 || title_item_nr<5 ) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  int ret;

  /* Allocate memory for TAC data */
  ret=tacAllocate(tac, csv->row_nr-1, 3);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return ret;
  /* Set initial SIF data size */
  tac->tacNr=2; 
  tac->sampleNr=csv->row_nr-1;
  /* Set basic header information */
  tac->format=TAC_FORMAT_SIF;
  tac->isframe=1; // SIF always contains frame start and end times
  tac->tunit=UNIT_SEC;
  tac->cunit=UNIT_COUNTS;
  
  /* Set SIF compatible titles into TAC struct */
  strcpy(tac->c[0].name, "Prompts");
  strcpy(tac->c[1].name, "Randoms");
  strcpy(tac->c[2].name, "Trues"); // computed, not saved
  
  /* Read SIF title line */
  /* scan start time */
  char sif_scan_start_time[20];
  ret=0;
  if(strlen(csv->c[0].content)<11 && strlen(csv->c[1].content)<9) {
    char temp[20];
    strcpy(temp, csv->c[0].content); strcat(temp, " ");
    strcat(temp, csv->c[1].content);
    if(verbose>4) printf("  datetime := '%s'\n", temp);
    ret=strDateTimeValid(temp, sif_scan_start_time);
    if(ret!=0) {
      if(verbose>1) printf("  invalid SIF date/time (%d).\n", ret);
      else if(verbose>0) printf("  invalid SIF date/time.\n");
      if(ret<0) ret=0; // its common that SIF time is 1/1/1970 etc
    }
  } else ret=1;
  if(ret!=0) {
    if(verbose>2) {
      printf(" cell0 := '%s'\n", csv->c[0].content);
      printf(" cell1 := '%s'\n", csv->c[1].content);
    }
    tacFree(tac);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  if(verbose>1) printf("sif_scan_start_time := %s\n", sif_scan_start_time);
  /* Nr of frames */
  int frameNr;
  frameNr=atoi(csv->c[2].content);
  if(verbose>1) printf("sif_frame_nr := %d\n", frameNr);
  if(frameNr<1 || frameNr>tac->_sampleNr) {
    tacFree(tac);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  tac->sampleNr=frameNr;
  /* Nr of columns */
  int colNr;
  colNr=atoi(csv->c[3].content);
  if(verbose>1) printf("sif_col_nr := %d\n", colNr);
  if(colNr<2) {
    tacFree(tac);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  /* SIF version; forget about it */
  /* Study number; may be missing */
  char studynr[MAX_STUDYNR_LEN+1]; studynr[0]='\0';
  if(title_item_nr>5) {
    strlcpy(studynr, csv->c[5].content, MAX_STUDYNR_LEN+1);
    if(strcmp(studynr, ".")==0) strcpy(studynr, "");
    if(verbose>2) printf("sif_studynr := %s\n", studynr);
  }
  /* Isotope; may be missing */
  char isotope[MAX_ISOTOPE_LEN+1]; strcpy(isotope, "unknown");
  if(title_item_nr>6) {
    int isocode;
    isocode=isotopeIdentify(csv->c[6].content);
    strcpy(isotope, isotopeName(isocode));
  }
  if(verbose>1) printf("sif_isotope := %s\n", isotope);
  
  /* Read SIF frames */
  ret=0;
  int fi, ri, ci;
  double v;
  for(fi=0; fi<tac->sampleNr; fi++) {
    ri=fi+1;
    /* frame start time */
    ci=0; v=atofVerified(csvCell(csv, ri, ci)); 
    if(isnan(v)) {ret++; break;} else tac->x1[fi]=v;
    /* frame end time */
    ci=1; v=atofVerified(csvCell(csv, ri, ci)); 
    if(isnan(v)) {ret++; break;} else tac->x2[fi]=v;
    tac->x[fi]=0.5*(tac->x1[fi]+tac->x2[fi]);
    /* Prompts, randoms, and trues, maybe missing */
    tac->c[0].y[fi]=0.0;
    tac->c[1].y[fi]=0.0;
    tac->c[2].y[fi]=0.0;
    if(colNr<3) continue;
    /* so prompts should be there */
    ci=2; v=atofVerified(csvCell(csv, ri, ci)); 
    if(isnan(v)) {ret++; break;} else tac->c[0].y[fi]=v;
    if(colNr<4) continue;
    /* so randoms should be there */
    ci=3; v=atofVerified(csvCell(csv, ri, ci)); 
    if(isnan(v)) {ret++; break;} else tac->c[1].y[fi]=v;
    /* If trues not available, then trues=prompts-randoms */
    if(colNr<5) {
      tac->c[2].y[fi]=tac->c[0].y[fi]-tac->c[1].y[fi];
      continue;
    }
    /* so trues should be there */
    ci=4; v=atofVerified(csvCell(csv, ri, ci)); 
    if(isnan(v)) {ret++; break;} else tac->c[2].y[fi]=v;
  }
  if(ret!=0) {
    tacFree(tac);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Set TAC header */
  /* Isotope */
  if(isotopeIdentify(isotope)<=0 && hdr!=NULL) {
    /* SIF did not contain valid isotope code, get it from argument hdr */
    ret=tacGetHeaderIsotope(hdr, isotope, NULL);
    /* and delete it from hdr */
    if(ret==TPCERROR_OK) tacSetHeaderIsotope(hdr, NULL);
  }
  if(isotopeIdentify(isotope)>0) tacSetHeaderIsotope(&tac->h, isotope);
  /* Studynumber */
  if(!studynr[0] && hdr!=NULL) {
    /* SIF did not contain studynumber, get it from argument hdr */
    ret=tacGetHeaderStudynr(hdr, studynr, NULL);
    /* and delete it from hdr */
    if(ret==TPCERROR_OK) {
      if(verbose>2) printf("sifhdr_studynr := %s\n", studynr);
      tacSetHeaderStudynr(hdr, NULL);
    }
  }
  if(studynr[0]) {
    if(verbose>3) printf("setting TAC studynr to '%s'\n", studynr);
    tacSetHeaderStudynr(&tac->h, studynr);
  }
  /* Scan start time */
  if(strDateTimeValid(sif_scan_start_time, NULL)<0 && hdr!=NULL) {
    /* SIF did not contain valid scan start time, get it from argument hdr */
    ret=tacGetHeaderScanstarttime(hdr, sif_scan_start_time, NULL);
    /* and delete it from hdr */
    if(ret==TPCERROR_OK) tacSetHeaderScanstarttime(hdr, NULL);
  }
  if(strDateTimeValid(sif_scan_start_time, NULL)==0)
    tacSetHeaderScanstarttime(&tac->h, sif_scan_start_time);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
