/** @file taccsv.c
 *  @brief I/O functions for CSV and TSV TAC file format.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file pointer in CSV TAC format.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacReadCSV, tacWrite
 */
int tacWriteCSV(
  /** Pointer to TAC struct, contents of which are to be written */
  TAC *tac,
  /** File pointer */
  FILE *fp,
  /** Write (1) or do not write (0) also extra header fields found in IFT;
      note that these are not supported by other than TPC SW. */
  int extra,
  /** File format code; accepted values are TAC_FORMAT_CSV_UK,
      TAC_FORMAT_CSV_INT, TAC_FORMAT_TSV_UK, TAC_FORMAT_TSV_INT, and
      TAC_FORMAT_UNKNOWN (0); this last one uses format inside TAC struct,
      or if that is invalid, then the default format TAC_FORMAT_CSV_UK */
  tacformat format,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(tac==NULL || tac->tacNr<1 || tac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  
  int ret;

  /* Create local copy of headers, which will be edited here and written
     into CSV file, to preserve the original headers with TAC struct */
  if(verbose>99) iftWrite(&tac->h, stdout, NULL);
  IFT localh; iftInit(&localh);
  ret=iftDuplicate(&tac->h, &localh);
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }

  CSV csv;
  csvInit(&csv);
  if(csvAllocate(&csv, (tac->tacNr+3)*(tac->sampleNr+1))) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    iftFree(&localh); return TPCERROR_FAIL;
  }

  /* Check and set the CSV format */
  int tointl=0; // Decimal point (0) or comma (1)
  if(format==TAC_FORMAT_UNKNOWN) format=tac->format;
  if(format==TAC_FORMAT_UNKNOWN) format=TAC_FORMAT_CSV_UK;
  if(format==TAC_FORMAT_CSV_UK) {
    tointl=0; csv.separator=',';
  } else if(format==TAC_FORMAT_CSV_INT) {
    tointl=1; csv.separator=';';
  } else if(format==TAC_FORMAT_TSV_UK) {
    tointl=0; csv.separator='\t';
  } else if(format==TAC_FORMAT_TSV_INT) {
    tointl=1; csv.separator='\t';
  } else {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    iftFree(&localh); return TPCERROR_INVALID_FORMAT;
  }

  /* Make sure that TAC names are available */
  tacEnsureNames(tac);

  /* Fill CSV struct */
  char tmp[1024];
  
  if(verbose>1) printf("  writing header\n");

  /* Add file format ID */
  ret=csvPutString(&csv, "TAC", 0);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    csvFree(&csv); iftFree(&localh); return ret;
  }

  /* Add study number */
  tacGetHeaderStudynr(&tac->h, tmp, NULL);
  ret=csvPutString(&csv, tmp, 0);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    csvFree(&csv); iftFree(&localh); return ret;
  }

  
  if(verbose>1) printf("  writing data table title\n");
  
  if(tac->isframe==0) {
    sprintf(tmp, "time[%s]", unitName(tac->tunit));
    ret=csvPutString(&csv, tmp, 1);
  } else {
    sprintf(tmp, "start[%s]", unitName(tac->tunit));
    ret=csvPutString(&csv, tmp, 1);
    sprintf(tmp, "end[%s]", unitName(tac->tunit));
    if(!ret) ret=csvPutString(&csv, tmp, 0);
  }
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    csvFree(&csv); iftFree(&localh); return ret;
  }

  for(int ri=0; ri<tac->tacNr; ri++) {
    /* add TAC names, first one with unit */
    if(ri==0) {
      sprintf(tmp, "%s[%s]", tac->c[ri].name, unitName(tac->cunit));
    } else {
      sprintf(tmp, "%s", tac->c[ri].name);
    }
    ret=csvPutString(&csv, tmp, 0); if(ret) break;
  }
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    csvFree(&csv); iftFree(&localh); return ret;
  }
  if(tacIsWeighted(tac)) {
    ret=csvPutString(&csv, "weight", 0);
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret);
      csvFree(&csv); iftFree(&localh); return ret;
    }
  }

  if(verbose>1) printf("writing data table\n");
  
  for(int fi=0; fi<tac->sampleNr; fi++) {
    /* Time(s) (x, or x1 and x2) */
    if(tac->isframe==0) {
      ret=csvPutDouble(&csv, tac->x[fi], 1, tointl);
    } else {
      ret=csvPutDouble(&csv, tac->x1[fi], 1, tointl);
      if(!ret) ret=csvPutDouble(&csv, tac->x2[fi], 0, tointl);
    }
    if(ret!=TPCERROR_OK) break;
    /* Concentrations (y values) */
    for(int ri=0; ri<tac->tacNr; ri++) {
      ret=csvPutDouble(&csv, tac->c[ri].y[fi], 0, tointl); if(ret) break;
    }
    /* Weight */
    if(tacIsWeighted(tac)) {
      ret=csvPutDouble(&csv, tac->w[fi], 0, tointl); if(ret) break;
    }
  } // next sample
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    csvFree(&csv); iftFree(&localh); return ret;
  }

  /* Add VOI volumes, if available */
  if(tacIsSize(tac)) {
    ret=csvPutString(&csv, "volume", 1);
    if(tac->isframe && !ret) ret=csvPutString(&csv, "", 0);
    for(int ri=0; ri<tac->tacNr && !ret; ri++) {
      ret=csvPutDouble(&csv, tac->c[ri].size, 0, tointl);
    }
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret);
      csvFree(&csv); iftFree(&localh); return ret;
    }
  }

  /* Write CSV */
  ret=csvWrite(&csv, 0, fp, status);
  csvFree(&csv);
  if(ret!=TPCERROR_OK) {iftFree(&localh); return ret;}
  
  /* Write extra header */
  // scan start time etc should be added to the first line I think
  if(extra && localh.keyNr>0) {
    int ret=iftWrite(&localh, fp, status);
    if(ret!=TPCERROR_OK) {iftFree(&localh); return ret;}
  }
  
  iftFree(&localh);
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read CSV TAC format from CSV struct into TAC struct.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacWriteCSV, tacRead, tacReadCarimasTxt
 */
int tacReadCSV(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to CSV from which data is read; if it contains only one column,
      then it is assumed to represent the first y column and x column is not filled. */
  CSV *csv,
  /** Pointer to possible header data, which, if available, if processed and copied to TAC too; 
      enter NULL if not available. */
  IFT *hdr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) printf("%s()\n", __func__);

  tacFree(tac);

  if(csv==NULL || csv->row_nr<2 || csv->col_nr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  

  /* Check from the first field that data indeed is CSV TAC data */
  if(strcasecmp(csv->c[0].content, "TAC")!=0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Get time type from the title line */
  int i, isframe=-1;
  for(i=2; i<csv->nr; i++) if(csv->c[i].row==1 && csv->c[i].col==0) break;
  if(strncasecmp(csv->c[i].content, "time", 4)==0) isframe=0;
  else if(strncasecmp(csv->c[i].content, "start", 5)==0 &&
          strncasecmp(csv->c[i+1].content, "end", 3)==0) isframe=1;
  if(isframe<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Allocate memory for TAC data */
  int n, ret;
  n=csv->col_nr-1; if(isframe==1) n--;
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  ret=tacAllocate(tac, csv->row_nr-2, n);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return ret;
  tac->tacNr=n; tac->sampleNr=csv->row_nr-2;
  tac->isframe=isframe;
  // initiate the format, may be changed later if necessary
  if(csv->separator=='\t') tac->format=TAC_FORMAT_TSV_UK;
  else if(csv->separator==',') tac->format=TAC_FORMAT_CSV_INT;
  else tac->format=TAC_FORMAT_CSV_UK;

  /* Copy header to TAC struct */
  iftDuplicate(hdr, &tac->h);

  /* Read study id from the second field */
  tacSetHeaderStudynr(&tac->h, csv->c[1].content);
  
  /* Get time unit from the first title field, inside parens */
  char *cptr;
  for(i=2; i<csv->nr; i++) if(csv->c[i].row==1 && csv->c[i].col==0) break;
  cptr=strchr(csv->c[i].content, '[');
  if(cptr==NULL) {
    tac->tunit=UNIT_UNKNOWN;
  } else {
    char *tmp=strdup(cptr+1);
    cptr=strrchr(tmp, ']'); if(cptr!=NULL) *cptr='\0';
    tac->tunit=unitIdentify(tmp);
    free(tmp);
  }

  /* Get concentration unit from 2nd or 3rd field, inside parens */
  i++; if(isframe) i++;
  cptr=strchr(csv->c[i].content, '[');
  if(cptr==NULL) {
    tac->cunit=UNIT_UNKNOWN;
  } else {
    char *tmp=strdup(cptr+1);
    cptr=strrchr(tmp, ']'); if(cptr!=NULL) *cptr='\0';
    tac->cunit=unitIdentify(tmp);
    free(tmp);
  }

  /* Read TAC names from the title line */
  n=0;
  for(; i<csv->nr; i++) {
    if(csv->c[i].row!=1 || n>=tac->tacNr) break;
    /* Copy TAC name, removing unit if necessary */
    if(strchr(csv->c[i].content, '[')) {
      char *tmp=strdup(csv->c[i].content);
      cptr=strchr(tmp, '['); *cptr='\0';
      strlcpy(tac->c[n].name, tmp, MAX_TACNAME_LEN+1);
      free(tmp);
    } else {
      strlcpy(tac->c[n].name, csv->c[i].content, MAX_TACNAME_LEN+1);
    }
    if(strcmp(tac->c[n].name, ".")==0) strcpy(tac->c[n].name, "");
    n++;
  }
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  tac->tacNr=n;
  if(verbose>2) printf("  tacNr=%d\n", tac->tacNr);

  /* Copy x and y data from CSV into TAC struct */
  int fi=0, ri=0, oknr=0, maxfi=0;
  ret=0;
  double v;
  for(i=2; i<csv->nr; i++) if(csv->c[i].row==2) break;
  for(; i<csv->nr; i++) {
    if(verbose>10)
      printf("i=%d\trow=%d\tcol=%d\t'%s'\n",
             i, csv->c[i].row, csv->c[i].col, csv->c[i].content);
    if(csv->c[i].col==0 && strcasecmp(csv->c[i].content, "volume")==0)
      break; // stop if VOI volume line is encountered
    fi=csv->c[i].row-2; if(fi<0 || fi>=tac->sampleNr) {ret++; continue;}
    if(fi>maxfi) maxfi=fi;
    if(csv->c[i].col<0) {ret++; continue;}
    ri=csv->c[i].col-1; if(tac->isframe) ri--; 
    if(ri>=tac->tacNr) {ret++; continue;}
    v=atofVerified(csv->c[i].content); if(!isnan(v)) oknr++;
    if(verbose>10) printf("   -> fi=%d\tri=%d\t=\t%g\n", fi, ri, v);
    if(ri<0) {
      if(tac->isframe) {
        if(ri==-2) tac->x1[fi]=v; else if(ri==-1) tac->x2[fi]=v;
      } else tac->x[fi]=v;
    } else tac->c[ri].y[fi]=v;
    /* check whether we have decimal comma or dot */
    if(tac->format==TAC_FORMAT_CSV_UK || tac->format==TAC_FORMAT_TSV_UK) {
      if(strHaveDecimalComma(csv->c[i].content)) {
        if(tac->format==TAC_FORMAT_TSV_UK) tac->format=TAC_FORMAT_TSV_INT;
        else tac->format=TAC_FORMAT_CSV_INT;
      } 
    }
  }
  tac->sampleNr=maxfi+1;
  if(verbose>2) printf("  sampleNr=%d\n", tac->sampleNr);
  if(verbose>0 && ret>0)
    printf("%d error(s) in reading CSV file format.\n", ret);
  if(oknr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  /* Set middle sample times when necessary */
  if(tac->isframe)
    for(fi=0; fi<tac->sampleNr; fi++)
      tac->x[fi]=0.5*(tac->x1[fi]+tac->x2[fi]);

  /* Get VOI volumes from the following line */
  if(i<csv->nr && strcasecmp(csv->c[i].content, "volume")==0) {
    if(tac->isframe) i++;
    for(ri=0; ri<tac->tacNr; ri++) {
      if(csv->c[i+ri].row!=csv->c[i].row) break;
      tac->c[ri].size=atofVerified(csv->c[i+ri].content);
    }
  }

  /* If units are not known, try to read units from header */
  if(tac->cunit==UNIT_UNKNOWN) tacGetHeaderUnit(tac, NULL);
  else tacSetHeaderUnit(&tac->h, UNIT_UNKNOWN);
  if(tac->tunit==UNIT_UNKNOWN) tacGetHeaderTimeunit(tac, NULL);
  else tacSetHeaderTimeunit(&tac->h, UNIT_UNKNOWN);

  /* Move column containing weights to its correct place in the struct */
  ret=tacWMove(tac, 1, status);
  if(ret!=TPCERROR_OK && ret!=TPCERROR_NO_WEIGHTS) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return(ret);
  }
  if(tac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read Inveon TAC format from CSV struct into TAC struct.

    Note that there are several Inveon TAC formats, and not all are supported.
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int tacReadInveonCSV(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to Inveon CSV from which data is read. */
  CSV *csv,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(verbose>0) printf("%s()\n", __func__);

  int ret;

  tacFree(tac);

  if(csv==NULL || csv->row_nr<2 || csv->col_nr<5) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  

  /* Check from the first row that data indeed is CSV TAC data */
  if(strstr(csv->c[0].content, " - Time")==NULL || 
     strstr(csv->c[2].content, "(upper bound)")==NULL || 
     strstr(csv->c[3].content, "(lower bound)")==NULL ||
     strstr(csv->c[4].content, "(standard deviation)")==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }


  /* Allocate memory for TAC data */
  ret=tacAllocate(tac, csv->row_nr-1, csv->col_nr/5);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return ret;
  tac->tacNr=csv->col_nr/5; tac->sampleNr=csv->row_nr-1;
  tac->format=TAC_FORMAT_INVEON;

  /* This format has only frame middle time */
  tac->isframe=0;

  /* Read TAC names and units */
  char *cptr, buf[1024];
  ret=0;
  for(int ri=0; ri<tac->tacNr; ri++) {
    int ci=5*ri;
    strlcpy(buf, csv->c[ci].content, 1024);
    cptr=strcasestr(buf, " - Time"); if(cptr==NULL) {ret++; break;}
    *cptr=(char)0; strlcpy(tac->c[ri].name, buf, MAX_TACNAME_LEN);
    if(ri==0) { // Time unit only from the first column
      cptr=strcasestr(csv->c[ci].content, "Time (");
      if(cptr!=NULL) {
        strlcpy(buf, cptr+6, 1024);
        cptr=strchr(buf, ')'); if(cptr!=NULL) *cptr=(char)0;
        tac->tunit=unitIdentify(buf);
      }
    }
    if(ri==0) { // Concentration unit only from the first TAC
      ci++;
      cptr=strcasestr(csv->c[ci].content, " - ");
      if(cptr!=NULL) {
        strlcpy(buf, cptr+3, 1024);
        cptr=strchr(buf, ','); if(cptr!=NULL) *cptr=(char)0;
        tac->cunit=unitIdentify(buf);
      }
    }
  }

  /* Read one line at a time */
  for(int si=0; si<tac->sampleNr; si++) {
    ret=atofCheck(csvCell(csv, 1+si, 0), &tac->x[si]); if(ret) break;
    for(int ri=0; ri<tac->tacNr; ri++) {
      ret=atofCheck(csvCell(csv, 1+si, 1+5*ri), &tac->c[ri].y[si]); if(ret) break;
    }
  }


  if(tac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  tac->format=TAC_FORMAT_PMOD;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
#if(0)
/** Reads Inveon type 2 data into DFT struct
\return Returns 0 when successful, otherwise an error code.
 */
int csv2dft_b(
  /** Pointer to CSV data to be converted */
  CSV *csv,
  /** Pointer to empty DFT struct which will be allocated and filled here */
  DFT *dft
) {
  int ri, fi, fip, ii, ret;
  char *cptr, *cptr2, tmp[256];
  double v1, v2;

  if(CSV_TEST>2) {printf("csv2dft_b()\n"); fflush(stdout);}
  if(csv==NULL || dft==NULL) return CSV_ERROR;
  //printf("row_nr=%d col_nr=%d\n", csv->row_nr, csv->col_nr);
  if(csv->row_nr<4 || csv->col_nr!=9) return CSV_INVALIDFORMAT;
  dftEmpty(dft);

  /* Check the format; first line (containing titles) */
  if(strcasecmp(csv->c[0].content, "#Subject ID")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[1].content, "Subject Weight")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[2].content, "Subject Sex")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[3].content, "Unique Series ID")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[4].content, "Series Date")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[5].content, "Series Description")!=0) return CSV_INVALIDFORMAT;

  /* Check the format; third line (containing titles) */
  if(strcasecmp(csv->c[12].content, "#Name")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[13].content, "Volume (mm^3)")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[14].content, "Mean")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[15].content, "SD")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[16].content, "Min")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[17].content, "Max")!=0) return CSV_INVALIDFORMAT;
  if(strcasecmp(csv->c[18].content, "Frame Index")!=0) return CSV_INVALIDFORMAT;
  if(strncasecmp(csv->c[19].content, "Mid time (sec)", 10)!=0) return CSV_INVALIDFORMAT;
  if(strncasecmp(csv->c[20].content, "Duration (sec)", 10)!=0) return CSV_INVALIDFORMAT;

  /* Calculate the number of ROIs and time frames */
  ri=1; fi=0; fip=-1; ii=21; cptr=csv->c[ii].content;
  //printf("cell[%d] := '%s'\n", ii, cptr); fflush(stdout);
  for(; ii<csv->nr; ii+=9) {
    cptr2=csv->c[ii].content;
    //printf("cell[%d] := '%s'\n", ii, cptr2); fflush(stdout);
    if(strcmp(cptr, cptr2)==0) fi++;
    else {
      ri++; cptr=cptr2;
      if(fip<0) fip=fi; else if(fi!=fip) return CSV_INVALIDFORMAT;
      fi=1;
    }
    //printf("ri=%d fi=%d fip=%d\n", ri, fi, fip);
  }
  //printf("ri=%d fi=%d\n", ri, fi);

  /* Allocate memory for DFT */
  if(CSV_TEST>2) {printf("frame_nr=%d voi_nr=%d\n", fi, ri); fflush(stdout);}
  ret=dftSetmem(dft, fi, ri);
  if(ret!=0) return CSV_OUTOFMEMORY;
  dft->voiNr=ri; dft->frameNr=fi;
  dft->timetype=DFT_TIME_STARTEND;
  dft->_type=DFT_FORMAT_STANDARD;
  dft->isweight=0;
  dftUnitToDFT(dft, CUNIT_UNKNOWN); dft->timeunit=TUNIT_UNKNOWN;
  for(fi=0; fi<dft->frameNr; fi++) dft->w[fi]=1.0;

  /* Fill DFT */
  /* time unit */
  ii=19;
  if(strstr(csv->c[ii].content, "min")!=NULL) dft->timeunit=TUNIT_MIN;
  else if(strstr(csv->c[ii].content, "sec")!=NULL) dft->timeunit=TUNIT_SEC;
  else dft->timeunit=TUNIT_UNKNOWN;
  /* study number */
  ii=6;
  cptr=csv->c[ii].content; strlcpy(dft->studynr, cptr, 11);
  cptr=strchr(dft->studynr, '.'); if(cptr!=NULL) *cptr=(char)0;
  cptr=strchr(dft->studynr, ','); if(cptr!=NULL) *cptr=(char)0;
  cptr=strchr(dft->studynr, ' '); if(cptr!=NULL) *cptr=(char)0;
  /* subject weight */
  ii=7; if(ii>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
  v1=atof_dpi(csv->c[ii].content);
  if(v1>0.0) sprintf(dft->comments, "# weight := %g\n", v1);
  /* scan start time */
  ii=10; if(ii>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
  if(strlen(csv->c[ii].content)>9) {
    sprintf(tmp, "# scan_start_time := %s\n", csv->c[ii].content);
    strcat(dft->comments, tmp);
  }
  /* frame times */
  for(fi=0; fi<dft->frameNr; fi++) {
    ii= 21 + fi*9 + 7; if(ii>csv->nr-2) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
    //printf("cell[%d] := '%s'\n", ii, csv->c[ii].content); fflush(stdout);
    v1=atof_dpi(csv->c[ii].content); v2=atof_dpi(csv->c[ii+1].content);
    dft->x[fi]=v1; dft->x1[fi]=v1-0.5*v2; dft->x2[fi]=v1+0.5*v2;
  }
  /* region names, volumes, and concentrations */
  for(ri=0; ri<dft->voiNr; ri++) {
    /* ROI name */
    ii= 21 + ri*dft->frameNr*9;
    if(ii>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
    //printf("ri=%d cell[%d] := '%s'\n", ri, ii, csv->c[ii].content); fflush(stdout);
    strlcpy(dft->voi[ri].name, csv->c[ii].content, MAX_REGIONNAME_LEN+1);
    rnameSplit(csv->c[ii].content, dft->voi[ri].voiname,
  	       dft->voi[ri].hemisphere, dft->voi[ri].place,
               MAX_REGIONSUBNAME_LEN);
    /* Volume */
    ii++; if(ii>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
    dft->voi[ri].size=atof_dpi(csv->c[ii].content);
    /* Frame concentrations */
    ii++; for(fi=0; fi<dft->frameNr; fi++) {
      if((ii+6)>=csv->nr) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
      /* Get concentration */
      dft->voi[ri].y[fi]=atof_dpi(csv->c[ii].content);
      /* Get concentration SD (probably not needed) */
      dft->voi[ri].y2[fi]=atof_dpi(csv->c[ii+1].content);
      /* check that frame times are correct */
      v1=atof_dpi(csv->c[ii+5].content);
      if(dft->x[fi]!=v1) {dftEmpty(dft); return CSV_INVALIDFORMAT;}
      ii+=9;
    } // next frame
  } // next region

  return CSV_OK;
}
#endif
/*****************************************************************************/

/*****************************************************************************/
/** Read Carimas TXT TAC format from CSV struct into TAC struct.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacReadCSV, tacRead
 */
int tacReadCarimasTxt(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to CSV from which data is read; if it contains only one column,
      then it is assumed to represent the first y column and x column is not filled. */
  CSV *csv,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(tac==NULL) {
    if(verbose>0) printf("tac := NULL\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }

  tacFree(tac);

  if(verbose>2) {
    printf("csv.row_nr := %d\n", csv->row_nr);
    printf("csv.col_nr := %d\n", csv->col_nr);
  }
  if(csv==NULL || csv->row_nr<6 || csv->col_nr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  

  /* Check that data indeed is CSV TAC data */
  if(strcasecmp(csvCell(csv, 1, 0), "Num Voxels")!=0 ||
     strncasecmp(csvCell(csv, 2, 0), "Volume (mm^3)", 6)!=0 ||
     strncasecmp(csvCell(csv, 3, 0), "Times(s)", 5)!=0 )
  {
    if(verbose>0) {
      printf("cell[%d][%d] := '%s'\n", 1, 0, csvCell(csv, 1, 0));
      printf("cell[%d][%d] := '%s'\n", 2, 0, csvCell(csv, 2, 0));
      printf("cell[%d][%d] := '%s'\n", 3, 0, csvCell(csv, 3, 0));
    }
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  char *cptr;
  int ret, fi, ri, oknr;

  /* Allocate memory for TAC data */
  ret=tacAllocate(tac, csv->row_nr-4, csv->col_nr-1);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return ret;
  tac->tacNr=csv->col_nr-1; tac->sampleNr=csv->row_nr-4;
  tac->isframe=1;
  // initiate the format, may be changed later if necessary
  tac->format=TAC_FORMAT_CARIMAS_TXT;

  /* Get time unit from the 4th nonempty title line, inside parens */
  tac->tunit=UNIT_UNKNOWN;
  cptr=csvCell(csv, 3, 0);
  if(cptr!=NULL && strncasecmp(cptr, "Times(s)", 5)==0) {
    char *tmp=strdup(cptr+5);
    strCleanPars(tmp); tac->tunit=unitIdentify(tmp);
    free(tmp);
  }

  /* Get concentration unit the first title line, inside parens */
  tac->cunit=UNIT_UNKNOWN;
  cptr=csvCell(csv, 0, 1);
  if(cptr!=NULL && strchr(cptr, '(')!=0) {
    char *tmp=strdup(strchr(cptr, '('));
    strCleanPars(tmp); tac->cunit=unitIdentify(tmp);
    free(tmp);
  }

  /* Read TAC names from the first title line */
  if(verbose>4) printf("reading VOI names\n");
  ret=oknr=0;
  for(ri=0; ri<tac->tacNr; ri++) {
    cptr=csvCell(csv, 0, ri+1); if(!cptr) {ret++; break;}
    if(strlen(cptr)==0) break;
    strlcpy(tac->c[ri].name, cptr, MAX_TACNAME_LEN+1);
    cptr=strchr(tac->c[ri].name, '('); if(cptr) *cptr='\0';
    if(strcmp(tac->c[ri].name, ".")==0) strcpy(tac->c[ri].name, "");
    oknr++;
  }
  if(ret || oknr<1) {
    if(verbose>0) printf("could not read TAC names.\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  /* Set tacNr, in case that txt file had empty fields */
  tac->tacNr=oknr;
  if(verbose>2) printf("  tacNr=%d\n", tac->tacNr);

  /* Get VOI volumes from the 3rd line */
  if(verbose>4) printf("reading VOI volumes\n");
  ret=oknr=0;
  for(ri=0; ri<tac->tacNr; ri++) {
    cptr=csvCell(csv, 2, ri+1); if(!cptr) {ret++; break;}
    if(atofCheck(cptr, &tac->c[ri].size)!=0) tac->c[ri].size=nan("");
    oknr++;
  }
  if(ret || oknr<1) {
    if(verbose>0) printf("could not read TAC volumes.\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Copy x and y data from CSV into TAC struct */
  if(verbose>4) printf("reading x and y values\n");
  ret=oknr=0;
  for(fi=0; fi<tac->sampleNr; fi++) {
    /* Frame times */
    cptr=csvCell(csv, fi+4, 0); if(!cptr) {ret++; break;}
    if(sscanf(cptr, "%lf - %lf", &tac->x1[fi], &tac->x2[fi])!=2) {ret++; break;}
    /* Concentrations */
    for(ri=0; ri<tac->tacNr; ri++) {
      cptr=csvCell(csv, fi+4, ri+1); if(!cptr) {ret++; break;}
      ret=atofCheck(cptr, &tac->c[ri].y[fi]); if(ret) break;
    }
    oknr++;
  }
  if(ret || oknr<1) {
    if(verbose>0) printf("could not read data.\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  if(verbose>2) printf("  sampleNr=%d\n", tac->sampleNr);

  /* Set middle sample times */
  if(tac->isframe)
    for(fi=0; fi<tac->sampleNr; fi++)
      tac->x[fi]=0.5*(tac->x1[fi]+tac->x2[fi]);

  /* Move column containing weights to its correct place in the struct */
  ret=tacWMove(tac, 1, status);
  if(ret!=TPCERROR_OK && ret!=TPCERROR_NO_WEIGHTS) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return(ret);
  }
  if(tac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
