/** @file tacxmlio.c
 *  @brief I/O functions for TAC XML format.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write TAC data into specified file pointer in Excel compatible XML format.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacWrite
 */
int tacWriteXML(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** File pointer */
  FILE *fp,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(tac==NULL || tac->tacNr<1 || tac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  char *cptr, tunit[128], cunit[128];
  int n;

  /* Set units to format accepted by PMOD */
  if(tac->tunit==UNIT_SEC) strcpy(tunit, "seconds");
  else if(tac->tunit==UNIT_MIN) strcpy(tunit, "minutes");
  else strcpy(tunit, unitName(tac->tunit));
  strcpy(cunit, unitName(tac->cunit));
  if(strcasestr(cunit, "dL")==NULL) {
    /* Replace mL by cc, but only if unit does not contain 'dL' */
    cptr=strcasestr(cunit, "mL"); 
    if(cptr!=NULL) {*cptr='c'; cptr++; *cptr='c';}
  }

  /* Make sure that TAC names are available */
  if(verbose>2) printf("constructing TAC names\n");
  tacEnsureNames(tac);

  /* Write XML header */
  n=fprintf(fp, "<?xml version=\"1.0\"?>\n");
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  fprintf(fp, "<ss:Workbook xmlns:ss=\"urn:schemas-microsoft-com:office:spreadsheet\">\n");
  fprintf(fp, "  <ss:Styles>\n");
  fprintf(fp, "    <ss:Style ss:ID=\"1\">\n");
  fprintf(fp, "      <ss:Font ss:Bold=\"1\"/>\n");
  fprintf(fp, "    </ss:Style>\n");
  fprintf(fp, "  </ss:Styles>\n");
  fprintf(fp, "  <ss:Worksheet ss:Name=\"Sheet1\">\n");
  fprintf(fp, "    <ss:Table>\n");

  /* Set column widths */
  n=tac->tacNr+1;
  if(tac->isframe) n++;
  if(tacIsWeighted(tac)) n++;
  for(int i=0; i<n; i++) fprintf(fp, "      <ss:Column ss:Width=\"80\"/>\n");

  /* Write the title line */
  fprintf(fp, "      <ss:Row ss:StyleID=\"1\">\n");
  if(tac->isframe==0) {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">time[%s]</ss:Data>\n", tunit);
    fprintf(fp, "        </ss:Cell>\n");
  } else {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">start[%s]</ss:Data>\n", tunit);
    fprintf(fp, "        </ss:Cell>\n");
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">end[%s]</ss:Data>\n", cunit);
    fprintf(fp, "        </ss:Cell>\n");
  }
  for(int ri=0; ri<tac->tacNr; ri++) {
    /* write TAC names */
    fprintf(fp, "        <ss:Cell>\n");
    if(ri==0 && tac->isframe==0)
      fprintf(fp, "          <ss:Data ss:Type=\"String\">%s[%s]</ss:Data>\n", tac->c[ri].name, cunit);
    else
      fprintf(fp, "          <ss:Data ss:Type=\"String\">%s</ss:Data>\n", tac->c[ri].name);
    fprintf(fp, "        </ss:Cell>\n");
  }
  if(tacIsWeighted(tac)) {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">weight</ss:Data>\n");
    fprintf(fp, "        </ss:Cell>\n");
  }
  n=fprintf(fp, "      </ss:Row>\n");
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Write data */
  if(verbose>2) printf("writing data table\n");
  for(int fi=0; fi<tac->sampleNr; fi++) {

    fprintf(fp, "      <ss:Row>\n");

    /* Note: missing values are written as empty cells, with string type, because with
       number type Excel seems to read those as zeroes. */

    /* Time(s) (x, or x1 and x2) */
    double v; if(tac->isframe==0) v=tac->x[fi]; else v=tac->x1[fi];
    fprintf(fp, "        <ss:Cell>\n");
    if(isnan(v)) fprintf(fp, "          <ss:Data ss:Type=\"String\"></ss:Data>\n");
    else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", v);
    fprintf(fp, "        </ss:Cell>\n");

    if(tac->isframe) {
      v=tac->x2[fi];
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(v)) fprintf(fp, "          <ss:Data ss:Type=\"String\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", v);
      fprintf(fp, "        </ss:Cell>\n");
    }

    /* Concentrations (y values) */
    for(int ri=0; ri<tac->tacNr; ri++) {
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(tac->c[ri].y[fi])) fprintf(fp, "          <ss:Data ss:Type=\"String\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", tac->c[ri].y[fi]);
      fprintf(fp, "        </ss:Cell>\n");
    }

    /* Weight */
    if(tacIsWeighted(tac)) {
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(tac->w[fi])) fprintf(fp, "          <ss:Data ss:Type=\"String\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", tac->w[fi]);
      fprintf(fp, "        </ss:Cell>\n");
    }

    fprintf(fp, "      </ss:Row>\n");
  }

  /* Write XML end part */
  fprintf(fp, "    </ss:Table>\n");
  fprintf(fp, "  </ss:Worksheet>\n");
  n=fprintf(fp, "</ss:Workbook>\n");
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
