/** @file tacy.c
 *  @brief Working with y values (concentrations) in TAC struct.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Get the range of y values (concentrations) in TAC struct.
    @details Data is not modified. Data does not need to be sorted.
    Data can contain NaNs, but y values omitted if corresponding x is NaN. 
    @sa tacYRangeInXRange, tacXRange, tacYUnitConvert, tacCopyTacc
    @author Vesa Oikonen
    @return Returns <>0 in case of failure.
 */
int tacYRange(
  /** Pointer to TAC structure. */
  TAC *d,
  /** Index of TAC, [0, ..., tacNr-1] ; enter <0 to search for max and/or min from all TACs. */
  int i, 
  /** Pointer to variable for min y value (NULL if not needed). */
  double *ymin,
  /** Pointer to variable for max y value (NULL if not needed). */
  double *ymax,
  /** Pointer to variable where sample index of min y value will be stored (NULL if not needed). */
  int *smin,
  /** Pointer to variable where sample index of max y value will be stored (NULL if not needed). */
  int *smax,
  /** Pointer to variable where tac index of min y value will be stored (NULL if not needed). */
  int *imin,
  /** Pointer to variable where tac index of max y value will be stored (NULL if not needed). */
  int *imax
) {
  if(ymin!=NULL) *ymin=nan("");
  if(ymax!=NULL) *ymax=nan("");
  if(smin!=NULL) *smin=-1;
  if(smax!=NULL) *smax=-1;
  if(imin!=NULL) *imin=-1;
  if(imax!=NULL) *imax=-1;
  /* Check the data */
  if(d==NULL || d->sampleNr<1) return(1);
  /* Check the index */
  if(i>=d->tacNr) return(2);

  /* Find the min and max time */
  double *xi, *xa;
  if(d->isframe) {xi=d->x1; xa=d->x2;} else xi=xa=d->x;
  double mi, ma;
  mi=ma=nan("");
  for(int j=0; j<d->sampleNr; j++) {
    if(isnan(xi[j]) || isnan(xa[j])) continue;
    for(int k=0; k<d->tacNr; k++) {
      if(i>=0 && i!=k) continue;
      if(isnan(mi) || isnan(ma)) {
        mi=d->c[k].y[j]; ma=d->c[k].y[j]; 
        if(smin!=NULL) *smin=j;
        if(smax!=NULL) *smax=j;
        if(imin!=NULL) *imin=k;
        if(imax!=NULL) *imax=k;
        continue;
      }
      if(d->c[k].y[j]<mi) {
        mi=d->c[k].y[j]; 
        if(smin!=NULL) *smin=j;
        if(imin!=NULL) *imin=k;
      } else if(d->c[k].y[j]>ma) {
        ma=d->c[k].y[j];
        if(smax!=NULL) *smax=j;
        if(imax!=NULL) *imax=k;
      }
    }
  }
  if(ymin!=NULL) *ymin=mi;
  if(ymax!=NULL) *ymax=ma;
  if(isnan(mi) || isnan(ma)) return(2);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Get the range of y values (concentrations) in TAC struct, including
    only samples with x (times) inside given range.
    @details Data is not modified. Data does not need to be sorted.
    Data can contain NaNs, but y values omitted if corresponding x is NaN. 
    @sa tacYRange, tacXRange, tacYUnitConvert
    @author Vesa Oikonen
    @return Returns <>0 in case of failure.
 */
int tacYRangeInXRange(
  /** Pointer to TAC structure. */
  TAC *d,
  /** Index of TAC, [0, ..., tacNr-1] ; enter <0 to search for max and/or min from all TACs. */
  int i, 
  /** Min x value. */
  const double xmin,
  /** Max x value. */
  const double xmax,
  /** Pointer to variable for min y value (NULL if not needed). */
  double *ymin,
  /** Pointer to variable for max y value (NULL if not needed). */
  double *ymax,
  /** Pointer to variable where sample index of min y value will be stored (NULL if not needed). */
  int *smin,
  /** Pointer to variable where sample index of max y value will be stored (NULL if not needed). */
  int *smax,
  /** Pointer to variable where tac index of min y value will be stored (NULL if not needed). */
  int *imin,
  /** Pointer to variable where tac index of max y value will be stored (NULL if not needed). */
  int *imax
) {
  if(ymin!=NULL) *ymin=nan("");
  if(ymax!=NULL) *ymax=nan("");
  if(smin!=NULL) *smin=-1;
  if(smax!=NULL) *smax=-1;
  if(imin!=NULL) *imin=-1;
  if(imax!=NULL) *imax=-1;
  /* Check the data */
  if(d==NULL || d->sampleNr<1) return(1);
  /* Check the index */
  if(i>=d->tacNr) return(2);
  /* Check x range */
  if(xmin>xmax) return(3);

  /* Find the min and max time */
  double *xi, *xa, x;
  if(d->isframe) {xi=d->x1; xa=d->x2;} else xi=xa=d->x;
  double mi, ma;
  mi=ma=nan("");
  for(int j=0; j<d->sampleNr; j++) {
    if(isnan(xi[j]) || isnan(xa[j])) continue;
    x=0.5*(xi[j]+xa[j]); if(x<xmin || x>xmax) continue;
    for(int k=0; k<d->tacNr; k++) {
      if(i>=0 && i!=k) continue;
      if(isnan(mi) || isnan(ma)) {
        mi=d->c[k].y[j]; ma=d->c[k].y[j]; 
        if(smin!=NULL) *smin=j;
        if(smax!=NULL) *smax=j;
        if(imin!=NULL) *imin=k;
        if(imax!=NULL) *imax=k;
        continue;
      }
      if(d->c[k].y[j]<mi) {
        mi=d->c[k].y[j]; 
        if(smin!=NULL) *smin=j;
        if(imin!=NULL) *imin=k;
      } else if(d->c[k].y[j]>ma) {
        ma=d->c[k].y[j];
        if(smax!=NULL) *smax=j;
        if(imax!=NULL) *imax=k;
      }
    }
  }
  if(ymin!=NULL) *ymin=mi;
  if(ymax!=NULL) *ymax=ma;
  if(isnan(mi) || isnan(ma)) return(2);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
