/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
int test_tacXCopy(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int ret;
  TAC tac1, tac2; 
  tacInit(&tac1); tacInit(&tac2);

  ret=tacXCopy(&tac1, &tac2, 0, 0);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 1;}
  
  if(verbose>1) printf("\ncreating test data\n");
  ret=create_tac(&tac1); if(ret==TPCERROR_OK) ret=create_tac(&tac2);  
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  if(verbose>1) printf("\nchecking indices\n");
  ret=tacXCopy(&tac1, &tac2, 2, 0);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 11;}
  ret=tacXCopy(&tac1, &tac2, -1, 1);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 12;}
  ret=tacXCopy(&tac1, &tac2, 0, tac1._sampleNr);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 13;}
  
  if(verbose>1) printf("\nchecking copying\n");
  for(int i=0; i<tac2.sampleNr; i++) tac2.x[i]=tac2.x1[i]=tac2.x2[i]=nan("");
  ret=tacXCopy(&tac1, &tac2, 0, tac1.sampleNr-1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 21;}
  ret=tacCompareTimes(&tac1, &tac2, 1.0E-08, -1.0, NULL);
  if(ret!=0) {tacFree(&tac1); tacFree(&tac2); return 22;}
  
  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacCorrectFrameOverlap(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("not tested yet.\n");

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacXRange(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  
  int ret;
  TAC tac; tacInit(&tac);
  double mi, ma;

  if(verbose>1) printf("\ntrying with NULL TAC.\n");
  ret=tacXRange(NULL, NULL, NULL);
  if(ret==TPCERROR_OK) return 1;
  ret=tacXRange(NULL, &mi, &ma);
  if(ret==TPCERROR_OK) return 2;
  if(!isnan(mi) || !isnan(ma)) return 3;

  if(verbose>1) printf("\ntrying with empty TAC.\n");
  ret=tacXRange(&tac, NULL, NULL);
  if(ret==TPCERROR_OK) return 5;
  ret=tacXRange(&tac, &mi, &ma);
  if(ret==TPCERROR_OK) return 6;
  if(!isnan(mi) || !isnan(ma)) return 7;
  
  if(verbose>1) printf("\ncreating test data\n");
  ret=create_tac(&tac);  
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("\ntrying with valid and NULL output pointers.\n");
  ret=tacXRange(&tac, NULL, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 11;}
  ret=tacXRange(&tac, &mi, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 12;}
  ret=tacXRange(&tac, NULL, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 13;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 0.0, 1.0E-04)) {tacFree(&tac); return 14;}
  if(!doubleMatch(ma, 5.0, 1.0E-04)) {tacFree(&tac); return 15;}

  if(verbose>1) printf("\ntrying with frame middle times.\n");
  tac.isframe=0;
  ret=tacXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 21;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 0.5, 1.0E-04)) {tacFree(&tac); return 22;}
  if(!doubleMatch(ma, 4.5, 1.0E-04)) {tacFree(&tac); return 23;}

  if(verbose>1) printf("\ntrying with some NaNs inside TAC.\n");

  tac.isframe=1;
  tac.x1[0]=tac.x2[0]=tac.x[0]=nan("");
  tac.x1[tac.sampleNr-1]=tac.x2[tac.sampleNr-1]=tac.x[tac.sampleNr-1]=nan("");
  ret=tacXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 31;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-04)) {tacFree(&tac); return 32;}
  if(!doubleMatch(ma, 4.0, 1.0E-04)) {tacFree(&tac); return 33;}

  tac.isframe=0;
  ret=tacXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 35;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.5, 1.0E-04)) {tacFree(&tac); return 36;}
  if(!doubleMatch(ma, 3.5, 1.0E-04)) {tacFree(&tac); return 37;}
  tac.isframe=1;

  if(verbose>1) printf("\nall x values are NaNs inside TAC.\n");
  for(int i=0; i<tac.sampleNr; i++) tac.x1[i]=tac.x2[i]=tac.x[i]=nan("");
  ret=tacXRange(&tac, &mi, &ma); 
  if(ret==TPCERROR_OK) {tacFree(&tac); return 41;}

  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacSampleXRange(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  
  int ret;
  TAC tac; tacInit(&tac);
  double mi, ma;

  if(verbose>1) printf("\ntrying with NULL TAC.\n");
  ret=tacSampleXRange(NULL, NULL, NULL);
  if(ret==TPCERROR_OK) return 1;
  ret=tacSampleXRange(NULL, &mi, &ma);
  if(ret==TPCERROR_OK) return 2;
  if(!isnan(mi) || !isnan(ma)) return 3;

  if(verbose>1) printf("\ntrying with empty TAC.\n");
  ret=tacSampleXRange(&tac, NULL, NULL);
  if(ret==TPCERROR_OK) return 5;
  ret=tacSampleXRange(&tac, &mi, &ma);
  if(ret==TPCERROR_OK) return 6;
  if(!isnan(mi) || !isnan(ma)) return 7;
  
  if(verbose>1) printf("\ncreating test data (with no weighting)\n");
  ret=create_tac(&tac);  
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}
  ret=tacSetWeights(&tac, WEIGHTING_OFF, tac.sampleNr, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("\ntrying with valid and NULL output pointers.\n");
  ret=tacSampleXRange(&tac, NULL, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 11;}
  ret=tacSampleXRange(&tac, &mi, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 12;}
  ret=tacSampleXRange(&tac, NULL, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 13;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 0.0, 1.0E-04)) {tacFree(&tac); return 14;}
  if(!doubleMatch(ma, 5.0, 1.0E-04)) {tacFree(&tac); return 15;}

  if(verbose>1) printf("\ntrying with frame middle times.\n");
  tac.isframe=0;
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 21;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 0.5, 1.0E-04)) {tacFree(&tac); return 22;}
  if(!doubleMatch(ma, 4.5, 1.0E-04)) {tacFree(&tac); return 23;}

  if(verbose>1) printf("\ntrying with some NaNs inside TAC.\n");

  tac.isframe=1;
  tac.x1[0]=tac.x2[0]=tac.x[0]=nan("");
  tac.x1[tac.sampleNr-1]=tac.x2[tac.sampleNr-1]=tac.x[tac.sampleNr-1]=nan("");
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 31;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-04)) {tacFree(&tac); return 32;}
  if(!doubleMatch(ma, 4.0, 1.0E-04)) {tacFree(&tac); return 33;}

  tac.isframe=0;
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 35;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.5, 1.0E-04)) {tacFree(&tac); return 36;}
  if(!doubleMatch(ma, 3.5, 1.0E-04)) {tacFree(&tac); return 37;}
  tac.isframe=1;

  if(verbose>1) printf("\nall x values are NaNs inside TAC.\n");
  for(int i=0; i<tac.sampleNr; i++) tac.x1[i]=tac.x2[i]=tac.x[i]=nan("");
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret==TPCERROR_OK) {tacFree(&tac); return 41;}

  if(verbose>1) printf("\nSome y values are NaNs inside TAC.\n");
  tacFree(&tac); create_tac(&tac);
  tac.c[0].y[0]=nan("");
  tac.c[1].y[tac.sampleNr-1]=nan("");
  tac.isframe=1;
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 51;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);

  if(!doubleMatch(mi, 0.0, 1.0E-04)) {tacFree(&tac); return 52;}
  if(!doubleMatch(ma, 5.0, 1.0E-04)) {tacFree(&tac); return 53;}

  tac.isframe=0;
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 55;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);

  if(!doubleMatch(mi, 0.5, 1.0E-04)) {tacFree(&tac); return 56;}
  if(!doubleMatch(ma, 4.5, 1.0E-04)) {tacFree(&tac); return 57;}


  if(verbose>1) printf("\nAll y values of certain frames are NaNs.\n");
  tacFree(&tac); create_tac(&tac);
  for(int i=0; i<tac.tacNr; i++) {
    tac.c[i].y[0]=nan("");
    tac.c[i].y[tac.sampleNr-1]=nan("");
  }
  tac.isframe=1;
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 51;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-04)) {tacFree(&tac); return 61;}
  if(!doubleMatch(ma, 4.0, 1.0E-04)) {tacFree(&tac); return 62;}

  tac.isframe=0;
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 55;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.5, 1.0E-04)) {tacFree(&tac); return 63;}
  if(!doubleMatch(ma, 3.5, 1.0E-04)) {tacFree(&tac); return 64;}




  if(verbose>1) printf("\n\n re-creating test data with weights \n");
  tacFree(&tac);
  ret=create_tac(&tac);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 100;}
  ret=tacSetWeights(&tac, WEIGHTING_ON_F, tac.sampleNr, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 100;}

  if(verbose>1) printf("\n trying with all weights on.\n");

  ret=tacSampleXRange(&tac, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 101;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 0.0, 1.0E-04)) {tacFree(&tac); return 102;}
  if(!doubleMatch(ma, 5.0, 1.0E-04)) {tacFree(&tac); return 103;}

  if(verbose>1) printf("\ntrying with some zero weights inside TAC.\n");

  tac.isframe=1;
  tac.w[0]=tac.w[tac.sampleNr-1]=0.0;
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 111;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-04)) {tacFree(&tac); return 112;}
  if(!doubleMatch(ma, 4.0, 1.0E-04)) {tacFree(&tac); return 113;}

  tac.isframe=0;
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 115;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.5, 1.0E-04)) {tacFree(&tac); return 116;}
  if(!doubleMatch(ma, 3.5, 1.0E-04)) {tacFree(&tac); return 117;}
  tac.isframe=1;

  if(verbose>1) printf("\n all weights are 0 inside TAC.\n");
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=0.0;
  ret=tacSampleXRange(&tac, &mi, &ma); 
  if(ret==TPCERROR_OK) {tacFree(&tac); return 121;}
  if(verbose>2) printf("  results in error as it should.\n");

  if(verbose>1) printf("\n ... but all is fine when weighting turned off.\n");
  tac.weighting=WEIGHTING_OFF;
  ret=tacSampleXRange(&tac, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 131;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 0.0, 1.0E-04)) {tacFree(&tac); return 132;}
  if(!doubleMatch(ma, 5.0, 1.0E-04)) {tacFree(&tac); return 133;}


  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacMinX(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("\ntrying with NULL TAC.\n");
  if(tacMinX(NULL)!=-1) return 1;
  
  TAC tac; tacInit(&tac);

  if(verbose>1) printf("\ntrying with empty TAC.\n");
  if(tacMinX(&tac)!=-1) return 2;

  if(verbose>1) printf("\ncreating test data\n");
  if(create_tac(&tac)!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("\ntrying with valid TAC data.\n");
  if(tacMinX(&tac)!=0) {tacFree(&tac); return 21;}
  if(verbose>1) printf("\ntrying with frame middle times.\n");
  tac.isframe=0;
  if(tacMinX(&tac)!=0) {tacFree(&tac); return 22;}

  if(verbose>1) printf("\ntrying with missing first sample time.\n");
  tac.x1[0]=tac.x2[0]=tac.x[0]=nan("");
  tac.isframe=1;
  if(tacMinX(&tac)!=1) {tacFree(&tac); return 31;}
  tac.isframe=0;
  if(tacMinX(&tac)!=1) {tacFree(&tac); return 32;}

  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacIsX(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  
  TAC tac; tacInit(&tac);

  if(verbose>1) printf("\ntrying with NULL TAC.\n");
  if(tacIsX(NULL)!=0) return 1;

  if(verbose>1) printf("\ntrying with empty TAC.\n");
  if(tacIsX(&tac)!=0) return 2;
  
  if(verbose>1) printf("\ncreating test data\n");
  if(create_tac(&tac)!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("\ntrying with valid TAC data.\n");
  if(tacIsX(&tac)!=1) {tacFree(&tac); return 21;}
  if(verbose>1) printf("\ntrying with frame middle times.\n");
  tac.isframe=0;
  if(tacIsX(&tac)!=1) {tacFree(&tac); return 22;}

  if(verbose>1) printf("\ntrying with some NaNs inside TAC.\n");
  tac.x1[0]=tac.x2[0]=tac.x[0]=nan("");
  tac.x1[tac.sampleNr-1]=tac.x2[tac.sampleNr-1]=tac.x[tac.sampleNr-1]=nan("");
  tac.isframe=1;
  if(tacIsX(&tac)!=1) {tacFree(&tac); return 31;}
  tac.isframe=0;
  if(tacIsX(&tac)!=1) {tacFree(&tac); return 32;}

  if(verbose>1) printf("\ntrying with all zeroes inside TAC.\n");
  tac.isframe=1;
  for(int i=0; i<tac.sampleNr; i++) tac.x1[i]=tac.x2[i]=0.0;
  if(tacIsX(&tac)!=0) {tacFree(&tac); return 41;}
  tac.isframe=0;
  if(tacIsX(&tac)!=1) {tacFree(&tac); return 42;}
  for(int i=0; i<tac.sampleNr; i++) tac.x[i]=0.0;
  if(tacIsX(&tac)!=0) {tacFree(&tac); return 43;}

  if(verbose>1) printf("\nno range in x values, but not zeroes either.\n");
  for(int i=0; i<tac.sampleNr; i++) tac.x[i]=tac.x1[i]=tac.x2[i]=10.0;
  tac.isframe=1;
  if(tacIsX(&tac)!=1) {tacFree(&tac); return 51;}
  tac.isframe=0;
  if(tacIsX(&tac)!=1) {tacFree(&tac); return 52;}


  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacXMatch(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  
  int ret;
  TAC tac1, tac2; 
  tacInit(&tac1); tacInit(&tac2);

  if(verbose>1) printf("\ntesting with NULL input\n");
  ret=tacXMatch(NULL, NULL, verbose-1);
  if(ret!=0) {tacFree(&tac1); tacFree(&tac2); return 1;}

  if(verbose>1) printf("\ncreating test data\n");
  ret=create_tac(&tac1); if(ret==TPCERROR_OK) ret=create_tac(&tac2);  
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  if(verbose>1) printf("\ntesting the same data\n");
  tac1.isframe=1; tac2.isframe=1;
  ret=tacXMatch(&tac1, &tac1, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 11;}
  if(verbose>1) printf("\ntesting the same data\n");
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 12;}
  if(verbose>1) printf("\nboth with frame middle times\n");
  tac1.isframe=0; tac2.isframe=0;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 13;}

  if(verbose>1) printf("\nsame times but frames/middle times\n");
  tac1.isframe=0; tac2.isframe=1;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 21;}
  tac1.isframe=1; tac2.isframe=0;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 22;}

  if(verbose>1) printf("\nsame times but different units\n");
  tac1.isframe=1; tac2.isframe=1;
  tacXUnitConvert(&tac1, UNIT_SEC, NULL);
  tacXUnitConvert(&tac2, UNIT_MIN, NULL);
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 31;}
  tacXUnitConvert(&tac1, UNIT_MIN, NULL);
  tacXUnitConvert(&tac2, UNIT_MSEC, NULL);
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 32;}

  if(verbose>1) printf("\nsame times in min but unknown units\n");
  tac1.isframe=1; tac2.isframe=1;
  tacXUnitConvert(&tac1, UNIT_MIN, NULL);
  tacXUnitConvert(&tac2, UNIT_MIN, NULL);
  tac1.tunit=UNIT_UNKNOWN;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 41;}
  tac2.tunit=UNIT_UNKNOWN;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 42;}
  tac1.tunit=UNIT_MIN;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 43;}
  tac2.tunit=UNIT_MIN;

  if(verbose>1) printf("\nsame times in sec but unknown units\n");
  tac1.isframe=1; tac2.isframe=1;
  tacXUnitConvert(&tac1, UNIT_SEC, NULL);
  tacXUnitConvert(&tac2, UNIT_SEC, NULL);
  tac1.tunit=UNIT_UNKNOWN;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret!=0) {tacFree(&tac1); tacFree(&tac2); return 41;}
  tac2.tunit=UNIT_UNKNOWN;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 42;}
  tac1.tunit=UNIT_SEC;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret!=0) {tacFree(&tac1); tacFree(&tac2); return 43;}
  tac2.tunit=UNIT_SEC;
  
  if(verbose>1) printf("\nSlightly different times accepted\n");
  tacXUnitConvert(&tac1, UNIT_MIN, NULL);
  tacXUnitConvert(&tac2, UNIT_MIN, NULL);
  tac2.x1[0]+=0.01; tac2.x2[0]+=0.01; tac2.x[0]+=0.01;
  tac1.isframe=1; tac2.isframe=0;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 51;}
  tac1.isframe=0; tac2.isframe=1;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 52;}
  tac1.isframe=0; tac2.isframe=0;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 53;}
  tac1.isframe=1; tac2.isframe=1;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 54;}
  
  if(verbose>1) printf("\nDifferent times not accepted\n");
  tacXUnitConvert(&tac1, UNIT_MIN, NULL);
  tacXUnitConvert(&tac2, UNIT_MIN, NULL);
  tac2.x1[0]+=0.09; tac2.x2[0]+=0.09; tac2.x[0]+=0.09;
  tac1.isframe=1; tac2.isframe=0;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret!=0) {tacFree(&tac1); tacFree(&tac2); return 61;}
  tac1.isframe=0; tac2.isframe=1;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret!=0) {tacFree(&tac1); tacFree(&tac2); return 62;}
  tac1.isframe=0; tac2.isframe=0;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret!=0) {tacFree(&tac1); tacFree(&tac2); return 63;}
  tac1.isframe=1; tac2.isframe=1;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret!=0) {tacFree(&tac1); tacFree(&tac2); return 64;}
  
  if(verbose>1) printf("\nNaN times at the same pos accepted\n");
  tac1.x1[0]=tac1.x2[0]=tac1.x[0]=nan("");
  tac2.x1[0]=tac2.x2[0]=tac2.x[0]=nan("");
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 71;}
  
  if(verbose>1) printf("\nDifferent sample nr does not matter\n");
  tac2.sampleNr=tac1.sampleNr-1;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 81;}
  tac2.sampleNr=tac1.sampleNr;
  tac1.sampleNr=tac2.sampleNr-1;
  ret=tacXMatch(&tac1, &tac2, verbose-1);
  if(ret==0) {tacFree(&tac1); tacFree(&tac2); return 82;}
  tac1.sampleNr=tac2.sampleNr;

  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacAddZeroSample(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  
  int ret;
  TAC tac1, tac2; 
  tacInit(&tac1); tacInit(&tac2);

  if(verbose>1) printf("\n testing with NULL input \n");
  
  ret=tacAddZeroSample((TAC*)NULL, NULL);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 1;}

  if(verbose>1) printf("\n testing with empty input \n");
  
  ret=tacAddZeroSample(&tac1, NULL);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 2;}
  
  if(verbose>1) printf("\n creating test data \n");
  ret=create_tac(&tac1); if(ret==0) ret=create_tac(&tac2);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  if(verbose>1) printf("\n data already has zero sample \n");

  tac1.isframe=1;
  if(verbose>3) {
    printf("\n original TAC:\n"); 
    tacWrite(&tac1, stdout, TAC_FORMAT_PMOD, 1, NULL);
  }
  ret=tacAddZeroSample(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 11;}
  if(verbose>3) {
    printf("\n TAC after processing:\n"); 
    tacWrite(&tac1, stdout, TAC_FORMAT_PMOD, 1, NULL);
  }
  if(tacCompareUnit(&tac1, &tac2, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 12;}
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 13;}
  if(tacCompareConc(&tac1, &tac2, -1, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 14;}
  if(tacCompareTimes(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 15;}


  if(verbose>1) printf("\n data starts after zero \n");

  tac1.isframe=1;
  tac1.x1[0]=0.5*tac1.x2[0];
  if(verbose>3) {
    printf("\n original TAC:\n"); 
    tacWrite(&tac1, stdout, TAC_FORMAT_PMOD, 1, NULL);
  }
  ret=tacAddZeroSample(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 21;}
  if(verbose>3) {
    printf("\n TAC after processing:\n"); 
    tacWrite(&tac1, stdout, TAC_FORMAT_PMOD, 1, NULL);
  }
  if(tacCompareUnit(&tac1, &tac2, status)!=0) {tacFree(&tac1); tacFree(&tac2); return 22;}
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0) {tacFree(&tac1); tacFree(&tac2); return 23;}
  if(tac1.sampleNr!=tac2.sampleNr+1) {tacFree(&tac1); tacFree(&tac2); return 24;}
  if(!doubleMatch(0.0, tac1.x1[0], 1.0E-08)) {
    tacFree(&tac1); tacFree(&tac2); return 25;}
  if(!doubleMatch(tac1.x1[1], tac1.x2[0], 1.0E-08)) {tacFree(&tac1); tacFree(&tac2); return 26;}
  for(int j=0; j<tac1.tacNr; j++)
    if(!doubleMatch(0.0, tac1.c[j].y[0], 1.0E-08)) {tacFree(&tac1); tacFree(&tac2); return 27;}
  for(int i=0; i<tac2.sampleNr; i++) {
    if(i>0 && !doubleMatch(tac2.x1[i], tac1.x1[i+1], 1.0E-08)) {
      tacFree(&tac1); tacFree(&tac2); return 28;}
    if(i>0 && !doubleMatch(tac2.x2[i], tac1.x2[i+1], 1.0E-08)) {
      tacFree(&tac1); tacFree(&tac2); return 28;}
    for(int j=0; j<tac2.tacNr; j++)
      if(!doubleMatch(tac2.c[j].y[i], tac1.c[j].y[i+1], 1.0E-08)) {
        tacFree(&tac1); tacFree(&tac2); return 29;}
  }


  if(verbose>1) printf("\n what about mid frames? \n");


  if(verbose>1) printf("\n creating test data \n");
  tacFree(&tac1);
  ret=create_tac(&tac1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  if(verbose>1) printf("\n data already has zero sample \n");

  tac1.isframe=0; tac2.isframe=0;
  tac1.x[0]=tac2.x[0]=0.0;
  if(verbose>3) {
    printf("\n original TAC:\n"); 
    tacWrite(&tac1, stdout, TAC_FORMAT_PMOD, 1, NULL);
  }
  ret=tacAddZeroSample(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 51;}
  if(verbose>3) {
    printf("\n TAC after processing:\n"); 
    tacWrite(&tac1, stdout, TAC_FORMAT_PMOD, 1, NULL);
  }
  if(tacCompareUnit(&tac1, &tac2, status)!=0) {tacFree(&tac1); tacFree(&tac2); return 52;}
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0) {tacFree(&tac1); tacFree(&tac2); return 53;}
  if(tacCompareConc(&tac1, &tac2, -1, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 54;}
  if(tacCompareTimes(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 55;}


  if(verbose>1) printf("\n data starts after zero \n");
  
  tac1.x[0]=tac2.x[0]=0.5*(tac2.x1[0]+tac2.x2[0]);
  if(verbose>3) {
    printf("\n original TAC:\n"); 
    tacWrite(&tac1, stdout, TAC_FORMAT_PMOD, 1, NULL);
  }
  ret=tacAddZeroSample(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 61;}
  if(verbose>3) {
    printf("\n TAC after processing:\n"); 
    tacWrite(&tac1, stdout, TAC_FORMAT_PMOD, 1, NULL);
  }
  if(tacCompareUnit(&tac1, &tac2, status)!=0) {tacFree(&tac1); tacFree(&tac2); return 62;}
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0) {tacFree(&tac1); tacFree(&tac2); return 63;}
  if(tac1.sampleNr!=tac2.sampleNr+1) {tacFree(&tac1); tacFree(&tac2); return 64;}
  if(!doubleMatch(0.0, tac1.x[0], 1.0E-08)) {tacFree(&tac1); tacFree(&tac2); return 65;}
  for(int j=0; j<tac2.tacNr; j++)
    if(!doubleMatch(0.0, tac1.c[j].y[0], 1.0E-08)) {tacFree(&tac1); tacFree(&tac2); return 67;}
  for(int i=0; i<tac2.sampleNr; i++) {
    if(!doubleMatch(tac2.x[i], tac1.x[i+1], 1.0E-08)) {
      tacFree(&tac1); tacFree(&tac2); return 68;}
    for(int j=0; j<tac2.tacNr; j++)
      if(!doubleMatch(tac2.c[j].y[i], tac1.c[j].y[i+1], 1.0E-08)) {
        tacFree(&tac1); tacFree(&tac2); return 69;}
  }


  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacDeleteSample(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  
  int ret;
  TAC tac1, tac2; 
  tacInit(&tac1); tacInit(&tac2);

  if(verbose>1) printf("\n testing with NULL input \n");
  ret=tacDeleteSample((TAC*)NULL, 0);
  if(ret==TPCERROR_OK) return 1;

  if(verbose>1) printf("\n testing with empty input \n");
  ret=tacDeleteSample(&tac1, 0);
  if(ret==TPCERROR_OK) return 2;
  
  if(verbose>1) printf("\n creating test data \n");
  ret=create_tac(&tac1); if(ret==0) ret=create_tac(&tac2);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  if(verbose>1) printf("\n trying to delete sample that is not there \n");
  ret=tacDeleteSample(&tac1, -1);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 11;}
  ret=tacDeleteSample(&tac1, tac1.sampleNr);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 12;}


  if(verbose>1) printf("\n delete the first sample \n");
  
  tac1.isframe=tac2.isframe=0;
  ret=tacAddZeroSample(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 21;}
  ret=tacDeleteSample(&tac1, 0);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 22;}
  if(tacCompareUnit(&tac1, &tac2, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 23;}
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 24;}
  if(tacCompareConc(&tac1, &tac2, -1, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 25;}
  if(tacCompareTimes(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 26;}


  if(verbose>1) printf("\n delete the last sample \n");
  
  tac1.isframe=tac2.isframe=1;
  tac2.sampleNr--;
  ret=tacDeleteSample(&tac1, tac1.sampleNr-1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 32;}
  if(tacCompareUnit(&tac1, &tac2, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 33;}
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 34;}
  if(tacCompareConc(&tac1, &tac2, -1, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 35;}
  if(tacCompareTimes(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 36;}


  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacDeleteMissingSamples(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  if(verbose>1) printf("\n testing with NULL input \n");
  ret=tacDeleteMissingSamples((TAC*)NULL);
  if(ret==TPCERROR_OK) return 1;

  TAC tac; tacInit(&tac);

  if(verbose>1) printf("\n testing with empty input \n");
  ret=tacDeleteMissingSamples(&tac);
  if(ret!=TPCERROR_OK) return 2;

  if(verbose>1) printf("\n creating test data \n");
  ret=create_tac(&tac);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    tacFree(&tac); return 10;
  }
  if(verbose>10) {
    tacWrite(&tac, stdout, TAC_FORMAT_SIMPLE, 1, status);
    fflush(stdout);
  }

  if(verbose>1) printf("\n testing with correct data \n");
  int sampleNr=tac.sampleNr;
  ret=tacDeleteMissingSamples(&tac);
  if(ret!=TPCERROR_OK || tac.sampleNr!=sampleNr) return 11;
  
  if(verbose>1) printf("\n testing with data with missing values \n");
  tac.c[0].y[0]=nan("");
  ret=tacDeleteMissingSamples(&tac);
  if(ret!=TPCERROR_OK || tac.sampleNr!=sampleNr) return 12;

  if(verbose>1) printf("\n testing with data with more missing values \n");
  for(int i=0; i<tac.tacNr; i++) tac.c[i].y[0]=nan("");
  ret=tacDeleteMissingSamples(&tac);
  if(ret!=TPCERROR_OK || tac.sampleNr!=sampleNr-1) return 13;

  if(verbose>1) printf("\n testing with missing frame times \n");
  sampleNr=tac.sampleNr;
  tac.x[0]=tac.x1[0]=tac.x2[0]=nan("");
  ret=tacDeleteMissingSamples(&tac);
  if(ret!=TPCERROR_OK || tac.sampleNr!=sampleNr-1) return 14;

  if(verbose>10) {
    tacWrite(&tac, stdout, TAC_FORMAT_SIMPLE, 1, status);
    fflush(stdout);
  }

  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacExtractRange(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int ret;
  TAC tac1, tac2; 
  tacInit(&tac1); tacInit(&tac2);

  if(verbose>1) printf("\n testing with NULL input \n");
  ret=tacExtractRange((TAC*)NULL, (TAC*)NULL, 0.0, 0.0);
  if(ret==TPCERROR_OK) return 1;

  if(verbose>1) printf("\n creating test data \n");
  ret=create_tac(&tac1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  if(verbose>1) printf("\n testing with invalid time range \n");
  ret=tacExtractRange(&tac1, &tac2, 1000.0, 1.0);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 11;}
  ret=tacExtractRange(&tac1, &tac2, -1000.0, -200.0);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 12;}
  ret=tacExtractRange(&tac1, &tac2, 10000.0, 200000.0);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 13;}

  if(verbose>1) printf("\n testing with valid time range \n");
  
  ret=tacExtractRange(&tac1, &tac2, tac1.x1[1], tac1.x1[tac1.sampleNr-1]);
  if(verbose>2) printf("  ret:=%d\n", ret);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    tacFree(&tac1); tacFree(&tac2); return 21;
  }
  ret=tacDeleteSample(&tac1, 0);
  if(ret==TPCERROR_OK) ret=tacDeleteSample(&tac1, tac1.sampleNr-1);
  if(ret!=TPCERROR_OK) return 22;
  if(tacCompareUnit(&tac1, &tac2, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 23;}
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 24;}
  if(tacCompareConc(&tac1, &tac2, -1, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 25;}
  if(tacCompareTimes(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 26;}
  if(tacCompareWeights(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 27;}

  if(verbose>1) printf("\n testing in-place process \n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  tacFree(&tac2);
  ret=create_tac(&tac2); if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 100;}
  ret=tacExtractRange(&tac2, &tac2, tac2.x1[1], tac2.x1[tac2.sampleNr-1]);
  if(verbose>2) printf("  ret:=%d\n", ret);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    tacFree(&tac1); tacFree(&tac2); return 101;
  }
  if(tacCompareUnit(&tac1, &tac2, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 102;}
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 103;}
  if(tacCompareConc(&tac1, &tac2, -1, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 103;}
  if(tacCompareTimes(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 104;}
  if(tacCompareWeights(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 105;}


  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacExtractSamples(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int ret;
  TAC tac1, tac2; 
  tacInit(&tac1); tacInit(&tac2);

  if(verbose>1) printf("\n testing with NULL input \n");
  ret=tacExtractSamples((TAC*)NULL, (TAC*)NULL, 0, 0);
  if(ret==TPCERROR_OK) return 1;

  if(verbose>1) printf("\n creating test data \n");
  ret=create_tac(&tac1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  if(verbose>1) printf("\n testing with invalid sample range \n");
  ret=tacExtractSamples(&tac1, &tac2, 1000, 1);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 11;}
  ret=tacExtractSamples(&tac1, &tac2, -1, 2);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 12;}
  ret=tacExtractSamples(&tac1, &tac2, 10000, 200000);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 13;}

  if(verbose>1) printf("\n testing with valid sample range \n");
  
  ret=tacExtractSamples(&tac1, &tac2, 1, tac1.sampleNr-2);
  if(verbose>2) printf("  ret:=%d\n", ret);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    tacFree(&tac1); tacFree(&tac2); return 21;
  }
  ret=tacDeleteSample(&tac1, 0);
  if(ret==TPCERROR_OK) ret=tacDeleteSample(&tac1, tac1.sampleNr-1);
  if(ret!=TPCERROR_OK) return 22;
  if(tacCompareUnit(&tac1, &tac2, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 23;}
  if(tacCompareNames(&tac1, &tac2, -1, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 24;}
  if(tacCompareConc(&tac1, &tac2, -1, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 25;}
  if(tacCompareTimes(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 26;}
  if(tacCompareWeights(&tac1, &tac2, 1.0E-08, 1.0E-08, status)!=0) {
    tacFree(&tac1); tacFree(&tac2); return 27;}


  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacCheckX1X2X(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int i;
  double v;
  TAC tac; 
  tacInit(&tac);

  if(verbose>1) printf("\n testing with NULL input \n");
  if(tacCheckX1X2X((TAC*)NULL)) return 1;
  if(verbose>1) printf("\n testing with empty data \n");
  if(tacCheckX1X2X(&tac)) return 2;

  if(verbose>1) printf("\n creating test data \n");
  if(create_tac(&tac)!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("\n test data should be fine \n");
  if(!tacCheckX1X2X(&tac)) {tacFree(&tac); return 11;}

  if(verbose>1) printf("\n NaN in x1 or x2 or x checks as fail \n");
  v=tac.x[0]; tac.x[0]=nan("");
  if(tacCheckX1X2X(&tac)) {tacFree(&tac); return 21;}
  tac.x[0]=v; 
  v=tac.x2[1]; tac.x2[1]=nan("");
  if(tacCheckX1X2X(&tac)) {tacFree(&tac); return 22;}
  tac.x2[1]=v; 
  if(verbose>1) printf("\n ... but ok if all are NaN \n");
  tac.x[2]=tac.x1[2]=tac.x2[2]=nan("");
  if(!tacCheckX1X2X(&tac)) {tacFree(&tac); return 23;}

  if(verbose>1) printf("\n ok if x is very close to mean of x1 and x2 \n");
  tac.x[2]=15.2; tac.x1[2]=10.0; tac.x2[2]=20.0; tacSortByTime(&tac, NULL);
  if(!tacCheckX1X2X(&tac)) {tacFree(&tac); return 31;}
  i=tac.sampleNr-1;
  tac.x[i]=14.7; tac.x1[i]=10.0; tac.x2[i]=20.0;
  if(!tacCheckX1X2X(&tac)) {tacFree(&tac); return 32;}

  if(verbose>1) printf("\n not ok if x is not close to mean of x1 and x2 \n");
  i=tac.sampleNr-1;
  tac.x[i]=18.0; tac.x1[i]=10.0; tac.x2[i]=20.0; tacSortByTime(&tac, NULL);
  if(tacCheckX1X2X(&tac)) {tacFree(&tac); return 41;}
  tac.x[i]=13.0; tac.x1[i]=10.0; tac.x2[i]=20.0; tacSortByTime(&tac, NULL);
  if(tacCheckX1X2X(&tac)) {tacFree(&tac); return 42;}

  if(verbose>1) printf("\n recreating test data \n");
  tacFree(&tac);
  if(create_tac(&tac)!=TPCERROR_OK) {tacFree(&tac); return 100;}

  if(verbose>1) printf("\n ok if big gap in x values on one side\n");
  i=tac.sampleNr-1;
  tac.x1[i]=100.0; tac.x2[i]=200.0; tac.x[i]=150.0; 
  if(!tacCheckX1X2X(&tac)) {tacFree(&tac); return 111;}
  tac.x1[i]=100.0; tac.x2[i]=200.0; tac.x[i]=148.0; 
  if(!tacCheckX1X2X(&tac)) {tacFree(&tac); return 112;}
  
  if(verbose>1) printf("\n also ok if small overlap in x values \n");
  i=tac.sampleNr-2;
  tac.x1[i]=50.0; tac.x2[i]=101.0; tac.x[i]=75.7; 
  if(!tacCheckX1X2X(&tac)) {tacFree(&tac); return 121;}
  tac.x1[i]=50.0; tac.x2[i]=105.0; tac.x[i]=75.0; 
  if(!tacCheckX1X2X(&tac)) {tacFree(&tac); return 122;}
  
  if(verbose>1) printf("\n not ok if large overlap in x values \n");
  i=tac.sampleNr-2;
  tac.x1[i]=60.0; tac.x2[i]=120.0; tac.x[i]=90.0; 
  if(tacCheckX1X2X(&tac)) {tacFree(&tac); return 131;}
  tac.x1[i]=60.0; tac.x2[i]=120.0; tac.x[i]=85.0; 
  if(tacCheckX1X2X(&tac)) {tacFree(&tac); return 132;}

  if(verbose>1) printf("\n not ok if big gap in x values on both sides\n");
  i=tac.sampleNr-2;
  tac.x1[i]=20.0; tac.x2[i]=30.0; tac.x[i]=25.0; 
  i=tac.sampleNr-1;
  tac.x1[i]=100.0; tac.x2[i]=120.0; tac.x[i]=110.0; 
  if(tacCheckX1X2X(&tac)) {tacFree(&tac); return 141;}


  tacFree(&tac); 
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacSetX(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int i, ret;
  TAC tac1, tac2; 
  tacInit(&tac1); tacInit(&tac2);

  if(verbose>1) printf("\n testing with NULL input \n");
  ret=tacSetX((TAC*)NULL, NULL);
  if(ret==TPCERROR_OK) return 1;
  if(verbose>1) printf("\n testing with empty data \n");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) return 2;

  if(verbose>1) printf("\n creating test data \n");
  ret=create_tac(&tac1); if(ret==TPCERROR_OK) ret=create_tac(&tac2);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  if(verbose>1) printf("\n reasonable frame times are not changed \n");
  tac1.isframe=1; ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 11;}
  if(!tacXMatch(&tac1, &tac2, verbose-3)) {tacFree(&tac1); tacFree(&tac2); return 12;}
  tac1.isframe=0; ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 13;}
  if(!tacXMatch(&tac1, &tac2, verbose-3)) {tacFree(&tac1); tacFree(&tac2); return 14;}

  if(verbose>1) printf("\n frame mid times can be correctly calculated \n");
  tac1.isframe=1;
  for(i=0; i<tac1.sampleNr; i++) tac1.x[i]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 21;}
  if(!tacXMatch(&tac1, &tac2, verbose-3)) {tacFree(&tac1); tacFree(&tac2); return 22;}

  if(verbose>1) printf("\n estimating frame start and end times \n");

  if(verbose>1) printf("\n just one sample \n");
  tac1.isframe=0; tac1.sampleNr=1; tac1.x[0]=1.0; tac1.x1[0]=tac1.x2[0]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 31;}
  if(!doubleMatch(tac1.x1[0], 0.0, 1.0E-10)) {tacFree(&tac1); tacFree(&tac2); return 32;}
  if(!doubleMatch(tac1.x2[0], 2.0, 1.0E-10)) {tacFree(&tac1); tacFree(&tac2); return 33;}

  if(verbose>1) printf("\n just one negative sample \n");
  tac1.isframe=0; tac1.sampleNr=1; tac1.x[0]=-1.0; tac1.x1[0]=tac1.x2[0]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 41;}
  if(!doubleMatch(tac1.x1[0], -2.0, 1.0E-10)) {
    tacFree(&tac1); tacFree(&tac2); return 42;
  }
  if(!doubleMatch(tac1.x2[0], 0.0, 1.0E-10)) {tacFree(&tac1); tacFree(&tac2); return 43;}

  if(verbose>1) printf("\n just one zero sample should lead to error \n");
  tac1.isframe=0; tac1.sampleNr=1; tac1.x[0]=0.0; tac1.x1[0]=tac1.x2[0]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 51;}

  if(verbose>1) printf("\n all frame durations are 1 min, starting from 0 \n");
  tac1.sampleNr=tac2.sampleNr;
  tacXCopy(&tac2, &tac1, 0, tac2.sampleNr-1);
  tac1.isframe=0; tac1.tunit=UNIT_MIN;
  for(i=0; i<tac1.sampleNr; i++) tac1.x1[i]=tac1.x2[i]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 61;}
  if(verbose>2) {
    printf("x\tx1\tx2\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\n", tac2.x[i], tac1.x1[i], tac1.x2[i]);
  }
  tac1.isframe=1;
  if(!tacXMatch(&tac1, &tac2, verbose-3)) {tacFree(&tac1); tacFree(&tac2); return 62;}


  if(verbose>1) printf("\n all frame durations are 5 min, starting from 60 \n");
  tac2.tunit=UNIT_MIN; tac2.isframe=1;
  tac2.x1[0]=60.0; tac2.x2[0]=65.0;
  for(i=1; i<tac2.sampleNr; i++) {tac2.x1[i]=tac2.x2[i-1]; tac2.x2[i]=tac2.x1[i]+5.0;}
  tacSetX(&tac2, NULL);
  tacXCopy(&tac2, &tac1, 0, tac2.sampleNr-1);
  tac1.isframe=0;
  for(i=0; i<tac1.sampleNr; i++) tac1.x1[i]=tac1.x2[i]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 71;}
  if(verbose>2) {
    printf("x\tx1\tx2\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\n", tac1.x[i], tac1.x1[i], tac1.x2[i]);
  }
  tac1.isframe=1;
  if(!tacXMatch(&tac1, &tac2, verbose-3)) {tacFree(&tac1); tacFree(&tac2); return 72;}



  if(verbose>1) printf("\n creating longer test data \n");
  tacFree(&tac1); tacFree(&tac2);
  ret=create_tac2(&tac1); 
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 200;}

  if(verbose>1) printf("\n frames: 5x0.5, 5x2, and 5x5 min, starting from 0 \n");
  i=0; tac1.x1[i]=0.0; tac1.x2[i]=0.5; i++;
  for(; i<5; i++) {tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+0.5;}
  for(; i<10; i++) {tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+2.0;}
  for(; i<15; i++) {tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+5.0;}
  tac1.isframe=1; tacSetX(&tac1, NULL);
  if(tacDuplicate(&tac1, &tac2)) {tacFree(&tac1); tacFree(&tac2); return 210;}
  tac1.isframe=0;
  if(verbose>2) {
    printf("x\tx1\tx2\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\n", tac1.x[i], tac1.x1[i], tac1.x2[i]);
  }
  for(i=0; i<tac1.sampleNr; i++) tac1.x1[i]=tac1.x2[i]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 211;}
  if(verbose>2) {
    printf("x\tx1\tx2\tx2-x1\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\t%g\n", tac1.x[i], tac1.x1[i], tac1.x2[i], tac1.x2[i]-tac1.x1[i]);
  }
  tac1.isframe=1;
  if(!tacXMatch(&tac1, &tac2, verbose-3)) {tacFree(&tac1); tacFree(&tac2); return 212;}


  if(verbose>1) printf("\n frames: 1x15, 4x5, 5x10, and 5x30 sec, starting from 0 \n");
  i=0; tac1.x1[i]=0.0; tac1.x2[i]=15.0; i++;
  tac1.x1[i]=15.0; tac1.x2[i]=20.0; i++;
  for(; i<5; i++) {tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+5.0;}
  for(; i<10; i++) {tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+10.0;}
  for(; i<15; i++) {tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+30.0;}
  tac1.isframe=1; tacSetX(&tac1, NULL); tac1.tunit=UNIT_SEC;
  if(tacDuplicate(&tac1, &tac2)) {tacFree(&tac1); tacFree(&tac2); return 220;}
  tac1.isframe=0;
  if(verbose>2) {
    printf("x\tx1\tx2\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\n", tac1.x[i], tac1.x1[i], tac1.x2[i]);
  }
  for(i=0; i<tac1.sampleNr; i++) tac1.x1[i]=tac1.x2[i]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 221;}
  if(verbose>2) {
    printf("x\tx1\tx2\tx2-x1\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\t%g\n", tac1.x[i], tac1.x1[i], tac1.x2[i], tac1.x2[i]-tac1.x1[i]);
  }
  tac1.isframe=1;
  if(!tacXMatch(&tac1, &tac2, verbose-3)) {tacFree(&tac1); tacFree(&tac2); return 222;}



  if(verbose>1) printf("\n frames: 5x1, 1x5, 5x10 min, 30 min pause, 4x10 min\n");
  i=0; tac1.x1[i]=0.0; tac1.x2[i]=1.0; i++;
  for(; i<5; i++) {tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+1.0;}
  tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+5.0; i++;
  for(; i<11; i++) {tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+10.0;}
  tac1.x1[i]=90.0; tac1.x2[i]=100.0; i++;
  for(; i<15; i++) {tac1.x1[i]=tac1.x2[i-1]; tac1.x2[i]=tac1.x1[i]+10.0;}
  tac1.isframe=1; tacSetX(&tac1, NULL); tac1.tunit=UNIT_MIN;
  if(tacDuplicate(&tac1, &tac2)) {tacFree(&tac1); tacFree(&tac2); return 230;}
  tac1.isframe=0;
  if(verbose>2) {
    printf("x\tx1\tx2\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\n", tac1.x[i], tac1.x1[i], tac1.x2[i]);
  }
  for(i=0; i<tac1.sampleNr; i++) tac1.x1[i]=tac1.x2[i]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 231;}
  if(verbose>2) {
    printf("x\tx1\tx2\tx2-x1\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\t%g\n", tac1.x[i], tac1.x1[i], tac1.x2[i], tac1.x2[i]-tac1.x1[i]);
  }
  tac1.isframe=1;
  if(!tacXMatch(&tac1, &tac2, verbose-3)) {tacFree(&tac1); tacFree(&tac2); return 232;}



  if(verbose>1) printf("\n frames: 1, 2, 3, ... min\n");
  for(i=0; i<15; i++) {
    if(i==0) tac1.x1[i]=0.0; else tac1.x1[i]=tac1.x2[i-1];
    tac1.x2[i]=tac1.x1[i]+(1.0+i);
  }
  tac1.isframe=1; tacSetX(&tac1, NULL); tac1.tunit=UNIT_MIN;
  if(tacDuplicate(&tac1, &tac2)) {tacFree(&tac1); tacFree(&tac2); return 240;}
  tac1.isframe=0;
  if(verbose>2) {
    printf("x\tx1\tx2\tx2-x1\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\t%g\n", tac1.x[i], tac1.x1[i], tac1.x2[i], tac1.x2[i]-tac1.x1[i]);
  }
  for(i=0; i<tac1.sampleNr; i++) tac1.x1[i]=tac1.x2[i]=nan("");
  ret=tacSetX(&tac1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 241;}
  if(verbose>2) {
    printf("x\tx1\tx2\tx2-x1\n");
    for(i=0; i<tac1.sampleNr; i++)
      printf("%g\t%g\t%g\t%g\n", tac1.x[i], tac1.x1[i], tac1.x2[i], tac1.x2[i]-tac1.x1[i]);
  }
  tac1.isframe=1;
  if(!tacXMatch(&tac1, &tac2, verbose-3)) {tacFree(&tac1); tacFree(&tac2); return 242;}


  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacGetSampleInterval(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  
  int ret;
  TAC tac; tacInit(&tac);
  double mi=0.0, ma=0.0, li=nan("");

  if(verbose>1) printf("\ntrying with NULL TAC.\n");
  ret=tacGetSampleInterval(NULL, li, NULL, NULL);
  if(ret==TPCERROR_OK) return 1;
  ret=tacGetSampleInterval(NULL, li, &mi, &ma);
  if(ret==TPCERROR_OK) return 2;
  if(!isnan(mi) || !isnan(ma)) return 3;

  if(verbose>1) printf("\ntrying with empty TAC.\n");
  ret=tacGetSampleInterval(&tac, li, NULL, NULL);
  if(ret==TPCERROR_OK) return 5;
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret==TPCERROR_OK) return 6;
  if(!isnan(mi) || !isnan(ma)) return 7;
  
  if(verbose>1) printf("\ncreating test data\n");
  ret=create_tac(&tac);  
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("\ntrying with valid and NULL output pointers.\n");
  ret=tacGetSampleInterval(&tac, li, NULL, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 11;}
  ret=tacGetSampleInterval(&tac, li, &mi, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 12;}
  ret=tacGetSampleInterval(&tac, li, NULL, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 13;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-12)) {tacFree(&tac); return 14;}
  if(!doubleMatch(ma, 1.0, 1.0E-12)) {tacFree(&tac); return 15;}

  li=0.5;
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 16;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-12)) {tacFree(&tac); return 17;}
  if(!doubleMatch(ma, 1.0, 1.0E-12)) {tacFree(&tac); return 18;}

  if(verbose>1) printf("\ntrying with frame middle times.\n");
  tac.isframe=0;
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 21;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-12)) {tacFree(&tac); return 22;}
  if(!doubleMatch(ma, 1.0, 1.0E-12)) {tacFree(&tac); return 23;}

  if(verbose>1) printf("\ntrying with just one sample.\n");
  tac.sampleNr=1;
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 31;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 0.5, 1.0E-12)) {tacFree(&tac); return 32;}
  if(!doubleMatch(ma, 0.5, 1.0E-12)) {tacFree(&tac); return 33;}
  tac.isframe=1;
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 34;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-12)) {tacFree(&tac); return 35;}
  if(!doubleMatch(ma, 1.0, 1.0E-12)) {tacFree(&tac); return 36;}

  if(verbose>1) printf("\nreset test data\n");
  tacFree(&tac); ret=create_tac(&tac); if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}
  tac.x1[0]=0.5; tac.x2[0]=0.7; tac.x[0]=0.6;
  tac.x1[tac.sampleNr-1]=100.0; tac.x2[tac.sampleNr-1]=200.0; tac.x[tac.sampleNr-1]=150.0;
  li=nan("");
  if(verbose>4) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 0, NULL);

  if(verbose>1) printf("\ntrying with different frame lengths.\n");
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 101;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 0.2, 1.0E-12)) {tacFree(&tac); return 102;}
  if(!doubleMatch(ma, 100.0, 1.0E-12)) {tacFree(&tac); return 103;}

  if(verbose>1) printf("\ntrying with frame middle times.\n");
  tac.isframe=0;
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 111;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 0.9, 1.0E-12)) {tacFree(&tac); return 112;}
  if(!doubleMatch(ma, 146.5, 1.0E-12)) {tacFree(&tac); return 113;}

  if(verbose>1) printf("\nuse the limit.\n");
  li=0.3;
  tac.isframe=1; 
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 121;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-12)) {tacFree(&tac); return 122;}
  if(!doubleMatch(ma, 100.0, 1.0E-12)) {tacFree(&tac); return 123;}

  if(verbose>1) printf("\ntrying with frame middle times.\n");
  tac.isframe=0; 
  li=1.0;
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 131;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 1.0, 1.0E-12)) {tacFree(&tac); return 132;}
  if(!doubleMatch(ma, 146.5, 1.0E-12)) {tacFree(&tac); return 133;}
  li=1.1;
  ret=tacGetSampleInterval(&tac, li, &mi, &ma);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 134;}
  if(verbose>2) printf(" min := %g\n max := %g\n", mi, ma);
  if(!doubleMatch(mi, 146.5, 1.0E-12)) {tacFree(&tac); return 135;}
  if(!doubleMatch(ma, 146.5, 1.0E-12)) {tacFree(&tac); return 136;}

  tacFree(&tac);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacToBars(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("\n  tested with application tac4frpl\n");

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
