/** @file tpctac.h
 *  @brief Header file for library libtpctac.
 *  @details Time-activity curve (TAC) processing and file i/o.
 *  @author Vesa Oikonen
 *  @copyright (c) Turku PET Centre
 */
#ifndef _TPCTAC_H_
#define _TPCTAC_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
//#include "tpcextensions.h"
#include "tpccsv.h"
#include "tpcift.h"
#include "tpcisotope.h"
/*****************************************************************************/

/*****************************************************************************/
/** @public TAC file format codes; see tac_format */
typedef enum {
  TAC_FORMAT_UNKNOWN,     ///< Unknown format
  TAC_FORMAT_SIMPLE,      ///< x and y's with space delimiters
  TAC_FORMAT_DFT,         ///< Data format of Turku PET Centre
  TAC_FORMAT_IFT,         ///< Interfile-type data (supported for writing)
  TAC_FORMAT_NCI,         ///< Old Turku PET Centre format
  TAC_FORMAT_PMOD,        ///< PMOD TAC format
  TAC_FORMAT_CSV_INT,     ///< International CSV
  TAC_FORMAT_CSV_UK,      ///< UK CSV
  TAC_FORMAT_TSV_INT,     ///< International TSV (comma as decimal separator)
  TAC_FORMAT_TSV_UK,      ///< UK TSV (point as decimal separator)
  TAC_FORMAT_CPT,         ///< CPT format
  TAC_FORMAT_IDWC,        ///< Hammersmith tissue file format
  TAC_FORMAT_IF,          ///< Hammersmith input file format
  TAC_FORMAT_XML,         ///< XML format (supported for writing)
  TAC_FORMAT_HTML,        ///< HTML table format (supported for writing)
  TAC_FORMAT_SIF,         ///< Scan information file
  TAC_FORMAT_XELERIS,     ///< Xeleris format (reading supported) 
  TAC_FORMAT_INVEON,      ///< Inveon format (reading supported) 
  TAC_FORMAT_AMIDE,       ///< Amide format (reading supported)
  TAC_FORMAT_CARIMAS_TXT, ///< Carimas txt format (reading supported) 
  TAC_FORMAT_QVIEW,       ///< QView CSV TAC format (reading supported)
  TAC_FORMAT_MAT,         ///< Matlab matrix TAC format (reading supported)
  TAC_FORMAT_HRRT_HC,     ///< HRRT head curve format (reading supported)
  TAC_FORMAT_HRPLUS_HC,   ///< HR+ head curve format (reading supported)
  TAC_FORMAT_ABSS_SCANDITRONICS, ///< Scanditronics ABSS data; reading supported
  TAC_FORMAT_ABSS_GEMS,   ///< GEMS ABSS data; reading supported
  TAC_FORMAT_ABSS_ALLOGG_OLD, ///< ALLOGG ABSS data (old format); reading supported
  TAC_FORMAT_ABSS_ALLOGG, ///< ALLOGG ABSS data; reading supported
  TAC_FORMAT_BINARY,      ///< Binary format (currently not supported)
  TAC_FORMAT_LAST
} tacformat;
/*****************************************************************************/

/*****************************************************************************/
/** Concentrations, name, size etc for a single time-activity curve inside TAC struct.
   @sa TAC, PAR, PARR, PARN
 */
typedef struct TACC {
  /** Content type: TACID_UNKNOWN, TACID_REF, TACID_PLASMA, TACID_BLOOD,
      TACID_PARENT, TACID_METABOLITE, TACID_VOI, TACID_VOXEL */
  tactype   type;  
  /** VOI volume, ROI size, sample mass, or something like that. */
  double    size;
  /** Size unit: UNIT_UNKNOWN, UNIT_UL, ... */
  unit      sunit;
  /** Pointer to y (concentration) values. */
  double   *y;
  /** Temporary switch. */
  char      sw;
  /** Temporary switch. */
  char      sw2;
  /** Full name string, using '_' to separate any name tokens when necessary. */
  char      name[MAX_TACNAME_LEN+1];
} TACC;
/*****************************************************************************/
/** Definition for a set of TACs with common header information and x (sample time) values.
   @sa TACC, MTAC, PAR, CSV, IFT, tacInit, tacFree
 */
typedef struct TAC {
  /** Nr of samples (frames) in each TAC; note that more memory may be allocated than this. */
  int           sampleNr;
  /** Nr of TACCs (VOIs). */
  int           tacNr;
  /** File format code: TAC_FORMAT_* */  
  tacformat     format;
  /** Sample time: are sample start and end times available (1), or just the middle time point (0) */
  int           isframe;
  /** Sample time (or middle frame time); units specified with tunit */
  double       *x;
  /** Sampling (frame) start time; units specified with tunit. */
  double       *x1;
  /** Sampling (frame) end time; units specified with tunit. */
  double       *x2;
  /** Unit of y (concentration) values: UNIT_* 
     @sa unit, tunit
    */
  int           cunit;
  /** Unit of x (sample time) values: UNIT_*
     @sa unit, cunit
   */
  int           tunit;
  /** Pointer to weight factors */
  double       *w;
  /** Variable indicating whether weights are present:
      WEIGHTING_UNKNOWN, WEIGHTING_ON, WEIGHTING_OFF, WEIGHTING_ON_F, WEIGHTING_ON_FD, ...
      @sa tacIsWeighted */
  weights       weighting;
  /** Pointers to y (concentration) data */
  TACC         *c;
  /** Internal variable: Number of allocated TACCs (VOIs) */
  int           _tacNr;
  /** Internal variable: Number of allocated samples */
  int           _sampleNr;
  /** Internal variable: main pointer to all allocated memory for the
      arrays x[], x1[], x2[], w[], and all y[]'s.
      These data arrays just point to this array.
      Allocated memory size is (4+_tacNr)*_sampleNr. */
  double       *_data;
  /** @brief Optional (but often useful) header information.
      @details Reserved keys:
       - studynr and study_number
       - timeunit and time_unit
       - unit and calibration_unit
       - voiname[index]
       - sizes
       - radiopharmaceutical
       - isotope
       - decay_correction and decay correction
       - injection_time and injection time
       - scan_start_time, scan start time, scan_start, scan start, and scantime
      @sa iftFree, iftFindKey, iftPut
   */
  IFT           h;
} TAC;
/*****************************************************************************/

/*****************************************************************************/
/** Definition for a list of TAC data structures. 
   @sa TAC, TACC, mtacInit, mtacFree, mtacAllocate, mtacAddTAC
*/
typedef struct MTAC {
  /** List of TAC data structures. */
  TAC *tac;
  /** Number of TAC structures in the list. */
  int nr;
  /** Allocated TAC list size, _nr >= nr. */
  int _nr;
} MTAC;
/*****************************************************************************/

/*****************************************************************************/
/* tac */
extern void tacInit(TAC *tac);
extern void taccInit(TACC *tacc);
extern void tacFree(TAC *tac);
extern void tacFreeExceptHeader(TAC *tac);
extern void taccFree(TACC *tacc);
extern int tacAllocate(TAC *tac, int sampleNr, int tacNr);
extern int tacAllocateMore(TAC *tac, int tacNr);
extern int tacCopyTacc(TACC *d1, TACC *d2, int sampleNr);
extern int tacCopyTaccdata(TACC *d1, TACC *d2, int sampleNr);
extern int tacCopyTacchdr(TACC *d1, TACC *d2);
extern int tacCopyHdr(TAC *tac1, TAC *tac2);
extern int tacIsSize(TAC *d);
extern int tacDuplicate(TAC *tac1, TAC *tac2);
extern int tacExtract(TAC *d1, TAC *d2, const int i);
extern int tacAllocateMoreSamples(TAC *tac, int addNr);
/*****************************************************************************/

/*****************************************************************************/
/* mtac */
extern void mtacInit(MTAC *mtac);
extern void mtacFree(MTAC *mtac);
extern int mtacAllocate(MTAC *mtac, int nr);
extern int mtacAllocateMore(MTAC *mtac, int nr);
extern int mtacAddTAC(MTAC *mtac, TAC *tac);
/*****************************************************************************/

/*****************************************************************************/
/* tacio */
extern char *tacFormattxt(tacformat c);
extern int tacFormatIdentify(const char *s);
extern char *tacDefaultExtension(tacformat c);
extern int tacFormatDetermine(const char *fname, TPCSTATUS *status);
extern int tacFormatWriteSupported(tacformat format);
extern int tacWrite(TAC *tac, FILE *fp, tacformat format, int extra, TPCSTATUS *status);
extern int tacRead(TAC *d, const char *fname, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* tacift */
extern int tacGetHeaderStudynr(IFT *h, char *s, TPCSTATUS *status);
extern int tacSetHeaderStudynr(IFT *h, const char *s);
extern int tacGetHeaderUnit(TAC *tac, TPCSTATUS *status);
extern int tacSetHeaderUnit(IFT *h, int u);
extern int tacGetHeaderTimeunit(TAC *tac, TPCSTATUS *status);
extern int tacSetHeaderTimeunit(IFT *h, int u);
extern int tacGetHeaderIsotope(IFT *h, char *s, TPCSTATUS *status);
extern int tacSetHeaderIsotope(IFT *h, const char *s);
extern int tacGetHeaderScanstarttime(IFT *h, char *s, TPCSTATUS *status);
extern int tacSetHeaderScanstarttime(IFT *h, const char *s);
extern int tacGetHeaderInjectiontime(IFT *h, char *s, TPCSTATUS *status);
extern int tacSetHeaderInjectiontime(IFT *h, const char *s);
extern decaycorrection tacGetHeaderDecayCorrection(IFT *h);
int tacSetHeaderDecayCorrection(IFT *h, decaycorrection dc);
/*****************************************************************************/

/*****************************************************************************/
/* simpleio */
extern int tacWriteSimple(TAC *tac, FILE *fp, int extra, TPCSTATUS *status);
extern int tacReadSimple(TAC *tac, CSV *csv, IFT *hdr, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* dftio */
extern int tacNameSplit(
  const char *rname, char *name1, char *name2, char *name3, unsigned int max_name_len
);
extern int tacWriteDFT(TAC *tac, FILE *fp, int extra, TPCSTATUS *status);
extern int tacReadDFT(TAC *tac, CSV *csv, IFT *hdr, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* sifio */
extern int tacWriteSIF(TAC *tac, FILE *fp, int extra, TPCSTATUS *status);
extern int tacReadSIF(TAC *tac, CSV *csv, IFT *hdr, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* pmodio */
extern int tacWritePMOD(TAC *tac, FILE *fp, int extra, TPCSTATUS *status);
extern int tacReadPMOD(TAC *tac, CSV *csv, IFT *hdr, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* qviewio */
extern int tacReadQView(TAC *tac, CSV *csv, const int grouponly, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* matio */
extern int tacReadMat(TAC *tac, CSV *csv, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* taccsv */
extern int tacWriteCSV(TAC *tac, FILE *fp, int extra, tacformat format, TPCSTATUS *status);
extern int tacReadCSV(TAC *tac, CSV *csv, IFT *hdr, TPCSTATUS *status);
extern int tacReadInveonCSV(TAC *tac, CSV *csv, TPCSTATUS *status);
extern int tacReadCarimasTxt(TAC *tac, CSV *csv, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* hcio */
extern int tacReadHRRTHC(TAC *tac, CSV *csv, TPCSTATUS *status);
extern int tacReadHRPLUSHC(TAC *tac, CSV *csv, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* tacnan */
extern int tacXNaNs(TAC *tac);
extern int tacYNaNs(TAC *tac, const int i);
extern int tacNaNs(TAC *tac);
extern int tacNotNaNs(TAC *tac, const int i);
extern int tacFixNaNs(TAC *tac);
extern int tacNthSample(TAC *tac, const int sn, const int i);
/*****************************************************************************/

/*****************************************************************************/
/* tacselect */
extern int tacSelectTACs(TAC *d, const char *region_name, int reset, TPCSTATUS *status);
extern int tacSelectedTACs(TAC *d);
extern int tacFirstSelected(TAC *d);
extern int tacSelectBestReference(TAC *d);
//int tacNameMatch(const char *tacname, const char *test_str, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* tacorder */
extern int tacVerifyTimeOrder(TAC *d, TPCSTATUS *status);
extern int tacSortByTime(TAC *d, TPCSTATUS *status);
extern int tacSortByConc(TAC *d, const int i, TPCSTATUS *status);
extern int tacSortByName(TAC *d, TPCSTATUS *status);
extern int tacSortByAUC(TAC *d, TPCSTATUS *status);
extern int tacMoveTACC(TAC *d, int from, int to);
extern int tacSwapTACCs(TAC *d, int i1, int i2);
extern int tacDeleteTACC(TAC *d, int i);
extern int tacMultipleSamples(TAC *d1, const int fixMode, TAC *d2, const int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* taccomp */
extern int tacCompareUnit(TAC *d1, TAC *d2, TPCSTATUS *status);
extern int tacCompareNames(TAC *d1, TAC *d2, const int i, TPCSTATUS *status);
extern int tacCompareConc(
  TAC *d1, TAC *d2, const int i, const double test_abs, const double test_rel,
  TPCSTATUS *status
);
extern int tacCompareTimes(
  TAC *d1, TAC *d2, const double test_abs, const double test_rel,
  TPCSTATUS *status
);
extern int tacCompareWeights(
  TAC *d1, TAC *d2, const double test_abs, const double test_rel,
  TPCSTATUS *status
);
/*****************************************************************************/

/*****************************************************************************/
/* tacunits */
extern int tacXUnitConvert(TAC *d, const int u, TPCSTATUS *status);
extern int tacYUnitConvert(TAC *d, const int u, TPCSTATUS *status);
extern int tacYUnitMass2Volume(TAC *tac, const double density, TPCSTATUS *status);
extern int tacYUnitVolume2Mass(TAC *tac, const double density, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* tacname */
extern int tacIndividualNames(TAC *tac);
extern void tacEnsureNames(TAC *tac);
/*****************************************************************************/

/*****************************************************************************/
/* tacx */
extern int tacXCopy(TAC *tac1, TAC *tac2, int i1, int i2);
extern int tacCorrectFrameOverlap(TAC *d, TPCSTATUS *status);
extern int tacXRange(TAC *d, double *xmin, double *xmax);
extern int tacSampleXRange(TAC *d, double *xmin, double *xmax);
extern int tacMinX(TAC *d);
extern int tacIsX(TAC *d);
extern int tacXMatch(TAC *d1, TAC *d2, const int verbose);
extern int tacAddZeroSample(TAC *d, TPCSTATUS *status);
extern int tacDeleteSample(TAC *d, int i);
extern int tacDeleteMissingSamples(TAC *d);
extern int tacExtractRange(TAC *d1, TAC *d2, double startT, double endT);
extern int tacExtractSamples(TAC *d1, TAC *d2, int si, int ei);
extern int tacCheckX1X2X(TAC *d);
extern int tacSetX(TAC *d, TPCSTATUS *status);
extern int tacGetSampleInterval(TAC *d, double ilimit, double *minfdur, double *maxfdur);
extern int tacToBars(TAC *tac1, TAC *tac2);
/*****************************************************************************/

/*****************************************************************************/
/* tacy */
extern int tacYRange(
  TAC *d, int i, double *ymin, double *ymax,
  int *smin, int *smax, int *imin, int *imax
);
extern int tacYRangeInXRange(
  TAC *d, int i, const double xmin, const double xmax, 
  double *ymin, double *ymax, int *smin, int *smax, int *imin, int *imax
);
/*****************************************************************************/

/*****************************************************************************/
/* tacw */
extern int tacIsWeighted(TAC *tac);
extern int tacWCopy(TAC *tac1, TAC *tac2, int i1, int i2);
extern int tacWMove(TAC *tac, int ow, TPCSTATUS *status);
extern int tacWByFreq(TAC *tac, isotope isot, TPCSTATUS *status);
extern unsigned int tacWSampleNr(TAC *tac);
extern int tacWeightNorm(TAC *tac, TPCSTATUS *status);
extern int tacWeightModerate(
  TAC *tac, const double minprop, const int doZeroes, const int doNaNs, TPCSTATUS *status
);
extern int sifWeight(TAC *sif, isotope isot, TPCSTATUS *status);
extern int tacSetWeights(TAC *tac, weights weightMethod, int weightNr, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* tacdc */
extern int tacGetIsotope(TAC *tac);
extern void tacSetIsotope(TAC *tac, int isotope);
extern int tacDecayCorrection(TAC *tac, int isotope, int mode, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* tacabss */
extern int tacReadAllogg(TAC *tac, IFT *hdr, TPCSTATUS *status);
extern int tacReadOldAllogg(TAC *tac, IFT *hdr, TPCSTATUS *status);
extern int tacReadScanditronics(TAC *tac, IFT *hdr, TPCSTATUS *status);
extern int tacReadGEMS(TAC *tac, IFT *hdr, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
/* tacxmlio */
extern int tacWriteXML(TAC *tac, FILE *fp, TPCSTATUS *status);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCTAC */
