/** @file litac.c
 *  @brief Linear interpolation of TACs.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcli.h"
#include "tpctac.h"
/*****************************************************************************/
#include "tpctacmod.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Interpolate and/or integrate TACs from one TAC struct into a new TAC struct, 
    using sample times from another TAC struct which is not changed.
    @details PET frame lengths of output TACs are taken into account in interpolation, if available.
    Input frame lengths are taken into account if the framing is same as with output TACs, 
    otherwise frame middle times are used.
    @sa tacInterpolateInto, tacAUC, tacFramesToSteps, tacXCopy, liInterpolate
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */ 
int tacInterpolate(
  /** Pointer to source TAC struct; make sure that x (time) and y (concentration) units are the same 
      as in output TAC struct, because input time range is verified to cover the output data range. 
      Detailed verification of suitability of TAC for interpolation must be done elsewhere.
      @pre Data must be sorted by increasing x. */
  TAC *inp,
  /** Pointer to TAC struct which contains the x values (sample times) for the
      interpolation and integration. Data in this TAC struct is not modified.
      @pre Data must be sorted by increasing x. */
  TAC *xinp,
  /** Pointer to target TAC struct into which the interpolated input TAC(s) are added.
      Struct must be initiated; any previous contents are deleted. Enter NULL if not needed. */
  TAC *tac,
  /** Pointer to target TAC struct into which the integrated input TAC(s) are added.
      Struct must be initiated; any previous contents are deleted. Enter NULL if not needed. */
  TAC *itac,
  /** Pointer to target TAC struct into which 2nd integrals of input TAC(s) are added.
      Struct must be initiated; any previous contents are deleted. Enter NULL if not needed. */
  TAC *iitac,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(inp==NULL || xinp==NULL || xinp==tac || xinp==itac || xinp==iitac) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(tac==NULL && itac==NULL && iitac==NULL) { // there's nothing to do
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return TPCERROR_OK;
  }
  /* Check the function input */
  if(inp->sampleNr<1 || inp->tacNr<1 || xinp->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Check that input does not have any missing values */
  if(tacNaNs(inp)>0 || tacXNaNs(xinp)>0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_MISSING_DATA);
    return TPCERROR_MISSING_DATA;
  } 


  /* If input data have similar x values as requested already, then
     copy x1 and x2 into input if available */
  if(xinp->isframe && tacXMatch(inp, xinp, verbose-2) &&
     inp->sampleNr<=xinp->sampleNr)
  {
    for(int i=0; i<inp->sampleNr; i++) {
      inp->x[i]=xinp->x[i]; inp->x1[i]=xinp->x1[i]; inp->x2[i]=xinp->x2[i];
    }
    inp->isframe=xinp->isframe;
  }


  /* Check that there is no need for excess extrapolation */
  {
    double start1, end1, start2, end2, range2;
    if(tacXRange(inp, &start1, &end1) || tacXRange(xinp, &start2, &end2)) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_X);
      return TPCERROR_INVALID_X;
    }
    range2=end2-start2;
    if(verbose>2) 
      printf("time ranges: %g - %g vs %g - %g\n", start1, end1, start2, end2);
    if(start1>start2+0.2*range2 || end1<end2-0.3*range2) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
      return TPCERROR_INVALID_XRANGE;
    }
  }

  /* Delete any previous contents in output structs */
  if(tac!=NULL) tacFree(tac);
  if(itac!=NULL) tacFree(itac);
  if(iitac!=NULL) tacFree(iitac);

  /* Allocate memory for output data */
  int ret=TPCERROR_OK;
  if(tac!=NULL) ret=tacAllocate(tac, xinp->sampleNr, inp->tacNr); 
  if(ret==TPCERROR_OK && itac!=NULL) 
    ret=tacAllocate(itac, xinp->sampleNr, inp->tacNr);
  if(ret==TPCERROR_OK && iitac!=NULL) 
    ret=tacAllocate(iitac, xinp->sampleNr, inp->tacNr);
  if(ret!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
    return TPCERROR_OUT_OF_MEMORY;
  }
  if(tac!=NULL) {tac->tacNr=inp->tacNr; tac->sampleNr=xinp->sampleNr;}
  if(itac!=NULL) {itac->tacNr=inp->tacNr; itac->sampleNr=xinp->sampleNr;}
  if(iitac!=NULL) {iitac->tacNr=inp->tacNr; iitac->sampleNr=xinp->sampleNr;}


  /* Copy header information */
  if(tac!=NULL) {
    tacCopyHdr(inp, tac);
    for(int i=0; i<inp->tacNr; i++) tacCopyTacchdr(inp->c+i, tac->c+i);
  }
  if(itac!=NULL) {
    tacCopyHdr(inp, itac);
    for(int i=0; i<inp->tacNr; i++) tacCopyTacchdr(inp->c+i, itac->c+i);
  }
  if(iitac!=NULL) {
    tacCopyHdr(inp, iitac);
    for(int i=0; i<inp->tacNr; i++) tacCopyTacchdr(inp->c+i, iitac->c+i);
  }
  /* Copy X (time) information */
  if(tac!=NULL) {
    tac->isframe=xinp->isframe; tacXCopy(xinp, tac, 0, tac->sampleNr-1);
  }
  if(itac!=NULL) {
    itac->isframe=xinp->isframe; tacXCopy(xinp, itac, 0, itac->sampleNr-1);
  }
  if(iitac!=NULL) {
    iitac->isframe=xinp->isframe; tacXCopy(xinp, iitac, 0, iitac->sampleNr-1);
  }

  /* Check if TACs have the same frame times already */
  if(tacXMatch(inp, tac, verbose-2) && inp->sampleNr>=xinp->sampleNr) {
    if(verbose>2) printf("same frame times\n");
    int ret=0, ri, fi;
    double *i1, *i2;
    /* copy the values directly and integrate in place, if requested */
    for(ri=0; ri<inp->tacNr && ret==0; ri++) {
      if(tac!=NULL) 
        for(fi=0; fi<tac->sampleNr; fi++)
          tac->c[ri].y[fi]=inp->c[ri].y[fi];
      /* If integrals not requested then that's it for this TAC */
      if(itac==NULL && iitac==NULL) continue;
      /* otherwise set pointers for integrals and then integrate */
      if(itac!=NULL) i1=itac->c[ri].y; else i1=NULL;      
      if(iitac!=NULL) i2=iitac->c[ri].y; else i2=NULL;
      if(inp->isframe)
        ret=liIntegratePET(inp->x1, inp->x2, inp->c[ri].y,
                           xinp->sampleNr, i1, i2, verbose-2);
      else
        ret=liInterpolate(inp->x, inp->c[ri].y, inp->sampleNr,
                          xinp->x, NULL, i1, i2, xinp->sampleNr, 4, 1, verbose-2);
    }
    if(ret) {
      if(tac!=NULL) tacFree(tac); 
      if(itac!=NULL) tacFree(itac);
      if(iitac!=NULL) tacFree(iitac);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
      return TPCERROR_INVALID_VALUE;
    }
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(TPCERROR_OK); // that's it then
  }

  /* Interpolate and/or integrate inp data to tac sample times,
     taking into account tac frame lengths if available */
  ret=0;
  double *i0, *i1, *i2;
  for(int ri=0; ri<inp->tacNr && ret==0; ri++) {
    /* set pointers for output y arrays */
    if(tac!=NULL) i0=tac->c[ri].y; else i0=NULL;      
    if(itac!=NULL) i1=itac->c[ri].y; else i1=NULL;      
    if(iitac!=NULL) i2=iitac->c[ri].y; else i2=NULL;
    if(xinp->isframe)
      ret=liInterpolateForPET(inp->x, inp->c[ri].y, inp->sampleNr,
                xinp->x1, xinp->x2, i0, i1, i2, xinp->sampleNr, 4, 1, verbose-2);
    else
      ret=liInterpolate(inp->x, inp->c[ri].y, inp->sampleNr,
                xinp->x, i0, i1, i2, xinp->sampleNr, 4, 1, verbose-2);
  }
  if(ret) {
    if(tac!=NULL) tacFree(tac); 
    if(itac!=NULL) tacFree(itac);
    if(iitac!=NULL) tacFree(iitac);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Add TACs from one TAC struct into another TAC struct, interpolating the input TACs and 
    allocating space if necessary.
    @details PET frame lengths of output TACs are taken into account in interpolation, if available.
    Input frame lengths are taken into account if the framing is same as with
    output TACs, otherwise frame middle times are used.
    @sa tacInterpolate, tacAUC, tacInterpolateToEqualLengthFrames, liInterpolateForPET
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */ 
int tacInterpolateInto(
  /** Pointer to source TAC struct; make sure that x (time) and y (concentration) units are the same 
      as in output TAC struct, because input time range is verified to cover the output data range.
      Detailed verification of suitability of TAC for interpolation must be done elsewhere.
      @pre Data must be sorted by increasing x. */
  TAC *inp,
  /** Pointer to target TAC struct into which the input TAC(s) are added.
      Struct must contain at least one TAC, and the x or x1 and x2 values.
      @pre Data must be sorted by increasing x. */
  TAC *tac,
  /** Pointer to target TAC struct into which integrated input TAC(s) are added;
      enter NULL if not needed. */
  TAC *itac,
  /** Pointer to target TAC struct into which 2nd integrals of input TAC(s) 
      are added; enter NULL if not needed. */
  TAC *iitac,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(tac==NULL || inp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  
  /* Check the function input */
  if(tac->sampleNr<1 || inp->sampleNr<1 || inp->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(itac!=NULL && itac->sampleNr!=tac->sampleNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_X);
    return TPCERROR_INVALID_X;
  }
  if(iitac!=NULL && iitac->sampleNr!=tac->sampleNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_X);
    return TPCERROR_INVALID_X;
  }


  /* Check that input does not have any missing values */
  /* Check that there are no missing sample times in output TACs */
  if(tacNaNs(inp)>0 || tacXNaNs(tac)>0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_MISSING_DATA);
    return TPCERROR_MISSING_DATA;
  } 
  
  /* Check that there is no need for excess extrapolation */
  {
    double start1, end1, start2, end2, range2;
    if(tacXRange(inp, &start1, &end1) || tacXRange(tac, &start2, &end2)) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_X);
      return TPCERROR_INVALID_X;
    }
    range2=end2-start2;
    if(verbose>2) 
      printf("time ranges: %g - %g vs %g - %g\n", start1, end1, start2, end2);
    if(start1>(start2+0.2*range2) || end1<(end2-0.3*range2)) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
      return TPCERROR_INVALID_XRANGE;
    }
  }

  /* Allocate memory for interpolated data */
  if(tacAllocateMore(tac, inp->tacNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
    return TPCERROR_OUT_OF_MEMORY;
  }
  /* Copy header information */
  for(int i=0; i<inp->tacNr; i++) tacCopyTacchdr(inp->c+i, tac->c+tac->tacNr+i);
  /* Same for the integrals */
  if(itac!=NULL) {
    if(tacAllocateMore(itac, inp->tacNr)) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
      return TPCERROR_OUT_OF_MEMORY;
    }
    for(int i=0; i<inp->tacNr; i++) 
      tacCopyTacchdr(inp->c+i, itac->c+itac->tacNr+i);
  }
  if(iitac!=NULL) {
    if(tacAllocateMore(iitac, inp->tacNr)) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
      return TPCERROR_OUT_OF_MEMORY;
    }
    for(int i=0; i<inp->tacNr; i++) tacCopyTacchdr(inp->c+i, iitac->c+iitac->tacNr+i);
  }

  /* Check if TACs have the same frame times already */
  if(tacXMatch(inp, tac, verbose-2) && inp->sampleNr>=tac->sampleNr) {
    if(verbose>2) printf("same frame times\n");
    int ret=0, ri, fi;
    double *i1, *i2;
    /* copy the values directly and integrate in place, if requested */
    for(ri=0; ri<inp->tacNr && ret==0; ri++) {
      for(fi=0; fi<tac->sampleNr; fi++)
        tac->c[tac->tacNr+ri].y[fi]=inp->c[ri].y[fi];
      /* If integrals not requested then that's it for this TAC */
      if(itac==NULL && iitac==NULL) continue;
      /* otherwise set pointers for integrals and then integrate */
      if(itac!=NULL) i1=itac->c[itac->tacNr+ri].y; else i1=NULL;      
      if(iitac!=NULL) i2=iitac->c[iitac->tacNr+ri].y; else i2=NULL;
      if(tac->isframe)
        ret=liIntegratePET(tac->x1, tac->x2, tac->c[tac->tacNr+ri].y,
                           tac->sampleNr, i1, i2, verbose-2);
      else
        ret=liInterpolate(tac->x, tac->c[tac->tacNr+ri].y, tac->sampleNr,
               tac->x, NULL, i1, i2, tac->sampleNr, 4, 1, verbose-2);
    }
    if(ret) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
      return TPCERROR_INVALID_VALUE;
    }
    tac->tacNr+=inp->tacNr;
    if(itac!=NULL) itac->tacNr+=inp->tacNr;
    if(iitac!=NULL) iitac->tacNr+=inp->tacNr;
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(TPCERROR_OK); // that's it then
  }

  /* Interpolate and, if requested, integrate inp data to tac sample times,
     taking into account tac frame lengths if available */
  int ret=0;
  double *i1, *i2;
  for(int ri=0; ri<inp->tacNr && ret==0; ri++) {
    /* set pointers for integrals */
    if(itac!=NULL) i1=itac->c[itac->tacNr+ri].y; else i1=NULL;
    if(iitac!=NULL) i2=iitac->c[iitac->tacNr+ri].y; else i2=NULL;
    if(tac->isframe)
      ret=liInterpolateForPET(inp->x, inp->c[ri].y, inp->sampleNr,
                tac->x1, tac->x2, tac->c[tac->tacNr+ri].y, 
                i1, i2, tac->sampleNr, 4, 1, verbose-2);
    else
      ret=liInterpolate(inp->x, inp->c[ri].y, inp->sampleNr,
                tac->x, tac->c[tac->tacNr+ri].y, 
                i1, i2, tac->sampleNr, 4, 1, verbose-2);
  }
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  tac->tacNr+=inp->tacNr;
  if(itac!=NULL) itac->tacNr+=inp->tacNr;
  if(iitac!=NULL) iitac->tacNr+=inp->tacNr;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Interpolate TAC data into data with equal frame lengths, starting from zero.
    @details By default, the shortest frame length or sampling interval in the input data is used.
    @remark This is needed for TAC convolution.
    @sa tacInterpolate, liInterpolateForPET, tacGetSampleInterval, simIsSteadyInterval
    @return enum tpcerror (TPCERROR_OK when successful).
*/
int tacInterpolateToEqualLengthFrames(
  /** Pointer to source TAC struct. Missing values are not allowed.
      @pre Data must be sorted by increasing x. */
  TAC *inp,
  /** Minimum frame length; if NaN or <=0 then not applied, otherwise frame lengths are not allowed 
      to be lower. */
  double minfdur,
  /** Maximum frame length; if NaN or <=0 then not applied, otherwise frame lengths are not allowed 
      to be higher. */
  double maxfdur,
  /** Pointer to target TAC struct into which the interpolated TACs are written.
      @pre Struct must be initiated. */
  TAC *tac,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(tac1, %g, %g, ...)\n", __func__, minfdur, maxfdur); fflush(stdout);}
  if(tac==NULL || inp==NULL || inp->sampleNr<1 || inp->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(minfdur>0.0 && maxfdur>0.0 && minfdur>maxfdur) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_X);
    return TPCERROR_INVALID_X;
  }
  if(!tacIsX(inp)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_X);
    return TPCERROR_INVALID_X;
  }

  /* Get the minimum sample interval (that is higher than 0) */
  double freq=nan("");
  int ret=tacGetSampleInterval(inp, 1.0E-08, &freq, NULL);
  //printf("dataset based freq: %g\n", freq);
  if(ret!=TPCERROR_OK) {statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);}
  if(minfdur>0.0 && freq<minfdur) freq=minfdur;
  if(maxfdur>0.0 && freq>maxfdur) freq=maxfdur;
  //printf("refined freq: %g\n", freq);

  /* Get the x range */
  double xmax=nan("");
  if(tacXRange(inp, NULL, &xmax)!=0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return TPCERROR_INVALID_XRANGE;
  }
  //printf("xmax: %g\n", xmax);
  if(freq>0.5*xmax) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return TPCERROR_INVALID_XRANGE;
  }


  /* Calculate the number of required interpolated samples */
  int nreq=ceil(xmax/freq); //printf("required_n := %d\n", nreq);
  /* Check that it is not totally stupid */
  if(nreq<1 || nreq>2E+06) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_TOO_BIG);
    return TPCERROR_TOO_BIG;
  }

  /* Setup the output TAC */
  if(tacAllocate(tac, nreq, inp->tacNr)!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
    return TPCERROR_OUT_OF_MEMORY;
  }
  tac->tacNr=inp->tacNr; tac->sampleNr=nreq;
  /* Copy header information */
  tacCopyHdr(inp, tac);
  for(int i=0; i<inp->tacNr; i++) tacCopyTacchdr(inp->c+i, tac->c+i);

  /* Set X (sample time) information */
  tac->isframe=1;
  for(int i=0; i<nreq; i++) tac->x1[i]=(double)i*freq;
  for(int i=0; i<nreq; i++) tac->x2[i]=tac->x1[i]+freq;
  for(int i=0; i<nreq; i++) tac->x[i]=0.5*(tac->x1[i]+tac->x2[i]);

  /* Calculate mean value during each interval */
  if(verbose>2) {printf("  interpolating...\n"); fflush(stdout);}
  for(int i=0; i<inp->tacNr; i++) {
    if(liInterpolateForPET(inp->x, inp->c[i].y, inp->sampleNr, tac->x1, tac->x2, 
                           tac->c[i].y, NULL, NULL, tac->sampleNr, 2, 1, 0)!=0) {
      tacFree(tac);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
      return TPCERROR_INVALID_XRANGE;
    }
  }
  if(verbose>2) {printf("  ... done.\n"); fflush(stdout);}

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Transform TAC with frames into TAC with frames represented with
           stepwise changing dot-to-dot data. 

    @details Gaps are replaced with lines connecting the corners of frame rectangles.
     Small frame overlaps are replaced by step at the midpoint of the overlap.
    @sa tacInterpolate, tacToBars, tacInterpolateToEqualLengthFrames, liInterpolate
    @return enum tpcerror (TPCERROR_OK when successful).
 */ 
int tacFramesToSteps(
  /** Pointer to source TAC struct.
      @pre Data must be sorted by increasing x. */
  TAC *inp,
  /** Pointer to the initiated target TAC struct; any previous contents are deleted. */
  TAC *out,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(inp==NULL || out==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  /* Check the function input */
  if(inp->sampleNr<1 || inp->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Check that input does not have any missing values */
  if(tacNaNs(inp)>0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_MISSING_DATA);
    return TPCERROR_MISSING_DATA;
  } 

  /* First, duplicate the TAC data */
  if(tacDuplicate(inp, out)!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }

  /* Make sure that frame start and end times are present */
  if(inp->isframe==0) return(TPCERROR_OK); // if not, consider that work is done

  /* Add room for more frames */
  if(tacAllocateMoreSamples(out, 1+2*out->sampleNr)!=0) {
    tacFree(out);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
    return TPCERROR_OUT_OF_MEMORY;
  }

  /* Make separate 'samples' for each frame start and end time */
  out->isframe=0;
  int fi=0, fj=0;
  if(fabs(inp->x1[fi])>1.0E-20) { // add zero sample, if data does not already have it
    out->x[fj]=inp->x1[fi];
    for(int ri=0; ri<out->tacNr; ri++) out->c[ri].y[fj]=0.0;
    fj++;
  }
  for(fi=0; fi<inp->sampleNr; fi++) {
    if(fi>0 && inp->x1[fi]<inp->x2[fi-1]) { // overlap
      double x=0.5*(inp->x1[fi]+inp->x2[fi-1]);
      if(x<out->x[fj-2]) { // too large overlap
        tacFree(out);
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OVERLAPPING_DATA);
        return TPCERROR_OVERLAPPING_DATA;
      }
      out->x[fj-1]=out->x[fj]=x;
    } else { // no overlap
      out->x[fj]=inp->x1[fi];
    }
    for(int ri=0; ri<out->tacNr; ri++) out->c[ri].y[fj]=inp->c[ri].y[fi];
    fj++;
    out->x[fj]=inp->x2[fi];
    for(int ri=0; ri<out->tacNr; ri++) out->c[ri].y[fj]=inp->c[ri].y[fi];
    fj++;
  }
  out->sampleNr=fj;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Move TAC y values (concentrations) in time, keeping sample times (x values) intact.

    @sa tacInterpolate, tacFramesToSteps, simDispersion, liInterpolate, tacVb
    @return enum tpcerror (TPCERROR_OK when successful).
 */ 
int tacDelay(
  /** Pointer to TAC structure.
      @pre Data must be sorted by increasing x.
      @post Data y values are modified, x values will stay the same. */
  TAC *tac,
  /** Time (in TAC tunit's) to move the TAC forward (negative time moves TAC backward). */
  double dt,
  /** Index [0..tacNr-1] of the TAC to move, or <0 to move all TACs. */
  int ti,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(tac, %g, %d)\n", __func__, dt, ti);
  /* Check the function input */
  if(tac==NULL || tac->sampleNr<1 || tac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(ti>=tac->tacNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(fabs(dt)<1.0E-12) {
    if(verbose>1) printf("  requested delay time is zero; nothing done.\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(TPCERROR_OK);
  }
  /* Check that input does not have any missing values */
  if(tacXNaNs(tac)>0 || tacYNaNs(tac, ti)>0) {
    if(verbose>1) printf("  data contains NaN(s).\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_MISSING_DATA);
    return TPCERROR_MISSING_DATA;
  } 

  /* Get and check the x range */
  double xmin=nan(""), xmax=nan("");
  if(tacXRange(tac, &xmin, &xmax)!=0 || dt>=xmax || dt<=-xmax) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return TPCERROR_INVALID_XRANGE;
  }
  if(verbose>2) printf("  xrange: %g - %g\n", xmin, xmax);


  /* If TAC contains just frame mid times, then simple dot-to-dot interpolation */
  if(tac->isframe==0) {
    /* Make temporary x array */
    double tx[tac->sampleNr];
    for(int i=0; i<tac->sampleNr; i++) tx[i]=tac->x[i]+dt;
    /* One TAC at a time */
    for(int j=0; j<tac->tacNr; j++) if(ti<0 || ti==j) {
      /* Make temporary y array */
      double ty[tac->sampleNr];
      for(int i=0; i<tac->sampleNr; i++) ty[i]=tac->c[j].y[i];
      /* Interpolate temporary data to original times */
      if(liInterpolate(tx, ty, tac->sampleNr, tac->x, tac->c[j].y, NULL, NULL, tac->sampleNr, 0, 1, 
                       verbose-10)!=0)
      {
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
        return TPCERROR_INVALID_VALUE;
      }
    }
    /* That's it then */
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(TPCERROR_OK);
  }

  /* We have frame start and end times */
  /* Convert the data into stepwise dot-to-dot data */
  TAC stac; tacInit(&stac);
  {
    int ret=tacFramesToSteps(tac, &stac, NULL);
    if(ret!=TPCERROR_OK) {
      tacFree(&stac);
      statusSet(status, __func__, __FILE__, __LINE__, ret);
      return(ret);
    }
  }
  /* Change times of the stepwise data */
  for(int i=0; i<stac.sampleNr; i++) stac.x[i]+=dt;
  /* Interpolate stepwise data into original frame times, one TAC at a time */
  for(int j=0; j<tac->tacNr; j++) if(ti<0 || ti==j) {
    if(liInterpolateForPET(stac.x, stac.c[j].y, stac.sampleNr, tac->x1, tac->x2, tac->c[j].y, 
                           NULL, NULL, tac->sampleNr, 0, 1, verbose-10))
    {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
      return TPCERROR_INVALID_VALUE;
    }
  }
  tacFree(&stac);


  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Calculates TAC AUC from t1 to t2.

    @sa tacInterpolate, liInterpolate, liIntegrate
    @return AUC, or NaN in case of an error.
 */ 
double tacAUC(
  /** Pointer to TAC structure.
      @pre Data must be sorted by increasing x. */
  TAC *tac,
  /** Index [0..tacNr-1] of the TAC to calculate AUC from. */
  int ti,
  /** AUC start time. */
  double t1,
  /** AUC end time. */
  double t2,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(tac, %d, %g, %g)\n", __func__, ti, t1, t2);
  /* Check the function input */
  if(tac==NULL || tac->sampleNr<1 || tac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(nan(""));
  }
  if(ti<0 || ti>=tac->tacNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return(nan(""));
  }
  if(t1>=t2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return(nan(""));
  }

  /* Get and check the x range */
  double xmin=nan(""), xmax=nan("");
  if(tacXRange(tac, &xmin, &xmax)!=0 || t1>=xmax || t2<=xmin) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return(nan(""));
  }

  /* Make temporary x and yi array */
  double tx[2]; tx[0]=t1; tx[1]=t2;
  double tyi[2];

  /* If TAC contains just frame mid times, then simple dot-to-dot integration */
  if(tac->isframe==0) {
    if(liInterpolate(tac->x, tac->c[ti].y, tac->sampleNr, tx, NULL, tyi, NULL, 2, 3, 1, verbose-10))
    {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
      return(nan(""));
    }
    /* That's it then */
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(tyi[1]-tyi[0]);
  }

  /* We have frame start and end times */
  /* Convert the data into stepwise dot-to-dot data */
  TAC stac; tacInit(&stac);
  {
    int ret=tacFramesToSteps(tac, &stac, NULL);
    if(ret!=TPCERROR_OK) {
      tacFree(&stac);
      statusSet(status, __func__, __FILE__, __LINE__, ret);
      return(nan(""));
    }
  }
  /* Then simple dot-to-dot integration */
  if(liInterpolate(stac.x, stac.c[ti].y, stac.sampleNr, tx, NULL, tyi, NULL, 2, 3, 1, verbose-10))
  {
    tacFree(&stac);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return(nan(""));
  }
  tacFree(&stac);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(tyi[1]-tyi[0]);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Correct TTACs for vascular blood, or simulate its effect.
    @sa tacDelay, tacReadModelingInput, tacInterpolate
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int tacVb(
  /** Pointer to TAC data to process. */
  TAC *ttac,
  /** Index of TAC to be processed; enter <0 to process all. */
  const int i,
  /** Pointer to BTAC data to subtract or add; must contain exactly same sample times as TTAC
      and y values must be in the same units. */
  TAC *btac,
  /** Vb fraction [0,1]. */
  double Vb,
  /** Switch to either subtract vascular volume (0) or to simulate it (1). */
  const int simVb,
  /** Switch to model vascular volume as either
      0 : Cpet = (1-Vb)*Ct + Vb*Cb, or
      1 : Cpet = Ct + Vb*Cb */
  const int petVolume,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(ttac, %d, btac, %g, %d, %d)\n", __func__, i, Vb, simVb, petVolume);
  if(ttac==NULL || btac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(ttac->tacNr<1 || btac->tacNr<1 || ttac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(ttac->sampleNr>btac->sampleNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INCOMPATIBLE_DATA);
    return TPCERROR_INCOMPATIBLE_DATA;
  }
  if(!(Vb>=0.0) || !(Vb<=1.0)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  if(Vb==1.0 && petVolume==0 && simVb==0) { // combination would lead to divide by zero
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }
  if(i>=ttac->tacNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }


  if(simVb==0) {
    if(verbose>1) printf("subtracting blood\n");
    for(int ri=0; ri<ttac->tacNr; ri++) if(i<0 || i==ri) {
      for(int fi=0; fi<ttac->sampleNr; fi++) {
        ttac->c[ri].y[fi]-=Vb*btac->c[0].y[fi];
        if(petVolume==0) ttac->c[ri].y[fi]/=(1.0-Vb);
      }
    }
  } else {
    if(verbose>1) printf("adding blood\n");
    for(int ri=0; ri<ttac->tacNr; ri++) if(i<0 || i==ri) {
      for(int fi=0; fi<ttac->sampleNr; fi++) {
        if(petVolume==0) ttac->c[ri].y[fi]*=(1.0-Vb);
        ttac->c[ri].y[fi]+=Vb*btac->c[0].y[fi];
      }
    }
  }

  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
