/// @file tacfitplot.c
/// @brief Plot measured and fitted TACs in SVG format.
/// @author Vesa Oikonen
///
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
#include "libtpcsvg.h"
/*****************************************************************************/
#include "tpctacmod.h"
/*****************************************************************************/

/*****************************************************************************/
/** Writes specified range of plots of original and fitted TACs in SVG format.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa tacPlotLineSVG, mtgaPlotSVG
    @author Vesa Oikonen
 */
int tacPlotFitSVG(
  /** Measured data points. */
  TAC *tac1,
  /** Fitted data points. Times can be different but unit must be the same as in tac1. */
  TAC *tac2,
  /** String for plot main title, or "". */
  const char *main_title,
  /** Start time; NaN if determined from data. */
  const double x1,
  /** End time; NaN if determined from data. */
  const double x2,
  /** Minimum y value; NaN if determined from data. */
  const double y1,
  /** Maximum y value; NaN if determined from data. */
  const double y2,
  /** SVG file name. */ 
  const char *fname,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  /* Check data */
  if(tac1==NULL || tac2==NULL || tac1->tacNr<1 || tac2->tacNr<1 || 
     tac1->sampleNr<1 || tac2->sampleNr<1) 
  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(fname==NULL || strnlen(fname, 1)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FILENAME);
    return TPCERROR_INVALID_FILENAME;
  }
  if(tac1->tacNr!=tac2->tacNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INCOMPATIBLE_DATA);
    return TPCERROR_INCOMPATIBLE_DATA;
  }
  if(status!=NULL && status->forgiving==0) {
    // any strict tests in here
  }

  int ret, n, ri, si, ei;
  char x_title[64], y_title[64], tac_id[32], tac_title[64];
  double minx, maxx, miny, maxy, tx1, tx2, ty1, ty2, f;
  struct svg_viewports viewports; svg_init_viewports(&viewports);
  int max_color_nr=0, color_nr=0;
  int max_symbol_nr=0, symbol_nr=0;
  SVG_LEGENDS legends; svg_init_legends(&legends);
  FILE *fp_svg=NULL;

  int is_label=0; if(tac1->tacNr>1) is_label=1;

  /* Determine the plot min and max x values */
  if(tacSampleXRange(tac1, &tx1, &tx2)) {
    if(verbose>0) {printf("  tac1 failed\n"); fflush(stdout);}
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return TPCERROR_INVALID_XRANGE;
  }
  minx=tx1; maxx=tx2;
  if(tacSampleXRange(tac2, &tx1, &tx2)) {
    if(verbose>0) {printf("  tac2 failed\n"); fflush(stdout);}
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return TPCERROR_INVALID_XRANGE;
  }
  if(minx>tx1) minx=tx1; 
  if(maxx<tx2) maxx=tx2;
  if(minx>0.0) {double f=maxx-minx; minx-=0.05*f; if(minx<0.0) minx=0.0;}
  if(!isnan(x1)) minx=x1; 
  if(!isnan(x2)) maxx=x2;
  if(verbose>1) printf("  minx := %g\n  maxx:=%g\n", minx, maxx);

  /* Determine the plot min and max y values inside the x range */
  if(tacYRangeInXRange(tac1, -1, minx, maxx, &ty1, &ty2, NULL, NULL, NULL, NULL)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return TPCERROR_INVALID_XRANGE;
  }
  miny=ty1; maxy=ty2;
  if(tacYRangeInXRange(tac2, -1, minx, maxx, &ty1, &ty2, NULL, NULL, NULL, NULL)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return TPCERROR_INVALID_XRANGE;
  }
  if(miny>ty1) miny=ty1; 
  if(maxy<ty2) maxy=ty2;
  if(miny>0.0) {f=maxy-miny; miny-=0.05*f; if(miny<0.0) miny=0.0;}
  if(!isnan(y1)) miny=y1; 
  if(!isnan(y2)) maxy=y2;
  if(verbose>1) printf("  miny := %g\n  maxy:=%g\n", miny, maxy);

  /* Calculate the axis ticks */
  viewports.x.min=minx; viewports.x.max=maxx;
  viewports.y.min=miny; viewports.y.max=maxy;
  viewports.label_area_viewport.is=is_label; // needed for x axis ticks
  ret=svg_calculate_axes(&viewports, verbose-3); 
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Set x and y axis titles based on activity and time units */
  strcpy(x_title, "");
  if(unitIsTime(tac1->tunit)) 
    sprintf(x_title, "Time (%s)", unitName(tac1->tunit));
  else if(tac1->tunit!=UNIT_UNKNOWN)
    strcpy(x_title, unitName(tac1->tunit));
  strcpy(y_title, "");
  if(tac1->cunit!=UNIT_UNKNOWN) strcpy(y_title, unitName(tac1->cunit));

  /* Set the plot window and window area sizes */
  ret=svg_define_viewports(0, 0, strlen(main_title), strlen(y_title),
    strlen(x_title), is_label, &viewports, verbose-3);
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Initiate graphics file */
  fp_svg=svg_initiate(fname, 0, 0, &viewports, NULL, verbose-3);
  if(fp_svg==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Put the graph titles into their own viewports */
  ret=svg_create_main_title(fp_svg, main_title, "", &viewports, NULL,verbose-3);
  if(!ret) 
    ret=svg_create_yaxis_title(fp_svg, y_title, &viewports, NULL, verbose-3);
  if(!ret) 
    ret=svg_create_xaxis_title(fp_svg, x_title, &viewports, NULL, verbose-3);

  /* Put the plot into its own viewport */
  if(!ret) 
    ret=svg_start_plot_viewport(fp_svg, &viewports, NULL, verbose-3);

  /*  Start coordinate area viewport */
  if(!ret) 
    ret=svg_start_coordinate_viewport(fp_svg, &viewports, NULL, verbose-3);

  /* Write plot axes */
  if(!ret) ret=svg_write_axes(fp_svg, &viewports, NULL, verbose-3);

  if(ret) {
    fclose(fp_svg);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /*
   *  Draw the plots
   */
  max_color_nr=0; while(svgColorName(max_color_nr)!=NULL) max_color_nr++;
  if(verbose>3) printf("max_color_nr := %d\n", max_color_nr);
  max_symbol_nr=0; while(svgSymbolName(max_symbol_nr)!=NULL) max_symbol_nr++;
  if(verbose>3) printf("max_symbol_nr := %d\n", max_symbol_nr);
  if(tac1->tacNr==1) color_nr=0; else color_nr=1;
  symbol_nr=0;
  for(ri=0, n=0; ri<tac1->tacNr; ri++) {
    sprintf(tac_id, "plot_%d", n);
    strcpy(tac_title, tac1->c[ri].name);
    /* Draw the fitted line */
    for(si=0; si<tac2->sampleNr; si++) if(!isnan(tac2->c[ri].y[si])) break;
    for(ei=tac2->sampleNr-1; ei>=si; ei--) if(!isnan(tac2->c[ri].y[ei])) break;
    if((ei-si)>0) {
      ret=svg_write_tac(fp_svg, &viewports, 1, tac_id, tac_title,
            tac2->x+si, tac2->c[ri].y+si, 1+ei-si,
            svgColorName(color_nr%max_color_nr), symbol_nr%max_symbol_nr, SYMBOLOPEN,
            NULL, verbose-3);
      if(ret) {
        svg_legend_empty(&legends); fclose(fp_svg);
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
        return TPCERROR_INVALID_VALUE;
      }
    }
    /* Draw the measured points */
    ret=svg_write_tac(fp_svg, &viewports, 2, tac_id, tac_title,
            tac1->x, tac1->c[ri].y, tac1->sampleNr,
            svgColorName(color_nr%max_color_nr), symbol_nr%max_symbol_nr, SYMBOLFILLED,
            NULL, verbose-3);
    if(ret) {
      svg_legend_empty(&legends); fclose(fp_svg);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
      return TPCERROR_INVALID_VALUE;
    }
    /* Set legend too, if requested */
    if(is_label!=0) {
      int si=0, ci=0; // remainder works only if 2nd operator > 0
      if(max_symbol_nr>0) si=symbol_nr%max_symbol_nr;
      if(max_color_nr>0) ci=color_nr%max_color_nr;

      svg_legend_add(&legends, 0, si, SYMBOLFILLED, ci, tac_title);
    }
    /* Prepare for the next plot */
    color_nr++; n++;
    if(color_nr==max_color_nr) {symbol_nr++; color_nr=0;}
    if(symbol_nr==max_symbol_nr) symbol_nr=0;
  }

  /* Close the coordinate viewport */
  ret=svg_end_coordinate_viewport(fp_svg, NULL, verbose-3);

  /* Write the axis ticks */
  if(!ret) {
    if(svg_write_xticks(fp_svg, &viewports, NULL, verbose-3)!=0) ret=1;}
  if(!ret) {
    if(svg_write_yticks(fp_svg, &viewports, NULL, verbose-3)!=0) ret=1;}

  /* Close the plot viewport */
  if(!ret) ret=svg_end_plot_viewport(fp_svg, NULL, verbose-3);

  if(ret) {
    fclose(fp_svg); svg_legend_empty(&legends);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Make the plot legends into their own viewport */
  ret=0;
  if(viewports.label_area_viewport.is!=0) {
    if(verbose>2) printf("creating plot legends\n");
    ret=svg_create_legends(fp_svg, &viewports, &legends, NULL, verbose-3);
  }
  svg_legend_empty(&legends);
  if(ret) {
    fclose(fp_svg);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Close the SVG file */
  ret=svg_close(fp_svg, NULL, verbose-3);
  if(ret) {
    fclose(fp_svg);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Writes specified range of line plots of TACs in SVG format.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa tacPlotFitSVG, mtgaPlotSVG
    @author Vesa Oikonen
 */
int tacPlotLineSVG(
  /** TAC data points. */
  TAC *tac,
  /** String for plot main title, or "". */
  const char *main_title,
  /** Start time; NaN if determined from data. */
  const double x1,
  /** End time; NaN if determined from data. */
  const double x2,
  /** Minimum y value; NaN if determined from data. */
  const double y1,
  /** Maximum y value; NaN if determined from data. */
  const double y2,
  /** SVG file name. */ 
  const char *fname,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}
  /* Check data */
  if(tac==NULL || tac->tacNr<1 || tac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(fname==NULL || strnlen(fname, 1)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FILENAME);
    return TPCERROR_INVALID_FILENAME;
  }
  if(status!=NULL && status->forgiving==0) {
    // any strict tests in here
  }

  int ret, n, ri, si, ei;
  char x_title[64], y_title[64], tac_id[32], tac_title[64];
  double minx, maxx, miny, maxy, tx1, tx2, ty1, ty2, f;
  struct svg_viewports viewports; svg_init_viewports(&viewports);
  unsigned int max_color_nr=0, color_nr=0;
  unsigned int max_symbol_nr=0, symbol_nr=0;
  SVG_LEGENDS legends; svg_init_legends(&legends);
  FILE *fp_svg=NULL;


  int is_label=0; if(tac->tacNr>1) is_label=1;

  /* Determine the plot min and max x values */
  if(tacSampleXRange(tac, &tx1, &tx2)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return TPCERROR_INVALID_XRANGE;
  }
  minx=tx1; maxx=tx2;
  if(minx>0.0) {double f=maxx-minx; minx-=0.05*f; if(minx<0.0) minx=0.0;}
  if(!isnan(x1)) minx=x1; 
  if(!isnan(x2)) maxx=x2;
  if(verbose>1) {
    printf("  minx := %g\n  maxx:=%g\n", minx, maxx); fflush(stdout);}

  /* Determine the plot min and max y values inside the x range */
  if(tacYRangeInXRange(tac, -1, minx, maxx, &ty1, &ty2, NULL, NULL, NULL, NULL)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_XRANGE);
    return TPCERROR_INVALID_XRANGE;
  }
  miny=ty1; maxy=ty2;
  if(miny>0.0) {f=maxy-miny; miny-=0.05*f; if(miny<0.0) miny=0.0;}
  if(!isnan(y1)) miny=y1; 
  if(!isnan(y2)) maxy=y2;
  if(verbose>1) {
    printf("  miny := %g\n  maxy := %g\n", miny, maxy); fflush(stdout);}

  /* Calculate the axis ticks */
  viewports.x.min=minx; viewports.x.max=maxx;
  viewports.y.min=miny; viewports.y.max=maxy;
  viewports.label_area_viewport.is=is_label; // needed for x axis ticks
  ret=svg_calculate_axes(&viewports, verbose-3); 
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Set x and y axis titles based on activity and time units */
  strcpy(x_title, "");
  if(unitIsTime(tac->tunit)) 
    sprintf(x_title, "Time (%s)", unitName(tac->tunit));
  else if(tac->tunit!=UNIT_UNKNOWN)
    strcpy(x_title, unitName(tac->tunit));
  strcpy(y_title, "");
  if(tac->cunit!=UNIT_UNKNOWN) strcpy(y_title, unitName(tac->cunit));

  /* Set the plot window and window area sizes */
  ret=svg_define_viewports(0, 0, strlen(main_title), strlen(y_title),
    strlen(x_title), is_label, &viewports, verbose-3);
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Initiate graphics file */
  fp_svg=svg_initiate(fname, 0, 0, &viewports, NULL, verbose-3);
  if(fp_svg==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Put the graph titles into their own viewports */
  ret=svg_create_main_title(fp_svg, main_title, "", &viewports, NULL,verbose-3);
  if(!ret) 
    ret=svg_create_yaxis_title(fp_svg, y_title, &viewports, NULL, verbose-3);
  if(!ret) 
    ret=svg_create_xaxis_title(fp_svg, x_title, &viewports, NULL, verbose-3);

  /* Put the plot into its own viewport */
  if(!ret) 
    ret=svg_start_plot_viewport(fp_svg, &viewports, NULL, verbose-3);

  /*  Start coordinate area viewport */
  if(!ret) 
    ret=svg_start_coordinate_viewport(fp_svg, &viewports, NULL, verbose-3);

  /* Write plot axes */
  if(!ret) ret=svg_write_axes(fp_svg, &viewports, NULL, verbose-3);

  if(ret) {
    fclose(fp_svg);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /*
   *  Draw the plots
   */
  max_color_nr=0; while(svgColorName(max_color_nr)!=NULL) max_color_nr++;
  if(verbose>3) {printf("max_color_nr := %d\n", max_color_nr); fflush(stdout);}
  if(tac->tacNr==1) color_nr=0; else color_nr=1;
  symbol_nr=0;
  for(ri=0, n=0; ri<tac->tacNr; ri++) {
    sprintf(tac_id, "plot_%d", n);
    strcpy(tac_title, tac->c[ri].name);
    /* Draw the line */
    for(si=0; si<tac->sampleNr; si++) if(!isnan(tac->c[ri].y[si])) break;
    for(ei=tac->sampleNr-1; ei>=si; ei--) if(!isnan(tac->c[ri].y[ei])) break;
    if((ei-si)>0) {
      unsigned int ci, si; // remainder works only if 2nd operator > 0
      if(max_color_nr<1) ci=0; else ci=color_nr % max_color_nr;
      if(max_symbol_nr<1) si=0; else si=symbol_nr % max_symbol_nr;
      ret=svg_write_tac(fp_svg, &viewports, 1, tac_id, tac_title,
            tac->x+si, tac->c[ri].y+si, 1+ei-si,
            svgColorName(ci), si, SYMBOLFILLED,
            NULL, verbose-3);
      if(ret) {
        svg_legend_empty(&legends); fclose(fp_svg);
        statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
        return TPCERROR_INVALID_VALUE;
      }
    }
    /* Set legend too, if requested */
    if(is_label!=0) {
      unsigned int ci, si; // remainder works only if 2nd operator > 0
      if(max_color_nr<1) ci=0; else ci=color_nr % max_color_nr;
      if(max_symbol_nr<1) si=0; else si=symbol_nr % max_symbol_nr;
      svg_legend_add(&legends, 0, si, SYMBOLFILLED, ci, tac_title);
    }
    /* Prepare for the next plot */
    color_nr++; n++;
    if(color_nr==max_color_nr) color_nr=0;
  }

  /* Close the coordinate viewport */
  ret=svg_end_coordinate_viewport(fp_svg, NULL, verbose-3);

  /* Write the axis ticks */
  if(!ret) {
    if(svg_write_xticks(fp_svg, &viewports, NULL, verbose-3)!=0) ret=1;}
  if(!ret) {
    if(svg_write_yticks(fp_svg, &viewports, NULL, verbose-3)!=0) ret=1;}

  /* Close the plot viewport */
  if(!ret) ret=svg_end_plot_viewport(fp_svg, NULL, verbose-3);

  if(ret) {
    fclose(fp_svg); svg_legend_empty(&legends);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Make the plot legends into their own viewport */
  ret=0;
  if(viewports.label_area_viewport.is!=0) {
    if(verbose>2) {printf("creating plot legends\n"); fflush(stdout);}
    ret=svg_create_legends(fp_svg, &viewports, &legends, NULL, verbose-3);
  }
  svg_legend_empty(&legends);
  if(ret) {
    fclose(fp_svg);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Close the SVG file */
  ret=svg_close(fp_svg, NULL, verbose-3);
  if(ret) {
    fclose(fp_svg);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** MTGA plots in SVG format.
    @return enum tpcerror (TPCERROR_OK when successful).
    @sa tacPlotLineSVG, tacPlotFitSVG
    @author Vesa Oikonen
 */
int mtgaPlotSVG(
  /** Pointer to MTAC structure, which contains plot data of tissue regions, 
      each in its own TAC structure. TAC must contain the plot x axis values,
      and three y columns: points included in regression, points not included,
      and points representing the fitted line. */
  MTAC *mtac,
  /** String for plot main title, or "". */
  const char *main_title,
  /** SVG file name. */ 
  const char *fname,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(MTAC, '%s', '%s')\n", __func__, main_title, fname); fflush(stdout);}
  /* Check data */
  if(mtac==NULL || mtac->nr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  if(fname==NULL || strnlen(fname, 1)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FILENAME);
    return TPCERROR_INVALID_FILENAME;
  }
  if(status!=NULL && status->forgiving==0) {
    // any strict tests in here
  }
  if(verbose>1) {printf("%d plot(s)\n", mtac->nr); fflush(stdout);}

  /* Do we show plot legends? */
  int is_label=0; if(mtac->nr>1) is_label=1;

  /* Get the plot x and y range */
  double minx=nan(""), maxx=nan(""), miny=nan(""), maxy=nan(""), maxly=nan("");
  for(int i=0; i<mtac->nr; i++) {
    if(mtac->tac[i].tacNr<3) continue;
    double x1, x2, y1, y2;
    if(tacSampleXRange(&mtac->tac[i], &x1, &x2)!=0) continue;
    if(isfinite(x1) && !(x1>minx)) minx=x1;
    if(isfinite(x2) && !(x2<maxx)) maxx=x2;
    if(tacYRange(mtac->tac+i, -1, &y1, &y2, NULL, NULL, NULL, NULL)!=0) continue;
    if(isfinite(y1) && !(y1>miny)) miny=y1;
    if(isfinite(y2) && !(y2<maxy)) maxy=y2;
    /* Get the y maximum from fitted line alone */
    if(tacYRange(mtac->tac+i, 2, &y1, &y2, NULL, NULL, NULL, NULL)!=0) continue;
    if(isfinite(y2) && !(y2<maxly)) maxly=y2;
  }
  if(verbose>1) {
    printf("x-range := %g - %g\n", minx, maxx);
    printf("y-range := %g - %g (%g for the line)\n", miny, maxy, maxly);
  }
  if(!(maxx>minx) || !(maxy>miny)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }
  /* Limit the y range to maximal line value plus 1/40 of y range */
  {
    double yrange=maxy-miny;
    double ylim=maxly+0.025*yrange;
    if(maxy>ylim) maxy=ylim;
    if(verbose>2) printf("  yrange := %g\n  ylim := %g\n  maxy := %g\n", yrange, ylim, maxy);
  }

  /* Start setting viewports */
  struct svg_viewports viewports; svg_init_viewports(&viewports);
  viewports.x.min=minx; viewports.x.max=maxx;
  viewports.y.min=miny; viewports.y.max=maxy;
  viewports.label_area_viewport.is=is_label; // needed for x axis ticks

  /* Calculate the axis ticks */
  if(svg_calculate_axes(&viewports, verbose-3)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Set x and y axis titles based on activity and time units */
  char x_title[64], y_title[64];
  strcpy(x_title, ""); strcpy(y_title, "");

  /* Set the plot window and window area sizes; ticks should be set before this */
  if(svg_define_viewports(0, 0, strlen(main_title), strlen(y_title),
    strlen(x_title), is_label, &viewports, verbose-3)) 
  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    return TPCERROR_INVALID_VALUE;
  }

  /* Initiate graphics file */
  FILE *fp=svg_initiate(fname, 0, 0, &viewports, NULL, verbose-3);
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }


  /* Put the graph titles into their own viewports */
  if(svg_create_main_title(fp, main_title, "", &viewports, NULL,verbose-3)) {
    fclose(fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(svg_create_yaxis_title(fp, y_title, &viewports, NULL, verbose-3)) {
    fclose(fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(svg_create_xaxis_title(fp, x_title, &viewports, NULL, verbose-3)) {
    fclose(fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Put the plot into its own viewport */
  if(svg_start_plot_viewport(fp, &viewports, NULL, verbose-3)) {
    fclose(fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /*  Start coordinate area viewport */
  if(svg_start_coordinate_viewport(fp, &viewports, NULL, verbose-3)) {
    fclose(fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Write plot axes */
  if(svg_write_axes(fp, &viewports, NULL, verbose-3)) {
    fclose(fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }


  /*
   *  Draw the plots
   */
  SVG_LEGENDS legends; svg_init_legends(&legends);

  int max_color_nr=0; while(svgColorName(max_color_nr)!=NULL) max_color_nr++;
  if(verbose>3) {printf("max_color_nr := %d\n", max_color_nr); fflush(stdout);}
  int color_nr; if(mtac->nr==1) color_nr=0; else color_nr=1;
  int max_symbol_nr=0; while(svgSymbolName(max_symbol_nr)!=NULL) max_symbol_nr++;
  if(verbose>3) printf("max_symbol_nr := %d\n", max_symbol_nr);
  int n=0, symbol_nr=0;
  for(int ri=0; ri<mtac->nr; ri++) {
    if(mtac->tac[ri].tacNr<3) continue;
    char tac_id[32], tac_title[64];
    sprintf(tac_id, "plot_%d", n);
    strcpy(tac_title, mtac->tac[ri].c[0].name);


    /* Draw the measured points */
    int ret=svg_write_tac(fp, &viewports, 2, tac_id, tac_title,
            mtac->tac[ri].x, mtac->tac[ri].c[1].y, mtac->tac[ri].sampleNr,
            svgColorName(color_nr%max_color_nr), symbol_nr%max_symbol_nr, SYMBOLOPEN,
            NULL, verbose-3);
    if(!ret) ret=svg_write_tac(fp, &viewports, 2, tac_id, tac_title,
            mtac->tac[ri].x, mtac->tac[ri].c[0].y, mtac->tac[ri].sampleNr,
            svgColorName(color_nr%max_color_nr), symbol_nr%max_symbol_nr, SYMBOLFILLED,
            NULL, verbose-3);
    /* Draw the regression line */
    if(!ret) ret=svg_write_tac(fp, &viewports, 1, tac_id, tac_title,
            mtac->tac[ri].x, mtac->tac[ri].c[2].y, mtac->tac[ri].sampleNr,
            svgColorName(color_nr%max_color_nr), symbol_nr%max_symbol_nr, SYMBOLOPEN,
            NULL, verbose-3);
    if(ret) {
      svg_legend_empty(&legends); fclose(fp);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
      return TPCERROR_INVALID_VALUE;
    }


    /* Set legend too, if requested */
    if(is_label!=0) {
      unsigned int ci, si; // remainder works only if 2nd operator > 0
      if(max_color_nr<1) ci=0; else ci=color_nr % max_color_nr;
      if(max_symbol_nr<1) si=0; else si=symbol_nr % max_symbol_nr;
      svg_legend_add(&legends, 0, si, SYMBOLFILLED, ci, tac_title);
    }
    /* Prepare for the next plot */
    color_nr++; n++;
    if(color_nr==max_color_nr) color_nr=0;
  }



  /* Close the coordinate viewport */
  if(svg_end_coordinate_viewport(fp, NULL, verbose-3)) {
    fclose(fp); svg_legend_empty(&legends);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Write the axis ticks */
  if(svg_write_xticks(fp, &viewports, NULL, verbose-3)!=0 ||
     svg_write_yticks(fp, &viewports, NULL, verbose-3)!=0) 
  {
    fclose(fp); svg_legend_empty(&legends);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Close the plot viewport */
  if(svg_end_plot_viewport(fp, NULL, verbose-3)) {
    fclose(fp); svg_legend_empty(&legends);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Make the plot legends into their own viewport */
  if(viewports.label_area_viewport.is!=0) {
    if(verbose>2) {printf("creating plot legends\n"); fflush(stdout);}
    int ret=svg_create_legends(fp, &viewports, &legends, NULL, verbose-3);
    if(ret) {
      fclose(fp); svg_legend_empty(&legends);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return TPCERROR_CANNOT_WRITE;
    }
  }
  svg_legend_empty(&legends);

  /* Close the SVG file */
  if(svg_close(fp, NULL, verbose-3)) {
    fclose(fp);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
