/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpctacmod.h"
/*****************************************************************************/

/*****************************************************************************/
int test_tacFittime(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret, n, st, et;
  TAC tac; tacInit(&tac);
  double t1, t2;

  if(verbose>1) printf("testing with NULL data.\n");
  n=tacFittime(NULL, &t1, &t2, NULL, NULL, NULL);
  if(n>=0) return(1);
  if(verbose>1) printf("testing with empty data.\n");
  t1=t2=0;
  n=tacFittime(&tac, &t1, &t2, &st, &et, status);
  if(n>0) return(2);

  if(verbose>1) printf("creating test data\n");
  ret=create_tac(&tac); if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("testing with large time range.\n");
  t1=-1.0E+99; t2=+1.0E+99; 
  n=tacFittime(&tac, &t1, &t2, &st, &et, status);
  if(verbose>2) printf("-> n=%d (%d-%d) t1=%g  t2=%g\n", n, st, et, t1, t2);
  if(n!=tac.sampleNr) {tacFree(&tac); return 11;}
  if(st!=0 || et!=tac.sampleNr-1) {tacFree(&tac); return 12;}
  if(!doubleMatch(t1, tac.x1[0], 1.0E-08)) {tacFree(&tac); return 13;}
  if(!doubleMatch(t2, tac.x2[tac.sampleNr-1], 1.0E-08)) {tacFree(&tac); return 14;}

  if(verbose>1) printf("same but with frame mid times.\n");
  t1=-1.0E+99; t2=+1.0E+99; 
  tac.isframe=0;
  n=tacFittime(&tac, &t1, &t2, &st, &et, status);
  if(verbose>2) printf("-> n=%d (%d-%d) t1=%g  t2=%g\n", n, st, et, t1, t2);
  if(n!=tac.sampleNr) {tacFree(&tac); return 21;}
  if(st!=0 || et!=tac.sampleNr-1) {tacFree(&tac); return 22;}
  if(!doubleMatch(t1, tac.x[0], 1.0E-08)) {tacFree(&tac); return 23;}
  if(!doubleMatch(t2, tac.x[tac.sampleNr-1], 1.0E-08)) {tacFree(&tac); return 24;}

  if(verbose>1) printf("testing with smaller time range.\n");
  t1=0.55; t2=4.4; 
  n=tacFittime(&tac, &t1, &t2, &st, &et, status);
  if(verbose>2) printf("-> n=%d (%d-%d) t1=%g  t2=%g\n", n, st, et, t1, t2);
  if(n!=tac.sampleNr-2) {tacFree(&tac); return 31;}
  if(st!=1 || et!=tac.sampleNr-2) {tacFree(&tac); return 32;}
  if(!doubleMatch(t1, tac.x[1], 1.0E-08)) {tacFree(&tac); return 33;}
  if(!doubleMatch(t2, tac.x[tac.sampleNr-2], 1.0E-08)) {tacFree(&tac); return 34;}

  if(verbose>1) printf("same but with frame start and end times.\n");
  t1=0.55; t2=4.4; 
  tac.isframe=1;
  n=tacFittime(&tac, &t1, &t2, &st, &et, status);
  if(verbose>2) printf("-> n=%d (%d-%d) t1=%g  t2=%g\n", n, st, et, t1, t2);
  if(n!=tac.sampleNr-2) {tacFree(&tac); return 41;}
  if(st!=1 || et!=tac.sampleNr-2) {tacFree(&tac); return 42;}
  if(!doubleMatch(t1, tac.x1[1], 1.0E-08)) {tacFree(&tac); return 43;}
  if(!doubleMatch(t2, tac.x2[tac.sampleNr-2], 1.0E-08)) {tacFree(&tac); return 44;}

  if(verbose>1) printf("TAC times in seconds.\n");
  tacXUnitConvert(&tac, UNIT_SEC, NULL);
  t1=0.55; t2=4.4; 
  n=tacFittime(&tac, &t1, &t2, &st, &et, status);
  if(verbose>2) printf("-> n=%d (%d-%d) t1=%g  t2=%g\n", n, st, et, t1, t2);
  if(n!=tac.sampleNr-2) {tacFree(&tac); return 51;}
  if(st!=1 || et!=tac.sampleNr-2) {tacFree(&tac); return 52;}
  tacXUnitConvert(&tac, UNIT_MIN, NULL);
  if(!doubleMatch(t1, tac.x1[1], 1.0E-08)) {tacFree(&tac); return 53;}
  if(!doubleMatch(t2, tac.x2[tac.sampleNr-2], 1.0E-08)) {tacFree(&tac); return 54;}


  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacReadModelingData(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int n, ret;
  double fitdur=0.0;
  TAC tac, inp; tacInit(&tac); tacInit(&inp);
  char tacfile[64], inpfile1[64], inpfile2[64], inpfile3[64];
  FILE *fp;

  tacfile[0]=inpfile1[0]=inpfile2[0]=inpfile3[0]=(char)0;

  if(verbose>1) printf("testing with NULL data.\n");
  ret=tacReadModelingData(NULL, NULL, NULL, NULL, &fitdur, 1, NULL, NULL, NULL, NULL);
  if(ret==TPCERROR_OK) {return 1;}
  if(verbose>1) printf("testing with empty data.\n");
  ret=tacReadModelingData(tacfile, inpfile1, inpfile2, inpfile3, &fitdur, 1, &n, 
                          &tac, &inp, status);
  if(ret==TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 2;}

  if(verbose>1) printf("creating test data\n");

  if(create_tis1(&tac)!=TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 3;}
  strcpy(tacfile, "tacmodelinput_tissue.tac"); 
  ret=1; fp=fopen(tacfile, "w");
  if(fp!=NULL) {ret=tacWrite(&tac, fp, TAC_FORMAT_PMOD, 1, NULL); fclose(fp);}
  if(ret) {tacFree(&tac); tacFree(&inp); return 3;}

  if(create_inp1(&inp)!=TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 4;}
  strcpy(inpfile1, "tacmodelinput_plasma.tac"); 
  ret=1; fp=fopen(inpfile1, "w");
  if(fp!=NULL) {ret=tacWrite(&inp, fp, TAC_FORMAT_PMOD, 1, NULL); fclose(fp);}
  if(ret) {tacFree(&tac); tacFree(&inp); return 4;}

  tacFree(&inp);
  if(create_inp2(&inp)!=TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 5;}
  strcpy(inpfile2, "tacmodelinput_metab.tac");
  ret=1; fp=fopen(inpfile2, "w");
  if(fp!=NULL) {ret=tacWrite(&inp, fp, TAC_FORMAT_PMOD, 1, NULL); fclose(fp);}
  if(ret) {tacFree(&tac); tacFree(&inp); return 5;}

  tacFree(&inp);
  if(create_inp3(&inp)!=TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 6;}
  strcpy(inpfile3, "tacmodelinput_blood.tac");
  /* Change units to check that function can fix them */
  tacXUnitConvert(&inp, UNIT_SEC, NULL);
  tacYUnitConvert(&inp, UNIT_BQ_PER_ML, NULL);
  ret=1; fp=fopen(inpfile3, "w");
  if(fp!=NULL) {ret=tacWrite(&inp, fp, TAC_FORMAT_PMOD, 1, NULL); fclose(fp);}
  if(ret) {tacFree(&tac); tacFree(&inp); return 6;}

  tacFree(&tac); tacFree(&inp);

  if(verbose>1) printf("try to read all files.\n");
  fitdur=1.0E+99;
  ret=tacReadModelingData(tacfile, inpfile1, inpfile2, inpfile3, &fitdur, 1, &n,
                          &tac, &inp, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 11;}
  if(tac.tacNr!=1 || tac.sampleNr!=2) {tacFree(&tac); tacFree(&inp); return 12;}
  if(inp.tacNr!=3 || inp.sampleNr!=4) {tacFree(&tac); tacFree(&inp); return 13;}
  if(!doubleMatch(tac.c[0].y[1], 1.0, 1.0E-08) ||
     !doubleMatch(inp.c[0].y[3], 0.0, 1.0E-08) ||
     !doubleMatch(inp.c[1].y[3], 10.0, 1.0E-08) ||
     !doubleMatch(inp.c[2].y[3], 6.0, 1.0E-08) )
  {
    tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacWrite(&inp, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); tacFree(&inp); 
    return 14;
  }
  if(!doubleMatch(fitdur, 14.0, 1.0E-08) || n!=2) {
    tacFree(&tac); tacFree(&inp); return 15;}


  if(verbose>1) printf("try to read all files except blood.\n");
  fitdur=1.0E+99;
  ret=tacReadModelingData(tacfile, inpfile1, inpfile2, "", &fitdur, 1, &n,
                          &tac, &inp, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 21;}
  if(tac.tacNr!=1 || tac.sampleNr!=2) {tacFree(&tac); tacFree(&inp); return 22;}
  if(inp.tacNr!=2 || inp.sampleNr!=4) {tacFree(&tac); tacFree(&inp); return 23;}
  if(!doubleMatch(tac.c[0].y[1], 1.0, 1.0E-08) ||
     !doubleMatch(inp.c[0].y[3], 0.0, 1.0E-08) ||
     !doubleMatch(inp.c[1].y[3], 10.0, 1.0E-08) )
  {
    tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacWrite(&inp, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); tacFree(&inp); 
    return 24;
  }
  if(!doubleMatch(fitdur, 14.0, 1.0E-08) || n!=2) {
    tacFree(&tac); tacFree(&inp); return 25;}

  if(verbose>1) printf("missing input file 1 or 2 should lead to an error.\n");
  fitdur=1.0E+99;
  ret=tacReadModelingData(tacfile, "", inpfile2, inpfile3, &fitdur, 1, &n,
                          &tac, &inp, status);
  if(ret==TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 31;}
  ret=tacReadModelingData(tacfile, inpfile1, NULL, inpfile3, &fitdur, 1, &n,
                          &tac, &inp, status);
  if(ret==TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 32;}


  if(verbose>1) printf("use smaller fit duration.\n");
  fitdur=7.0;
  ret=tacReadModelingData(tacfile, inpfile1, NULL, NULL, &fitdur, 1, &n,
                          &tac, &inp, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 41;}
  if(verbose>4) {
    tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacWrite(&inp, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  }
  if(tac.tacNr!=1 || tac.sampleNr!=2) {tacFree(&tac); tacFree(&inp); return 42;}
  if(inp.tacNr!=1 || inp.sampleNr!=4) {tacFree(&tac); tacFree(&inp); return 43;}
  if(!doubleMatch(fitdur, 6.0, 1.0E-08) || n!=1) {
    tacFree(&tac); tacFree(&inp); return 45;}


  tacFree(&tac); tacFree(&inp);
  remove(tacfile); remove(inpfile1); remove(inpfile2); remove(inpfile3);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacReadReference(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("\ntesting with NULL data.\n");
  if(tacReadReference(NULL, NULL, NULL, NULL, NULL)>0) {return 1;}

  if(verbose>1) printf("\nreference among other TACs.\n");

  TAC tac, inp; tacInit(&tac); tacInit(&inp);
  int i;

  if(verbose>1) printf("\ntesting with NULL data.\n");
  if(tacReadReference(&tac, "tac1", NULL, NULL, status)>0) {
    tacFree(&tac); tacFree(&inp); return 101;}

  /* making test data */
  if(create_tac(&tac)!=TPCERROR_OK) {tacFree(&tac); tacFree(&inp); return 110;}
  if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);

  if(verbose>1) printf("\nselect one tac with name.\n");
  if(tacReadReference(&tac, "roi2", NULL, &i, status)!=1 || i!=1) {
    tacFree(&tac); tacFree(&inp); return 111;}

  if(verbose>1) printf("\nselect one tac with number.\n");
  if(tacReadReference(&tac, "1", NULL, &i, status)!=1 || i!=0) {
    tacFree(&tac); tacFree(&inp); return 112;}

  if(verbose>1) printf("\nuse name that does not exist.\n");
  if(tacReadReference(&tac, "nonexisting", NULL, &i, status)>0) {
    tacFree(&tac); tacFree(&inp); return 113;}

  if(verbose>1) printf("\nuse name that matches all TAC names.\n");
  if(tacReadReference(&tac, "roi", NULL, &i, status)>0) {
    tacFree(&tac); tacFree(&inp); return 114;}
  tacFree(&tac); tacFree(&inp);

  /* the same but move to ref tac */
  int n;

  if(verbose>1) printf("\nselect one tac with name.\n");
  create_tac(&tac);
  n=tacReadReference(&tac, "roi2", &inp, &i, status);
  if(n!=1 || i!=0 || tac.tacNr!=2 || inp.tacNr!=1) {
    printf("status := %s\n", errorMsg(status->error));
    printf("n=%d i=%d tac.tacNr=%d inp.tacNr=%d\n", n, i, tac.tacNr, inp.tacNr);
    tacFree(&tac); tacFree(&inp); return 121;
  }
  tacFree(&tac); tacFree(&inp);
  
  if(verbose>1) printf("\nselect one tac with number.\n");
  create_tac(&tac);
  if(tacReadReference(&tac, "3", &inp, &i, status)!=1  || i!=0 || 
     tac.tacNr!=2 || inp.tacNr!=1 || strcmp(inp.c[0].name, "roi3")) {
    tacFree(&tac); tacFree(&inp); return 121;
  }
  tacFree(&tac); tacFree(&inp);

  if(verbose>1) printf("\nuse name that does not exist.\n");
  create_tac(&tac);
  if(tacReadReference(&tac, "nonexisting", &inp, &i, status)>0) {
    tacFree(&tac); tacFree(&inp); return 123;}

  if(verbose>1) printf("use name that matches all TAC names.\n");
  if(tacReadReference(&tac, "roi", &inp, &i, status)>0 || tac.tacNr!=3 ||
     inp.tacNr!=0) {
    tacFree(&tac); tacFree(&inp); return 124;}
  tacFree(&tac); tacFree(&inp);



  if(verbose>1) printf("\nreference in separate file.\n");

  FILE *fp;
  char inpfile[64]="tacmodelinput_ref.tac";
  create_tac(&inp);
  strcpy(inp.c[0].name, "cer_sin");
  strcpy(inp.c[1].name, "cer");
  strcpy(inp.c[2].name, "cerdx");
  n=1; fp=fopen(inpfile, "w");
  if(fp!=NULL) {n=tacWrite(&inp, fp, TAC_FORMAT_PMOD, 1, NULL); fclose(fp);}
  if(n) {tacFree(&tac); tacFree(&inp); return 200;}

  if(verbose>1) printf("\nadding to separate struct.\n");
  create_tac(&tac); tacFree(&inp);
  n=tacReadReference(&tac, inpfile, &inp, &i, status);
  if(n!=3 || i!=1 || tac.tacNr!=3 || inp.tacNr!=3) {
    printf("status := %s\n", errorMsg(status->error));
    printf("n=%d i=%d tac.tacNr=%d inp.tacNr=%d\n", n, i, tac.tacNr, inp.tacNr);
    tacFree(&tac); tacFree(&inp); return 211;
  }

  if(verbose>1) printf("\ndifferent time units.\n");
  tacXUnitConvert(&tac, UNIT_SEC, NULL);
  tacFree(&inp);
  n=tacReadReference(&tac, inpfile, &inp, &i, status);
  if(n!=3 || i!=1 || tac.tacNr!=3 || inp.tacNr!=3 || inp.tunit!=UNIT_SEC) {
    printf("status := %s\n", errorMsg(status->error));
    printf("n=%d i=%d tac.tacNr=%d inp.tacNr=%d\n", n, i, tac.tacNr, inp.tacNr);
    tacFree(&tac); tacFree(&inp); return 221;
  }
  tacFree(&inp);

  if(verbose>1) printf("\nadding to tissue struct, still different time units.\n");
  n=tacReadReference(&tac, inpfile, NULL, &i, status);
  if(n!=3 || i!=4 || tac.tacNr!=6 || tac.tunit!=UNIT_SEC || 
     tacSelectedTACs(&tac)!=3) {
    printf("status := %s\n", errorMsg(status->error));
    printf("n=%d i=%d tac.tacNr=%d inp.tacNr=%d\n", n, i, tac.tacNr, inp.tacNr);
    tacFree(&tac); tacFree(&inp); return 231;
  }

  if(verbose>1) printf("\nreading from file with different sample times should fail.\n");
  tacFree(&tac); create_tis1(&tac);
  n=tacReadReference(&tac, inpfile, NULL, &i, status);
  if(verbose>2) printf("status := %s\n", errorMsg(status->error));
  if(n>0 || status->error==TPCERROR_OK) {
    tacFree(&tac); tacFree(&inp); return 241;
  }
  tacFree(&inp);
  n=tacReadReference(&tac, inpfile, &inp, &i, status);
  if(verbose>2) printf("status := %s\n", errorMsg(status->error));
  if(n>0 || status->error==TPCERROR_OK) {
    tacFree(&tac); tacFree(&inp); return 242;
  }


  tacFree(&tac); tacFree(&inp);
  remove(inpfile);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacReadModelingInput(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;
  TAC inp; tacInit(&inp);
  char inpfile1[64], inpfile2[64], inpfile3[64];
  FILE *fp;

  inpfile1[0]=inpfile2[0]=inpfile3[0]=(char)0;

  if(verbose>1) printf("\ntesting with NULL data.\n");
  ret=tacReadModelingInput(NULL, NULL, NULL, NULL, NULL);
  if(ret==TPCERROR_OK) {return 1;}
  if(verbose>1) printf("testing with empty data.\n");
  ret=tacReadModelingInput(inpfile1, inpfile2, inpfile3, &inp, status);
  if(ret==TPCERROR_OK) {tacFree(&inp); return 2;}

  if(verbose>1) printf("creating test data\n");

  if(create_inp1(&inp)!=TPCERROR_OK) {tacFree(&inp); return 4;}
  strcpy(inpfile1, "tacmodelinput_plasma.tac"); 
  ret=1; fp=fopen(inpfile1, "w");
  if(fp!=NULL) {ret=tacWrite(&inp, fp, TAC_FORMAT_PMOD, 1, NULL); fclose(fp);}
  if(ret) {tacFree(&inp); return 4;}

  tacFree(&inp);
  if(create_inp2(&inp)!=TPCERROR_OK) {tacFree(&inp); return 5;}
  strcpy(inpfile2, "tacmodelinput_metab.tac");
  ret=1; fp=fopen(inpfile2, "w");
  if(fp!=NULL) {ret=tacWrite(&inp, fp, TAC_FORMAT_PMOD, 1, NULL); fclose(fp);}
  if(ret) {tacFree(&inp); return 5;}

  tacFree(&inp);
  if(create_inp3(&inp)!=TPCERROR_OK) {tacFree(&inp); return 6;}
  strcpy(inpfile3, "tacmodelinput_blood.tac");
  /* Change units to check that function can fix them */
  tacXUnitConvert(&inp, UNIT_SEC, NULL);
  tacYUnitConvert(&inp, UNIT_BQ_PER_ML, NULL);
  ret=1; fp=fopen(inpfile3, "w");
  if(fp!=NULL) {ret=tacWrite(&inp, fp, TAC_FORMAT_PMOD, 1, NULL); fclose(fp);}
  if(ret) {tacFree(&inp); return 6;}

  tacFree(&inp);

  if(verbose>1) printf("\ntry to read all files.\n");
  ret=tacReadModelingInput(inpfile1, inpfile2, inpfile3, &inp, status);
  if(ret!=TPCERROR_OK) {tacFree(&inp); return 11;}
  if(inp.tacNr!=3 || inp.sampleNr!=4) {tacFree(&inp); return 13;}
  if(!doubleMatch(inp.c[0].y[3], 0.0, 1.0E-08) ||
     !doubleMatch(inp.c[1].y[3], 10.0, 1.0E-08) ||
     !doubleMatch(inp.c[2].y[3], 6.0, 1.0E-08) )
  {
    tacWrite(&inp, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&inp); 
    return 14;
  }

  if(verbose>1) printf("\ntry to read all files except blood.\n");
  ret=tacReadModelingInput(inpfile1, inpfile2, "", &inp, status);
  if(ret!=TPCERROR_OK) {tacFree(&inp); return 21;}
  if(inp.tacNr!=2 || inp.sampleNr!=4) {tacFree(&inp); return 23;}
  if(!doubleMatch(inp.c[0].y[3], 0.0, 1.0E-08) ||
     !doubleMatch(inp.c[1].y[3], 10.0, 1.0E-08) )
  {
    tacWrite(&inp, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&inp); 
    return 24;
  }

  if(verbose>1) printf("\nmissing input file 1 or 2 should lead to an error.\n");
  ret=tacReadModelingInput("", inpfile2, inpfile3, &inp, status);
  if(ret==TPCERROR_OK) {tacFree(&inp); return 31;}
  ret=tacReadModelingInput(inpfile1, NULL, inpfile3, &inp, status);
  if(ret==TPCERROR_OK) {tacFree(&inp); return 32;}

  fflush(stdout); fflush(stderr);

  tacFree(&inp);
  remove(inpfile1); remove(inpfile2); remove(inpfile3);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
