/** @file addvers.c
 *  @brief Add version number to the file name. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Add version number to the file name, if file name does not already include",
  "version number. Version is given in format 'n.n.n' as the first command-line",
  "argument either directly or inside a file with keyword 'version'.",
  "Version is added to the file name in format '_n_n_n' before any extension.",
  " ",
  "Usage: @P [Options] version file",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1:",
  "     @P 2.13.4 patlak.exe",
  " ",
  "Example 2:",
  "     patlak --version > patlak.txt",
  "     iftren patlak.txt \"tpcclib version\" \"version\"",
  "     @P patlak.txt patlak.exe",
  " ",
  "See also: pdoc2htm, iftadd, iftren",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  char     *cptr;
  char      exefile[FILENAME_MAX], verfile[FILENAME_MAX], vers[FILENAME_MAX];
  int       ret;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  exefile[0]=verfile[0]=vers[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}  
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(verfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(exefile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing or wrong? */
  if(!exefile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("exefile := %s\n", exefile);
    fflush(stdout);
  }

  /* Try to open the first argument as IFT file */
  if(verbose>2) printf("trying to read %s\n", verfile);
  FILE *fp=fopen(verfile, "r"); 
  if(fp==NULL) {
    if(verbose>2) printf("  ... could not open %s\n", verfile);
    /* Assuming that version number was given directly */
    strncpyClean(vers, verfile, FILENAME_MAX); verfile[0]=(char)0;
    if(verbose>1) {
      printf("version := %s\n", vers);
      fflush(stdout);
    }
  } else {
    if(verbose>1) {
      printf("version_file := %s\n", verfile);
      fflush(stdout);
    }
    /* Trying to read version from file */
    IFT ift; iftInit(&ift);
    ret=iftRead(&ift, fp, 1, 1, &status); 
    fclose(fp);
    if(ret) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      iftFree(&ift); return(2);
    }
    if(verbose>2) printf("list size: %d item(s)\n", ift.keyNr);
    /* Search for the first occurrence of keyname 'version' */
    int li=iftFindKey(&ift, "version", 0);
    if(li<0) {
      fprintf(stderr, "Error: file contains no key '%s'\n", "version");
      iftFree(&ift); return(3);
    }
    if(verbose>3) printf("key_index := %d\n", li);
    strncpyClean(vers, ift.item[li].value, FILENAME_MAX);
    if(verbose>1) {
      printf("version := %s\n", vers);
      fflush(stdout);
    }
    iftFree(&ift);
  }


  /* Check that version number is somewhat valid */
  ret=0;
  {
    int i=0; 
    while(vers[i]) {
      if(vers[i]!='.' && isalnum(vers[i])==0) ret++;
      if(isblank(vers[i])!=0) ret++;
      i++;
    }
  }
  if(ret>0 || strlen(vers)<1 || strlen(vers)>20) {
    fprintf(stderr, "Error: invalid version.\n");
    return(2);
  }

  /* Replace dots with underlines */
  strReplaceChar(vers, '.', '_');
  if(verbose>1) {
    printf("version := %s\n", vers);
    fflush(stdout);
  }


  /*
   *  Process the file name
   */

  /* Get the path */
  char pathname[FILENAME_MAX];
  strcpy(pathname, exefile);
  filenameRmFile(pathname);
  if(verbose>1) printf("pathname := '%s'\n", pathname);
  /* Get the extensions */
  char extname[FILENAME_MAX];
  cptr=filenameGetExtensions(exefile);
  if(cptr!=NULL) strcpy(extname, cptr); else extname[0]=(char)0;
  if(verbose>1) printf("extname := '%s'\n", extname);
  /* Get the file name without path and extensions */
  char filename[FILENAME_MAX];
  strcpy(filename, exefile);
  filenameRmExtensions(filename);
  filenameRmPath(filename);
  if(verbose>1) printf("filename := '%s'\n", filename);
  if(strlen(filename)<1) {
    fprintf(stderr, "Error: invalid file name.\n");
    return(2);
  }
  /* Check that file name does not already contain the same version string */
  if(strstr(filename, vers)!=NULL) {
    fprintf(stderr, "Error: file name already contains version number.\n");
    return(4);
  } 
  /* Construct new file name */
  strcpy(verfile, pathname);
  strcat(verfile, "/");
  strcat(verfile, filename);
  if(strlen(verfile)+strlen(vers)+strlen(extname)+2 > FILENAME_MAX) {
    fprintf(stderr, "Error: too long file name or version.\n");
    return(5);
  }
  strcat(verfile, "_");
  strcat(verfile, vers);
  strcat(verfile, extname);
  if(verbose>1) {
    printf("newfile := %s\n", verfile);
    fflush(stdout);
  }

  /*
   *  Rename the file
   */
  remove(verfile); // rename() may return error if file exists (implementation-defined)
  if(rename(exefile, verfile)!=0) {
    fprintf(stderr, "Error: cannot rename '%s' to '%s'.\n", exefile, verfile);
    return(9);
  }
  if(verbose>0) printf("%s -> %s\n", exefile, verfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
