/** @file htmlgall.c
 *  @brief Simple HTML gallery of SVG plots. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Make simple HTML page working as a gallery of argument SVG plots.",
  " ",
  "Usage: @P [Options] htmlfile svgfiles",
  " ",
  "Options:",
  " -cols=<Max columns>",
  "     Maximum number of plots on one row (default is 5).",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P study.html study*.svg",
  " ",
  "See also: tac2svg",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  char   svgfile[FILENAME_MAX], htmlfile[FILENAME_MAX];
  int    firstfile=0, fileNr=0;
  int    maxCols=5;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  htmlfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "COLS=", 5)==0) {
      if(atoiCheck(cptr+5, &maxCols)==0 && maxCols>0) continue;
    } else if(strncasecmp(cptr, "COL=", 4)==0) {
      if(atoiCheck(cptr+4, &maxCols)==0 && maxCols>0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}  
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(htmlfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    firstfile=ai; fileNr=1;
    strlcpy(svgfile, argv[ai++], FILENAME_MAX);
  } else {
    fprintf(stderr, "Error: missing filename.\n"); return(1);
  }
  /* More file names? */
  int ret=0;
  for(; ai<argc; ai++) {
    if(access(argv[ai], 0) == -1) ret++; else fileNr++;
  }

  /* Is something missing or wrong? */
  if(!htmlfile[0] || fileNr<1) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("htmlfile := %s\n", htmlfile);
    printf("fileNr := %d\n", fileNr);
  }



  /* Open output file */
  if(verbose>1) printf("opening %s for writing\n", htmlfile);
  FILE *fp;
  if((fp=fopen(htmlfile, "w")) == NULL) {
    printf("Error: cannot open %s.\n", htmlfile);
    return(11);
  }

  /* Write HTML header */
  int len=fprintf(fp, "<!DOCTYPE html>\n");
  if(len<10) {fclose(fp); return(12);}
  fprintf(fp, "<html>\n");
  fprintf(fp, "<head>\n");
  fprintf(fp, "  <meta charset=\"UTF-8\">\n");
  fprintf(fp, "  <title>%s</title>\n", "Gallery");
  fprintf(fp, "  <style type=\"text/css\">\n");
  fprintf(fp, "    * {font-family: monospace;}\n");
  fprintf(fp, "    img {border-width: 0px; padding: 5px 0px 0px 10px; width:18%%;}\n");
  fprintf(fp, "  </style>\n");
  fprintf(fp, "</head>\n\n");

  /* Write HTML body */
  fprintf(fp, "<body>\n");

  /* Start figure */
  fprintf(fp, "<figure>\n");

  /* Add links to plots */
  int column=0, row=0;
  for(int ai=firstfile; ai<argc; ai++) {
    strlcpy(svgfile, argv[ai], FILENAME_MAX);
    /* Make link to the plot */
    fprintf(fp, "  <a href=\"%s\">", svgfile);
    /* Make img tag */
    fprintf(fp, "<img src=\"%s\" alt=\"%s\">", svgfile, svgfile);
    /* Close link */
    fprintf(fp, "</a>\n");
    column++;
    if(column==maxCols) {
      fprintf(fp, "  <br>\n");
      column=0; row++;
    }
  }

  /* Close figure */
  fprintf(fp, "</figure>\n");

  /* Close HTML */
  fprintf(fp, "</body>\n");
  fprintf(fp, "</html>\n");
  /* ... and file */
  fclose(fp);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
