/** @file pdoc2htm.c
 *  @brief Convert program user information text file into HTML format. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Convert user information text file, printed by command-line programs and",
  "scripts, to HTML format.",
  " ",
  "Usage: @P [Options] txtfile [htmlfile]",
  " ",
  "Options:",
  " -links",
  "     Sets HTML-links for 'See also:' list.",
  " -GPL=<yes|no>",
  "     Link to GNU GPL is added (default) or not added.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  program --help > temp.txt",
  "  pdoc2htm temp.txt program.html",
  " ",
  "See also: addvers",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  char     *cptr, txtfile[FILENAME_MAX], htmfile[FILENAME_MAX];
  int       set_links=0; // 0=no, 1=yes
  int       set_gpl=1;   // 0=no, 1=yes


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  txtfile[0]=htmfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "GPL=", 4)==0) {
      cptr+=4;
      if(strncasecmp(cptr, "NO", 1)==0) {
        set_gpl=0; continue;
      } else if(strncasecmp(cptr, "YES", 1)==0) {
        set_gpl=1; continue;
      }
    } else if(strncasecmp(cptr, "LINKS", 1)==0) {
      set_links=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}  
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(txtfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(htmfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing or wrong? */
  if(!txtfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  /* Construct the output file name, if not specified by user */
  if(!htmfile[0]) {
    strcpy(htmfile, txtfile);
    if(!filenameRmExtension(htmfile)) {
      fprintf(stderr, "Error: cannot construct output file name.\n");
      return(1);
    }
    strlcat(htmfile, ".html", FILENAME_MAX);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("txtfile := %s\n", txtfile);
    printf("htmlfile := %s\n", htmfile);
    printf("set_links := %d\n", set_links);
    printf("set_gpl := %d\n", set_gpl);
  }


  /* Open inputfile */
  FILE *fp;
  if(verbose>1) printf("opening %s for reading\n", txtfile);
  if((fp=fopen(txtfile, "r")) == NULL) {
    printf("Error: cannot open %s.\n", txtfile);
    return(2);
  }
  /* Get its size */
  size_t txtsize=asciiFileSize(fp, NULL);
  if(txtsize<3 || txtsize>99999) {
    printf("Error: invalid contents in %s.\n", txtfile);
    fclose(fp); return(3);
  }
  /* Read file into buffer */
  char *buf=asciiFileRead(fp, NULL, txtsize+1);
  fclose(fp);
  if(buf==NULL) {
    printf("Error: cannot read %s.\n", txtfile);
    return(4);
  }

  /* Open output file */
  if(verbose>1) printf("opening %s for writing\n", htmfile);
  if((fp=fopen(htmfile, "w")) == NULL) {
    printf("Error: cannot open %s.\n", htmfile);
    free(buf); return(11);
  }

  /* Write HTML header */
  int len=fprintf(fp, "<!DOCTYPE html>\n");
  if(len<10) {free(buf); fclose(fp); return(12);}
  fprintf(fp, "<html>\n");
  fprintf(fp, "<head>\n");
  fprintf(fp, "  <meta charset=\"UTF-8\">\n");
  fprintf(fp, "  <title>%s</title>\n", "User information");
  fprintf(fp, "  <style type=\"text/css\">\n");
  fprintf(fp, "    * {font-family: monospace;}\n");
  fprintf(fp, "    footer {\n");
  fprintf(fp, "      border:1px solid gray;\n");
  fprintf(fp, "      font-size: smaller;\n");
  fprintf(fp, "    }\n");
  fprintf(fp, "    img {border-width: 0px;}\n");
  fprintf(fp, "  </style>\n");
  fprintf(fp, "</head>\n\n");
  
  /* Write HTML body */
  fprintf(fp, "<body>\n");

  /* Print usage */
  fprintf(fp, "<pre>\n");

  int i=1;
  char docline[256];
  len=strTokenNCpy(buf, "\n\r", i, docline, 256);
  while(len>0) {
    /* Check if line contains (c) or (C) */
    cptr=strcasestr(docline, "(c)");
    if(cptr) {
      *cptr=(char)0;
      /* Check if there is also 'Turku PET Centre' */
      char *cptr2=strcasestr(cptr+3, "Turku PET Centre");
      if(cptr2) {
        *cptr2=(char)0;
        fprintf(fp, "%s&copy;%s", docline, cptr+3);
        // Phrasing content is allowed inside <pre>
        fprintf(fp, "<a href=\"http://www.turkupetcentre.fi\">Turku PET Centre</a>\n");
      } else {
        fprintf(fp, "%s&copy;%s\n", docline, cptr+3);
      }
      len=strTokenNCpy(buf, "\n\r", ++i, docline, 256);
      continue;
    }

    /* Set links to "see also" docs;
       phrasing content is allowed inside <pre>
    */
    if(set_links!=0 && strcasestr(docline, "See also: ")!=NULL) {
      /* copy until ':' */
      char *cptr2=docline;
      while(*cptr2!='\0') {
        fputc(*cptr2, fp);
        if(*cptr2==':') {cptr2++; break;}
        cptr2++;
      }
      /* the rest of line with token */
      int j=1, len2;
      char swname[32];
      len2=strTokenNCpy(cptr2, " ,;\t\n\r", j, swname, 32);
      while(len2>0) {
        if(j>1) fputc(',', fp);
        fprintf(fp, " <a href=\"./%s.html\">%s</a>", swname, swname);
        len2=strTokenNCpy(cptr2, " ,;\t\n\r", ++j, swname, 32);
      }
      fputs("\n", fp);
      len=strTokenNCpy(buf, "\n\r", ++i, docline, 256);
      continue;
    }

    /* Write the line, but replace <, >, and & characters with html codes */
    cptr=docline;
    while(*cptr!='\0') {
      if(*cptr=='<') fputs("&lt;", fp);
      else if(*cptr=='>') fputs("&gt;", fp);
      else if(*cptr=='&') fputs("&amp;", fp);
      else fputc(*cptr, fp);
      cptr++;
    }
    fputs("\n", fp);
    len=strTokenNCpy(buf, "\n\r", ++i, docline, 256);
  }
  fprintf(fp, "</pre>\n");

  /* Write footer */
  fprintf(fp, "\n<footer>\n");

  /* Licence info */
  if(set_gpl) {
    fprintf(fp, "<div>\n");
    /* Icon with link to GNU-GPL */
    fprintf(fp, "<a href=\"http://www.gnu.org/licenses/gpl-3.0-standalone.html\">\n");
    fprintf(fp, "<img alt=\"GNU GPL\" ");
    fprintf(fp, "style=\"width:88px; height:31px; float:left; margin: 5px 20px 5px 5px;\"");
    fprintf(fp, "\n src=\"http://www.turkupetcentre.net/software/gplv3-88x31.png\"></a>\n");
    fprintf(fp, "</div>\n");
  }

  fprintf(fp, "</footer>\n");
           
  /* Close HTML */
  fprintf(fp, "</body>\n");
  fprintf(fp, "</html>\n");
  /* ... and file */
  fclose(fp);

  free(buf);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
