/** @file parcoll.c
 *  @brief Collect parameters from specified files into one parameter file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Collect the contents of parameter files.",
  " ",
  "Usage: @P [options] newfile filename(s)",
  " ",
  "Options:",
  " -par=<list>",
  "     List of parameter numbers or names to be copied; all by default.",
  " -tac=<list>",
  "     List of TAC numbers or names to be copied; all by default.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P all.par ut????srtm.res",
  " ",
  "See also: paradd, iftedit, parformat, parrenp, parmean, parai, rescoll",
  " ",
  "Keywords: parameter, tool, format, CSV, RES, FIT",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  char newfile[FILENAME_MAX], parfile[FILENAME_MAX];
  char *pars=NULL, *tacs=NULL; // remember to free allocated memory before exit

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  newfile[0]=parfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "PAR=", 4)==0 && strnlen(cptr, 5)>4) {
      pars=strdup(cptr+4); continue;
    } else if(strncasecmp(cptr, "TAC=", 4)==0 && strnlen(cptr, 5)>4) {
      tacs=strdup(cptr+4); continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); free(pars); free(tacs); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); free(pars); free(tacs); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); free(pars); free(tacs); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;

  /* Next argument is the name of the collection file */
  if(ai<argc) strlcpy(newfile, argv[ai++], FILENAME_MAX);
  if(ai==argc) {tpcPrintUsage(argv[0], info, stdout); free(pars); free(tacs); return(1);}

  /* Next arguments are names of files to collect data from */
  int fileNr=0, ffi=0;
  for(; ai<argc; ai++) {
    if(verbose>2) {printf("file %d: %s\n", 1+fileNr, argv[ai]); fflush(stdout);}
    if(ffi<1) ffi=ai;
    fileNr++;
  }
  /* Is something missing? */
  if(fileNr<2) {
    fprintf(stderr, "Error: at least two parameter files are needed.\n");
    free(pars); free(tacs); return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("fileNr := %d\n", fileNr);
    printf("newfile := %s\n", newfile);
    if(pars!=NULL) printf("pars := '%s'\n", pars);
    if(tacs!=NULL) printf("tacs := '%s'\n", tacs);
    fflush(stdout);
  }


  /*
   *  Read all specified files and store the contents
   */
  if(verbose>0) {printf("reading %d files\n", fileNr); fflush(stdout);}
  PAR *parlist=(PAR*)malloc(fileNr*sizeof(PAR));
  if(parlist==NULL) {
    fprintf(stderr, "Error: cannot allocate memory.\n"); free(pars); free(tacs); return(2);
  }
  int pfileNr=0;
  for(int fi=0; fi<fileNr; fi++) parInit(parlist+fi);
  for(int fi=0; fi<fileNr; fi++) {
    /* Read file */
    strlcpy(parfile, argv[ffi+fi], FILENAME_MAX);
    if(verbose>0) {printf("%s\n", parfile); fflush(stdout);}
    if(verbose>2) printf("index=%d\n", pfileNr);
    if(parRead(parlist+pfileNr, parfile, &status)) {
      fprintf(stderr, "Error: %s.\n", errorMsg(status.error));
      for(int j=0; j<pfileNr; j++) parFree(parlist+j); 
      free(parlist); free(pars); free(tacs); return(2);
    }
    if(verbose>1) {
      printf("  fileformat := %s\n", parFormattxt(parlist[pfileNr].format));
      printf("  parNr := %d\n", parlist[pfileNr].parNr);
      printf("  tacNr := %d\n", parlist[pfileNr].tacNr);
      char *cptr=parIsStudyNr(parlist+pfileNr);
      if(cptr!=NULL) printf("  studyNr := %s\n", cptr);
    }
    /* Make sure that we got study number */
    if(!parIsStudyNr(parlist+pfileNr)) {
      char studynr[MAX_STUDYNR_LEN+1];
      if(studynrFromFilename(parfile, studynr, 1)==0)
        iftPut(&parlist[pfileNr].h, "studynr", studynr, 0, NULL);
    }
    /* Select specified parameters, or all */
    if(pars==NULL) {
      /* No parameters specified: select all */
      parSelectParameters(parlist+pfileNr, NULL, 0, &status);
    } else {
      char buf[512];
      for(int i=0; i<strTokenNr(pars, ",;"); i++) {
        if(strTokenNCpy(pars, ",;", i+1, buf, 512)<1) continue;
        parSelectParameters(parlist+pfileNr, buf, 0, &status);
      }
      int parNr=parSelectedParameters(parlist+pfileNr);
      if(parNr==0) {
        fprintf(stderr, "Warning: specified parameter(s) not found in %s.\n", parfile);
        parFree(parlist+pfileNr); continue; // do not store data from this file
      }
      if(parNr<parlist[pfileNr].parNr) { // delete non-selected parameters
        int i=parlist[pfileNr].parNr-1;
        while(i>=0) {
          if(parlist[pfileNr].n[i].sw==0) parDeletePar(parlist+pfileNr, i);
          i--;
        }
        if(verbose>1) printf("  parNrFinal := %d\n", parlist[pfileNr].parNr);
      }
    }
    /* Select specified TACs, or all */
    if(tacs==NULL) {
      /* No TACs specified: select all */
      parSelectTACs(parlist+pfileNr, NULL, 0, &status);
    } else {
      char buf[512];
      for(int i=0; i<strTokenNr(tacs, ",;"); i++) {
        if(strTokenNCpy(tacs, ",;", i+1, buf, 512)<1) continue;
        parSelectTACs(parlist+pfileNr, buf, 0, &status);
      }
      int tacNr=parSelectedTACs(parlist+pfileNr);
      if(tacNr==0) {
        fprintf(stderr, "Warning: specified TAC(s) not found in %s.\n", parfile);
        parFree(parlist+pfileNr); continue; // do not store data from this file
      }
      if(tacNr<parlist[pfileNr].tacNr) { // delete non-selected parameters
        int i=parlist[pfileNr].tacNr-1;
        while(i>=0) {
          if(parlist[pfileNr].r[i].sw==0) parDeleteTAC(parlist+pfileNr, i);
          i--;
        }
        if(verbose>1) printf("  tacNrFinal := %d\n", parlist[pfileNr].tacNr);
      }
    }
    pfileNr++;
  } // next file
  fileNr=pfileNr;
  /* par and tac strings are not needed later */
  free(pars); free(tacs);
  /* Check that we still have data to collect */
  if(fileNr<2) {
    fprintf(stderr, "Error: files do not contain specified TACs or parameters.\n");
    for(int j=0; j<fileNr; j++) parFree(parlist+j);
    free(parlist); return(3);
  }


  /*
   *  Allocate space for collected parameters
   */
  if(verbose>0) {printf("preparing the collection\n"); fflush(stdout);}
  /* List TAC names */
  if(verbose>1) {printf("listing TACs\n"); fflush(stdout);}
  IFT tacnames; iftInit(&tacnames);
  for(int fi=0; fi<fileNr; fi++)
    for(int ci=0; ci<parlist[fi].tacNr; ci++)
      iftPut(&tacnames, parlist[fi].r[ci].name, NULL, 0, NULL);
  iftDeleteDuplicateKeys(&tacnames, &status);
  if(verbose>2) {printf("TACs:\n"); iftWrite(&tacnames, stdout, NULL); printf("\n");}
  /* List parameter names */
  if(verbose>1) {printf("listing parameters\n"); fflush(stdout);}
  IFT parnames; iftInit(&parnames);
  for(int fi=0; fi<fileNr; fi++)
    for(int pi=0; pi<parlist[fi].parNr; pi++)
      iftPut(&parnames, parlist[fi].n[pi].name, NULL, 0, NULL);
  iftDeleteDuplicateKeys(&parnames, &status);
  if(verbose>2) {printf("Parameters:\n"); iftWrite(&parnames, stdout, NULL); printf("\n");}
  /* Allocating memory */
  if(verbose>2) {
    printf("TACs := %d\n", tacnames.keyNr);
    printf("Parameters := %d\n", parnames.keyNr);
  }
  PAR par; parInit(&par);
  if(parAllocate(&par, parnames.keyNr, fileNr*tacnames.keyNr)!=TPCERROR_OK) {
    fprintf(stderr, "Error: cannot allocate memory.\n");
    iftFree(&tacnames); iftFree(&parnames);
    for(int j=0; j<fileNr; j++) parFree(parlist+j);
    free(parlist); return(4);
  }
  /* Set the file format */
  par.format=parFormatFromExtension(newfile);
  if(par.format==PAR_FORMAT_UNKNOWN) par.format=PAR_FORMAT_TSV_UK;
  /* Set parameter names */
  for(int i=0; i<parnames.keyNr; i++)
    strcpy(par.n[i].name, parnames.item[i].key);
  par.parNr=parnames.keyNr;



  /*
   *  Collect parameters
   */
  par.tacNr=0;
  for(int fi=0; fi<fileNr; fi++) {
    for(int ti=0; ti<tacnames.keyNr; ti++) {
      /* search this TAC */
      for(int ri=0; ri<parlist[fi].tacNr; ri++) {
        if(strcasecmp(parlist[fi].r[ri].name, tacnames.item[ti].key)==0) {
          /* Copy name */
          char buf[MAX_TACNAME_LEN+1];
          char *studynr=parIsStudyNr(parlist+fi);
          if(studynr) strlcpy(buf, studynr, MAX_TACNAME_LEN);
          /* if more than one TAC name listed, copy TAC name, too */
          if(tacnames.keyNr>1) {
            if(studynr) strlcat(buf, " ", MAX_TACNAME_LEN);
            strlcat(buf, tacnames.item[ti].key, MAX_TACNAME_LEN);
          }
          strlcpy(par.r[par.tacNr].name, buf, MAX_TACNAME_LEN);
          /* Copy matching parameters */
          for(int pi=0; pi<parnames.keyNr; pi++) {
            for(int pj=0; pj<parlist[fi].parNr; pj++)
              if(strcasecmp(parlist[fi].n[pj].name, parnames.item[pi].key)==0)
                par.r[par.tacNr].p[pi]=parlist[fi].r[ri].p[pj];
          }
          par.tacNr++;
        }
      }
    }
  }

  /* Free the original data */
  iftFree(&tacnames); iftFree(&parnames);
  for(int i=0; i<fileNr; i++) parFree(parlist+i);
  free(parlist);


  /* 
   *  Write collected parameter file
   */
  if(verbose>1) printf("  saving\n");
  {
    FILE *fp=fopen(newfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      parFree(&par); return(11);
    }
    int ret=parWrite(&par, fp, PAR_FORMAT_UNKNOWN, 1, &status);
    fclose(fp); parFree(&par);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      return(12);
    }
    if(verbose>0) printf("written %s\n", newfile);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
