/** @file parsort.c
 *  @brief Sort contents in parameter file.
 *  @details Application name was previously ressort. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Test more file formats.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Sort the contents of parameter files by the TAC name.",
  " ",
  "Usage: @P [options] filename(s)",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P ut????srtm.res",
  " ",
  "See also: paradd, parformat, reslist, parmatch, rescoll",
  " ",
  "Keywords: parameter, tool, format, CSV, RES, FIT",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, n, ffi=0;
  //char *cptr, 
  char parfile[FILENAME_MAX];
  PAR par;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  parInit(&par);
  parfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Next arguments are names of files */
  n=0;
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai;
    n++;
  }
  /* Is something missing? */
  if(n==0) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("fileNr := %d\n", n);
  }

  /*
   *  Sort all specified files
   */
  FILE *fp;
  for(ai=ffi; ai<argc; ai++) {
    strlcpy(parfile, argv[ai], FILENAME_MAX);
    if(verbose>0) printf("%s\n", parfile);
    /* Read file */
    ret=parRead(&par, parfile, &status);
    if(ret) {
      fprintf(stderr, "Error: %s.\n", errorMsg(status.error));
      parFree(&par);
      return(2);
    }
    /* If less then one TAC, no need to sort and save */
    if(par.tacNr<2) {
      if(verbose>1) fprintf(stdout, "  contains only one TAC\n");
      parFree(&par);
      continue;      
    }
    /* Sort the parameters */
    if(verbose>1) printf("  sorting\n");
    parSortByName(&par, &status);
    /* Write sorted parameter file */
    if(verbose>1) printf("  saving\n");
    fp=fopen(parfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      parFree(&par); return(11);
    }
    ret=parWrite(&par, fp, PAR_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    parFree(&par);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      return(12);
    }
  } // next file
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
