/** @file sifcat.c
    @brief Catenate SIF files from interrupted PET studies.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpccsv.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Catenate the scan information files (SIF) from interrupted PET studies",
  "into a single SIF file.",
  " ",
  "Usage: @P [options] combined_sif sif1 sif2 [sif3]",
  " ",
  "Usage: @P [options] filename(s) [isotope code]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P ub0371dy.sif ub0371dy1.sif ub0371dy2.sif ub0371dy3.sif",
  " ",
  "See also: sifisot, taclist, eframe, tacframe, tacweigh, ecatcat",
  " ",
  "Keywords: SIF, image, catenate, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/*
 *  Main
 */
int main(int argc, char *argv[])
{
  int ai, help=0, version=0, verbose=1;
  char siffile[FILENAME_MAX], catfile[FILENAME_MAX];
  int ret, ffi=0, fileNr=0;



  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  siffile[0]=catfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(catfile, argv[ai++], FILENAME_MAX);}
  fileNr=ffi=0;
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai;
    fileNr++;
  }
  if(fileNr<2) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("catfile := %s\n", catfile);
    printf("fileNr := %d\n", fileNr);
    fflush(stdout);
  }

  /*
   *  Check that SIF files (except the result SIF file) do exist
   */
  for(ai=ffi; ai<argc; ai++) {
    if(access(argv[ai], 0) == -1) {
      fprintf(stderr, "Error: file %s not found.\n", argv[ai]);
      return(2);
    }
  }
  /* Check that output file does not exist */
  if(access(catfile, 0) != -1) {
    fprintf(stderr, "Warning: file %s exists.\n", catfile);
  }


  /*
   *  Read the first SIF (as TAC)
   */
  TAC tac; tacInit(&tac);
  strlcpy(siffile, argv[ffi], FILENAME_MAX);
  if(verbose>1) fprintf(stdout, "reading %s\n", siffile);
  ret=tacRead(&tac, siffile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    if(verbose>1) printf("ret := %d\n", ret);
    tacFree(&tac); return(4);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
  }
  if(tac.format!=TAC_FORMAT_SIF) {
    fprintf(stderr, "Error: file is not in SIF format.\n");
    tacFree(&tac); return(4);
  }
  /* Read scan start time */
  char scan_start_time[256];
  struct tm scantime1, scantime2;
  ret=tacGetHeaderScanstarttime(&tac.h, scan_start_time, &status);
  if(ret==TPCERROR_OK) ret=strDateTimeRead(scan_start_time, &scantime1);
  if(ret!=0) {
    if(verbose>0) fprintf(stderr, "Warning: cannot read scan start time.\n");
  }

  /*
   *  Add the following SIFs to the first one
   */
  TAC tac2; tacInit(&tac2);
  for(ai=ffi+1; ai<argc; ai++) {

    strlcpy(siffile, argv[ai], FILENAME_MAX);
    if(verbose>1) fprintf(stdout, "reading %s\n", siffile);
    ret=tacRead(&tac2, siffile, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      if(verbose>1) printf("ret := %d\n", ret);
      tacFree(&tac); return(5);
    }
    if(verbose>2) {
      printf("fileformat := %s\n", tacFormattxt(tac2.format));
      printf("tacNr := %d\n", tac2.tacNr);
      printf("sampleNr := %d\n", tac2.sampleNr);
    }
    if(tac2.format!=TAC_FORMAT_SIF) {
      fprintf(stderr, "Error: file is not in SIF format.\n");
      tacFree(&tac); tacFree(&tac2); return(5);
    }
    /* Delete extra columns if necessary */
    if(tac.tacNr<tac2.tacNr) tac2.tacNr=tac.tacNr;
    /* Allocate more space to the first dataset */
    ret=tacAllocateMoreSamples(&tac, tac2.sampleNr);
    if(ret!=0) {
      fprintf(stderr, "Error: cannot combine SIFs.\n");
      tacFree(&tac); tacFree(&tac2); return(6);
    }
    /* Calculate and correct the difference between scan times (sec) */
    ret=tacGetHeaderScanstarttime(&tac2.h, scan_start_time, &status);
    if(ret==TPCERROR_OK) ret=strDateTimeRead(scan_start_time, &scantime2);
    if(ret!=0) {
      if(verbose>0) fprintf(stderr, "Warning: cannot read scan start time.\n");
    } else {
      double tdif=tmDifference(&scantime2, &scantime1);
      if(verbose>1 || tdif<0.0)
        printf("scan_start_time_difference[s] := %g\n", tdif);
      if(tdif<0.0) fprintf(stderr, "Warning: SIFs in wrong order?\n");
      for(int i=0; i<tac2.sampleNr; i++) {
        tac2.x1[i]+=tdif; tac2.x2[i]+=tdif; tac2.x[i]+=tdif;
      }
    }
    /* Add new data to the end of the existing data */
    for(int i=0; i<tac2.sampleNr; i++) {
      tac.x1[tac.sampleNr]=tac2.x1[i];
      tac.x2[tac.sampleNr]=tac2.x2[i];
      tac.x[tac.sampleNr] =tac2.x[i];
      for(int j=0; j<tac2.tacNr; j++)
        tac.c[j].y[tac.sampleNr]=tac2.c[j].y[i];
      tac.sampleNr++;
    }
    tacFree(&tac2);
  } /* next SIF */

  /* Sort data by increasing frame time, in case SIFs were given 
     in wrong order */
  tacSortByTime(&tac, &status);

  /*
   *  Save the combined SIF 
   */
  if(verbose>1) printf("writing %s\n", catfile);
  FILE *fp; fp=fopen(catfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing.\n");
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_SIF, 1, &status);
  fclose(fp); 
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    if(verbose>1) printf("ret := %d\n", ret);
    tacFree(&tac); return(12);
  }
  if(verbose>0) printf("catenated file saved in %s\n", catfile);

  tacFree(&tac);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
