/** @file simboxes.c
    @brief Make matrix containing boxes with increasing integer values
           starting from 1 for image simulations and to be used as image mask
           in software testing. 
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpccsv.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Draw x,y,z matrix containing boxes with values starting from 1",
  "for simple simulations, SW testing, and creating ROI masks.",
  "Matrix and box x,y-dimension and z-dimension must be given in pixels.",
  " ",
  "Usage: @P [Options] matxydim matzdim boxxydim boxzdim outputfile",
  " ",
  "Options:",
  " -format=<<TSV>|<CSV>|<float>|<short>>",
  "     Matrix data can be saved as tab separated values (TSV, default),",
  "     comma separated values (CSV), or as binary flat files with either",
  "     floats or short ints.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example #1:",
  "  @P 128 15 16 1 boxes.txt",
  "Example #2:",
  "  @P -format=float 128 15 16 1 boxes.bin",
  "  flat2img -scanner=931 boxes.bin boxes.img 15 1 128 128",
  " ",
  "See also: flat2img, asc2flat, simcirc, imgshrink, imgadd, imglkup, dft2img",
  " ",
  "Keywords: simulation, image, mask, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  char     *cptr, outfile[FILENAME_MAX];
  int       matxydim, matzdim, boxxydim, boxzdim;
  int       csv_separator=2; // 0=comma, 1=semi-colon, 2=tab, 3=space
  int       output_format=0; // 0=ASCII, 1=float binary, 2=short int binary
  int       ret;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  matxydim=matzdim=boxxydim=boxzdim=-1;
  outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "F=", 2)==0) {
      cptr+=2;
      if(!strcasecmp(cptr, "TSV")) {csv_separator=2; output_format=0; continue;}
      if(!strcasecmp(cptr, "CSV")) {csv_separator=0; output_format=0; continue;}
      if(!strcasecmp(cptr, "FLOAT")) {output_format=1; continue;}
      if(!strcasecmp(cptr, "SHORT")) {output_format=2; continue;}
    } else if(strncasecmp(cptr, "FORMAT=", 7)==0) {
      cptr+=7;
      if(!strcasecmp(cptr, "TSV")) {csv_separator=2; output_format=0; continue;}
      if(!strcasecmp(cptr, "CSV")) {csv_separator=0; output_format=0; continue;}
      if(!strcasecmp(cptr, "FLOAT")) {output_format=1; continue;}
      if(!strcasecmp(cptr, "SHORT")) {output_format=2; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}  
  
  /* Process other arguments, starting from the first non-option */
  ret=0;
  if(ai<argc) {if(atoiCheck(argv[ai++], &matxydim)) ret++;}
  if(ai<argc) {if(atoiCheck(argv[ai++], &matzdim)) ret++;}
  if(ai<argc) {if(atoiCheck(argv[ai++], &boxxydim)) ret++;}
  if(ai<argc) {if(atoiCheck(argv[ai++], &boxzdim)) ret++;}
  if(ret || matxydim<1 || matzdim<1 || boxxydim>matxydim || boxzdim>matzdim) {
    fprintf(stderr, "Error: invalid dimension.\n");
    return(1);  
  }
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing or wrong? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("outfile := %s\n", outfile);
    printf("matxydim := %d\n", matxydim);
    printf("matzdim := %d\n", matzdim);
    printf("boxxydim := %d\n", boxxydim);
    printf("boxzdim := %d\n", boxzdim);
    printf("csv_separator := %d\n", csv_separator);
    printf("output_format := %d\n", output_format);
  }


  /*
   *  Compute and write ints into CSV or TSV file, if requested
   */
  if(output_format==0) {
    /* Check that size is reasonable */
    unsigned int size=matzdim*matxydim*matxydim;
    if(size>150000) {
      fprintf(stderr, "Error: too much data for text format.\n");
      return(2);
    }
    /* Create the data */
    int a, b, c, zi, yi, xi, mb=0, mc=0;
    int mat[matzdim][matxydim][matxydim];
    for(zi=0; zi<matzdim; zi++) {
      a=zi/boxzdim;
      for(yi=0; yi<matxydim; yi++) {
        b=yi/boxxydim;
        for(xi=0; xi<matxydim; xi++) {
          c=xi/boxxydim;
          mat[zi][yi][xi]= 1 + c + b*(mc+1) + a*(mc+1)*(mb+1);
          if(verbose>2)
            printf("mat[%d][%d][%d]=%d (%d %d %d) (%d %d)\n", 
                    zi, yi, xi, mat[zi][yi][xi], a, b, c, mb, mc);
          if(c>mc) mc=c;
        }
        if(b>mb) mb=b;
      }
    }
    /* Write the data */
    int newline, ret;
    CSV csv; csvInit(&csv);
    if(csvAllocate(&csv, matzdim*matxydim*matxydim)!=TPCERROR_OK) {
      fprintf(stderr, "Error: out of memory.\n");
      return(6);
    }
    ret=TPCERROR_OK; newline=0;
    for(zi=0; zi<matzdim && ret==TPCERROR_OK; zi++) {
      for(yi=0; yi<matxydim && ret==TPCERROR_OK; yi++) {
        for(xi=0; xi<matxydim && ret==TPCERROR_OK; xi++) {
          ret=csvPutInt(&csv, mat[zi][yi][xi], newline); newline=0;
        }
        newline=1;
      }
    }
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot fill CSV data.\n");
      csvFree(&csv); return(7);
    }  
    /* Set the field separator */
    if(csv_separator==0) csv.separator=',';
    else if(csv_separator==1) csv.separator=';';
    else if(csv_separator==2) csv.separator='\t';
    else csv.separator=' ';
    /* Open file */
    FILE *fp;
    if(strcasecmp(outfile, "stdout")==0) {
      fp=stdout;
    } else {
      fp=fopen(outfile, "w");
      if(fp==NULL) {
        fprintf(stderr, "Error: cannot open file for write.\n");
        csvFree(&csv); return(10);
      }
    }
    /* Write matrix data */
    ret=csvWrite(&csv, 0, fp, &status);
    csvFree(&csv); fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      if(verbose>1) printf("ret := %d\n", ret);
      return(11);
    }
    // We are ready
    return(0);
  }


  /*
   *  Write matrix data as flat binary file,
   *  calculating values pixel by pixel to prevent out-of-memory errors.
   */
  /* Open file */
  FILE *fp;
  fp=fopen(outfile, "wb");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for write.\n");
    return(10);
  }
  /* Compute and write data */
  int a, b, c, mb=0, mc=0, zi, yi, xi;
  unsigned int mv;
  float f;
  unsigned short int s;
  ret=TPCERROR_OK;
  for(zi=0; zi<matzdim && ret==TPCERROR_OK; zi++) {
    a=zi/boxzdim;
    for(yi=0; yi<matxydim && ret==TPCERROR_OK; yi++) {
      b=yi/boxxydim;
      for(xi=0; xi<matxydim && ret==TPCERROR_OK; xi++) {
        c=xi/boxxydim;
        mv=1 + c + b*(mc+1) + a*(mc+1)*(mb+1);
        if(output_format==1) { // binary floats
          f=(float)mv;
          if(fwrite(&f, sizeof(float), 1, fp) != 1) ret=TPCERROR_CANNOT_WRITE;
        } else if(output_format==2) { // binary short int
          s=(unsigned short int)mv;
          if(fwrite(&s, sizeof(unsigned short int), 1, fp) != 1) 
            ret=TPCERROR_CANNOT_WRITE;
        }
        if(c>mc) mc=c;
      }
      if(b>mb) mb=b;
    }
  }
  fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: cannot write in %s; disk full?\n", outfile);
    return(12);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
