#!/bin/bash
#: Title      : test_sim_3tcm
#: Date       : 2021-02-09
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None


# Set the exit status of a pipeline to the exit status of failed, not the last command in the pipeline
set -o pipefail

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../sim_3tcm$EXT;

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

if [ ! -f stead100.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f bolus.dat ] || [ ! -f bolus_short.dat ] || [ ! -f bolus_dt4.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f bolus_toofew.dat ] || [ ! -f bolus_wrongorder.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_3cmp.dat ] || [ ! -f correct_3cms.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f input_3cm.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f o15_h2o.dft ] || [ ! -f o15_h2o.fra ] || [ ! -f o15_h2o_rfit.dft ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f o15_h2o_min.dft ] ; then
  cp -f o15_h2o.dft o15_h2o_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xconv=min -yset=kBq/mL o15_h2o_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f f18_fdg.dft ] || [ ! -f f18_fdg.fra ] || [ ! -f f18_fdg_rfit.dft ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f f18_fdg_rfit_min.dft ] ; then
  cp -f f18_fdg_rfit.dft f18_fdg_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xset=sec -yset=kBq/mL f18_fdg_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xconv=min f18_fdg_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f c11_way100635.dft ] || [ ! -f c11_way100635.fra ] || [ ! -f c11_way_rfit.dft ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f c11_way_rfit_min.dft ] ; then
  cp -f c11_way_rfit.dft c11_way_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xset=sec -yset=kBq/mL c11_way_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xconv=min c11_way_rfit_min.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f multi.dat ] ; then
  tacadd -ovr multi.dat bolus.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -force multi.dat bolus.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f bolus_auc.dat ] ; then
  interpol -i -x=4990 bolus.dat bolus_auc.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case: Only PAR filename given as argument. \n"
printf " Expected result: Correct file saved. \n\n"

rm -f stdout.txt output.par
$PROGRAM output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat output.par &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt format TSV-UK
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.par model SER3TCM
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case: Option -parallel and PAR filename given as argument. \n"
printf " Expected result: Correct file saved. \n\n"

rm -f stdout.txt output.par
$PROGRAM --verbose -parallel output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat output.par &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt format TSV-UK
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.par model PAR3TCM
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Validity in perfect steady-state without vascular volume \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: Steady-state and 1TCM. \n"
printf " Test case 2: Save and do not save compartmental TACs. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f test1.par
printf "# model := PAR3TCM\n" > test1.par
printf "Parameters	K1[mL/(mL*min)]	k2[1/min]	k3[1/min]	k4[1/min]	k5[1/min]	k6[1/min]	Vb\n" >> test1.par
printf "tac1	0.6	0.3	0	0	0	0	0\n" >> test1.par

rm -f output.dat temp.dat temp.ift
$PROGRAM --debug=3 test1.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 200
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat temp.dat temp.ift
$PROGRAM -sub test1.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 200
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[2][1]
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.1 \n"
printf " Test case 1: Steady-state and 2TCM. \n"
printf " Test case 2: Save and do not save compartmental TACs. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f test2.par
printf "# model := SER3TCM\n" > test2.par
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb\n" >> test2.par
printf "tac1	0.6	0.3	0.2	0.4	0	0	0\n" >> test2.par

rm -f output.dat temp.dat temp.ift
$PROGRAM --debug=2 test2.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 300
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat temp.dat temp.ift
$PROGRAM -sub test2.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 300
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[2][1] 200
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[3][1] 100
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.2 \n"
printf " Test case 1: Steady-state and 3TCM, compartments in series. \n"
printf " Test case 2: Save and do not save compartmental TACs. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f test3.par
printf "# model := SER3TCM\n" > test3.par
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb\n" >> test3.par
printf "tac1	0.6	0.3	0.2	0.4	0.1	0.4	0\n" >> test3.par

rm -f output.dat temp.dat temp.ift
$PROGRAM --debug=2 test3.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 325
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat temp.dat temp.ift
$PROGRAM -sub test3.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 325
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[2][1] 200
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[3][1] 100
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[4][1] 25
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.0.3 \n"
printf " Test case 1: Steady-state and 3TCM, compartments in parallel. \n"
printf " Test case 2: Save and do not save compartmental TACs. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f test4.par
printf "# model := PAR3TCM\n" > test4.par
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb\n" >> test4.par
printf "tac1	0.6	0.4	0.5	0.2	1.0	1.2	0\n" >> test4.par

rm -f output.dat temp.dat temp.ift
$PROGRAM --verbose test4.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 650
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat temp.dat temp.ift
$PROGRAM -sub test4.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 650
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[2][1] 150
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[3][1] 375
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[4][1] 125
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Validity in perfect steady-state with vascular volume \n"
printf "===================================================================\n"


printf "\n 2.1.0 \n"
printf " Test case: Steady-state and 3TCM in series. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f test1.par
printf "# model := SER3TCM\n" > test1.par
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb	f	fA\n" >> test1.par
printf "tac1	0.5	5.0	0.25	0.5	0.6	0.1	0.10	0	0.3\n" >> test1.par

rm -f output.dat temp.dat temp2.dat temp.ift
$PROGRAM -sub test1.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -tv output.dat stead100.dat 10 temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut temp2.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 45
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf " Test case: Option -vvm=2. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat temp.dat temp2.dat temp.ift
$PROGRAM -vvm=2 -sub test1.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut output.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 55
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
taccbv --force -pv output.dat stead100.dat 10 temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut temp2.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 45
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.1 \n"
printf " Test case: Steady-state and 3TCM in parallel. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f test2.par
printf "# model := PAR3TCM\n" > test2.par
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb	f	fA\n" >> test2.par
printf "tac1	0.5	5.0	0.25	0.5	0.6	0.1	0.10	0	0.3\n" >> test2.par

rm -f output.dat temp.dat temp2.dat temp.ift
$PROGRAM test2.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -tv output.dat stead100.dat 10 temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut temp2.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 75
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf " Test case: Option -vvm=2. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat temp.dat temp2.dat temp.ift
$PROGRAM -vvm=2 test2.par stead100.dat stead100.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -pv output.dat stead100.dat 10 temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut temp2.dat 999.9 1000.1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.01 temp.ift value[1][1] 75
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Validity with perfect bolus steady-state without vascular volume \n"
printf "===================================================================\n"

printf "\n 2.2.0 \n"
printf " Test case: Bolus and 1TCM. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f test1.par
printf "Parameters	K1	k2	k3	k4	k5	k6\n" > test1.par
printf "tac1	0.01	0.10	0	0	0	0\n" >> test1.par

rm -f output.dat output_auc.dat temp.dat temp.ift
$PROGRAM test1.par bolus.dat bolus.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=4990 output.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[1][1] 0.1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.1 \n"
printf " Test case: Bolus and 2TCM. \n"
printf " Test case 2: string none is given in place of blood file. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f test2.par
printf "Parameters	K1	k2	k3	k4	k5	k6\n" > test2.par
printf "tac1	0.02	0.20	0.032	0.008	0	0\n" >> test2.par

rm -f output.dat output_auc.dat temp.dat temp.ift
$PROGRAM -sub test2.par bolus.dat none output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=4990 output.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[1][1] 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[2][1] 0.1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[3][1] 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.2 \n"
printf " Test case: Bolus and 3TCM in series. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f test3.par
printf "Parameters	K1	k2	k3	k4	k5	k6\n" > test3.par
printf "tac1	0.025	0.25	0.06	0.015	0.025	0.5\n" >> test3.par

rm -f output.dat output_auc.dat temp.dat temp.ift
$PROGRAM -sub test3.par bolus.dat bolus.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=4990 output.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 temp.ift value[1][1] 0.52
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[2][1] 0.1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[3][1] 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[4][1] 0.02
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.3 \n"
printf " Test case: Bolus and 3TCM in parallel. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f test4.par
printf "Parameters	K1	k2	k3	k4	k5	k6\n" > test4.par
#printf "tac1	0.6	0.4	0.5	0.2	1.0	1.2\n" >> test4.par
printf "tac1	0.03	0.02	0.025	0.01	0.05	0.06\n" >> test4.par

rm -f output.dat output_auc.dat temp.dat temp.ift
$PROGRAM -sub -parallel test4.par bolus.dat bolus.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=4990 output.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[1][1] 6.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[2][1] 1.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[3][1] 3.75
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0001 temp.ift value[4][1] 1.25
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.4 \n"
printf " Test case: Delay time. \n"
printf " Expected result: TAC correctly moved in time. \n\n"

rm -f test1.par test2.par output1.dat output2.dat
printf "Parameters	K1	k2	k3	k4	k5	k6\n" > test1.par
printf "tac1	0.025	0.25	0	0	0	0\n" >> test1.par

printf "Parameters	K1	k2	k3	k4	k5	k6	dT\n" > test2.par
printf "tac1	0.025	0.25	0	0	0	0	4\n" >> test2.par

$PROGRAM test1.par bolus_dt4.dat bolus_dt4.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test2.par bolus.dat bolus.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.0001 output1.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Validity in perfect bolus with vascular volume \n"
printf "===================================================================\n"

rm -f test.par
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb	f	fA\n" > test.par
printf "tac1	0.5	5.0	0.25	0.5	0.6	0.1	0.10	0	0.3\n" >> test.par

printf "\n 2.3.0 \n"
printf " Test case: Bolus and 3TCM in series. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f output.dat temp.dat temp2.dat output_auc.dat temp.ift
$PROGRAM -series test.par bolus.dat bolus.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -tv output.dat bolus.dat 10 temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=4990 temp2.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[1][1] 0.45
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf " Test case: Option -vvm=2. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat temp.dat temp2.dat output_auc.dat temp.ift
$PROGRAM -vvm=2 -series test.par bolus.dat bolus.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -pv output.dat bolus.dat 10 temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=4990 temp2.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[1][1] 0.45
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.1 \n"
printf " Test case: Bolus and 3TCM in parallel. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f output.dat temp.dat temp2.dat output_auc.dat temp.ift
$PROGRAM -paral test.par bolus.dat bolus.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -tv output.dat bolus.dat 10 temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=4990 temp2.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[1][1] 0.75
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf " Test case: Option -vvm=2. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat temp.dat temp2.dat output_auc.dat temp.ift
$PROGRAM -vvm=2 -paral test.par bolus.dat bolus.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -pv output.dat bolus.dat 10 temp2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=4990 temp2.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift temp.dat > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 temp.ift value[1][1] 0.75
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.3.2 \n"
printf " Test case: with delay time and sub-compartments. \n"
printf " Expected result: TACs correctly moved in time. \n\n"

rm -f output.dat output2.dat
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb	f	fA	dT\n" > test2.par
printf "tac1	0.5	5.0	0.25	0.5	0.6	0.1	0.10	0	0.3	4.0\n" >> test2.par

$PROGRAM -paral -sub test.par bolus_dt4.dat bolus_dt4.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -paral -sub test2.par bolus.dat bolus.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.0001 output.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Testing against simulation in spreadsheet \n"
printf "===================================================================\n"


printf "\n 2.4.0 \n"
printf " Test case: Bolus and 3TCM in series with Vb but no perfusion effect. \n"
printf " Expected result: simulated PET TAC is correct. \n\n"

rm -f test1.par
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb	fA\n" > test1.par
printf "tac1	0.4	0.6	0.3	0.4	0.08	0.01	0.04	1.0\n" >> test1.par

rm -f sims.dat
$PROGRAM -series -nosub test1.par input_3cm.dat input_3cm.dat sims.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=n -mt="3TCM-S" sims.svg -l correct_3cms.dat sims.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -xunit=y -yunit=y -tacnames=n -abs=0.001 -rel=0.01 correct_3cms.dat sims.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.4.1 \n"
printf " Test case: Bolus and 3TCM in parallel with Vb but no perfusion effect. \n"
printf " Expected result: simulated PET TAC is correct. \n\n"

rm -f test2.par
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb	fA\n" > test2.par
printf "tac1	0.4	0.6	0.2	0.05	0.2	0.2	0.04	1.0\n" >> test2.par

rm -f simp.dat
$PROGRAM -parallel -nosub test2.par input_3cm.dat input_3cm.dat simp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=n -mt="3TCM-P" simp.svg -l correct_3cmp.dat simp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -xunit=y -yunit=y -tacnames=n -abs=0.001 -rel=0.01 correct_3cmp.dat simp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Comparison to simulations made using RFIT 4.4 \n"
printf "===================================================================\n"


printf "\n 2.5.0 \n"
printf " Test case: Radiowater blood input, range of k1 and k2 values. \n"
printf " Test case 2: Simulate several TACs at once. \n"
printf " Expected result: Simulated TACs match the ones made with RFIT. \n\n"

rm -f test1.par
printf "Parameters	K1	k2	k3	k4	k5	k6\n" > test1.par
printf "tac1	0.01	0.01	0	0	0	0\n" >> test1.par
printf "tac2	0.04	0.05	0	0	0	0\n" >> test1.par
printf "tac3	0.30	0.40	0	0	0	0\n" >> test1.par
printf "tac4	0.80	1.00	0	0	0	0\n" >> test1.par
printf "tac5	1.50	2.00	0	0	0	0\n" >> test1.par

rm -f output.dat output.dft temp1.dft temp2.dft
$PROGRAM -nosub test1.par o15_h2o_min.dft o15_h2o_min.dft output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacunit -xconv=sec output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
simframe output.dat o15_h2o.fra output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="sim_3tcm vs RFIT 4.4: radiowater" o15_h2o_rfit.svg -s o15_h2o_rfit.dft -l output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

interpol -y -x=20,50,75,90,120,180,300 o15_h2o_rfit.dft temp1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -x=20,50,75,90,120,180,300 output.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=n -yunit=n -abs=0.15 -rel=0.01 temp1.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.5.1 \n"
printf " Test case: FDG plasma input, 1-3 TCM. \n"
printf " Test case 2: Simulate several TACs at once. \n"
printf " Expected result: Simulated TACs match the ones made with RFIT. \n\n"

rm -f test2.par
printf "Parameters	K1	k2	k3	k4	k5	k6\n" > test2.par
printf "tac1	0.100	0.130	0.060	0	0	0\n" >> test2.par
printf "tac2	0.100	0.130	0.060	0.01	0	0\n" >> test2.par
printf "tac3	0.017	0.224	0.039	0.019	0.025	0\n" >> test2.par
printf "tac4	0.022	0.178	0.123	0.003	0.720	0\n" >> test2.par
printf "tac5	0.800	0.900	0.600	0.400	0.400	0.6\n" >> test2.par


rm -f output.dat output.dft temp1.dft temp2.dft
$PROGRAM -nosub test2.par f18_fdg.dft f18_fdg.dft output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

simframe -mid output.dat f18_fdg.fra output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="sim_3tcm vs RFIT 4.4: FDG" f18_fdg_rfit.svg -s f18_fdg_rfit_min.dft -l output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

interpol -y -x=1.3,2,2.5,5,7,10,20,30,40,50,60,70,80 f18_fdg_rfit_min.dft temp1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -x=1.3,2,2.5,5,7,10,20,30,40,50,60,70,80 output.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=n -yunit=n -abs=0.015 -rel=0.01 temp1.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.5.2 \n"
printf " Test case: [C-11]WAY-100635 plasma input, 1-2 TCM. \n"
printf " Test case 2: Simulate several TACs at once. \n"
printf " Expected result: Simulated TACs match the ones made with RFIT. \n\n"

rm -f test3.par
printf "Parameters	K1	k2	k3	k4	k5	k6\n" > test3.par
printf "tac1	0.160	0.360	0.046	0.046	0	0\n" >> test3.par
printf "tac2	0.160	0.360	0.230	0.023	0	0\n" >> test3.par

rm -f output.dat output.dft temp1.dft temp2.dft
$PROGRAM -nosub test3.par c11_way100635.dft c11_way100635.dft output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

simframe -mid output.dat c11_way100635.fra output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="sim_3tcm vs RFIT 4.4: [C-11]WAY-100635" c11_way_rfit.svg -s c11_way_rfit_min.dft -l output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

interpol -y -x=1,2,3,4,5,7,10,15,20,30,40,50,60 c11_way_rfit_min.dft temp1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -y -x=1,2,3,4,5,7,10,15,20,30,40,50,60 output.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=n -yunit=n -abs=0.015 -rel=0.01 temp1.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Perfusion effect \n"
printf "===================================================================\n"

rm -f test.par
printf "Parameters	K1	k2	k3	k4	k5	k6	Vb	f	fA\n" > test.par
printf "tac1	0.2	0.4	0.3	0.33	0.6	0.1	0.10	0.5	0.2\n" >> test.par


printf "\n 3.0.0 \n"
printf " Test case 1: Bolus and 3TCM in series with perfusion effect. \n"
printf " Test case 2: Quantitative result is not yet tested. \n"
printf " Expected result: TACs can be simulated and plotted. \n\n"

rm -f sim1.bld simu1.bld
$PROGRAM -series -sub test.par bolus.dat bolus.dat sim1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
ainterp sim1.bld 0 simu1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y -x2=30 simu1.svg simu1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n 3.0.1 \n"
printf " Test case 1: Bolus and 3TCM in series with perfusion effect. \n"
printf " Test case 2: Quantitative result is not yet tested. \n"
printf " Expected result: TACs can be simulated and plotted. \n\n"

rm -f sim2.bld simu2.bld
$PROGRAM -parallel -sub test.par bolus.dat bolus.dat sim2.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
ainterp sim2.bld 0 simu2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y -x2=30 simu2.svg simu2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"





printf "\n===================================================================\n"
printf "   Error handling \n"
printf "===================================================================\n"


printf "\n 4.0.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM -stupidoption test.par bolus.dat bolus.dat tissue.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM test.par bolus.dat bolus.dat tissue.dat extra &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.2 \n"
printf " Test case: Missing argument(s). \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt
$PROGRAM bolus.dat tissue.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt Error "missing filename."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Input data \n"
printf "===================================================================\n"


printf "\n 4.1.0 \n"
printf " Test case: Non-existing input file. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dat
$PROGRAM nonexisting.par bolus.dat bolus.dat output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "cannot open file (nonexisting.par)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM test.par nonexisting.dat bolus.dat output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "cannot open file (input files)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM test.par bolus.dat nonexisting.dat output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "cannot open file (input files)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.1.1 \n"
printf " Test case: Wrong input file format. \n"
printf " Expected result: Error. \n\n"

printf "abc defg\nbubudum\nmeh\n" > bad.dat

rm -f stdout.txt output.dat
$PROGRAM bad.dat bolus.dat bolus.dat output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "unsupported file type (bad.dat)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM test.par bad.dat bolus.dat output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid file format (input files)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM test.par bolus.dat bad.dat output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid file format (input files)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.2 \n"
printf " Test case: Input TAC file contains more than one TAC. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dat
$PROGRAM --verbose test.par multi.dat bolus.dat output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "using only first TAC in multi.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM --verbose test.par bolus.dat multi.dat output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "using only first TAC in multi.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.3 \n"
printf " Test case: Input TAC file contains less than 3 samples. \n"
printf " Expected result: Error. \n\n"

rm -f stdout.txt output.dat
$PROGRAM test.par bolus_toofew.dat bolus.dat output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "file contains too few samples (input files)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM test.par bolus.dat bolus_toofew.dat output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "file contains too few samples (input files)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f stdout.txt output.dat
$PROGRAM test.par bolus_toofew.dat bolus_toofew.dat output.dat &> stdout.txt
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "file contains too few samples (input files)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.1.4 \n"
printf " Test case: Input TAC file samples are not in ascending order. \n"
printf " Expected result: No error (data is sorted). \n\n"

rm -f stdout.txt output.dat
$PROGRAM test.par bolus_wrongorder.dat bolus_short.dat output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n"

rm -f stdout.txt output.dat
$PROGRAM test.par bolus_short.dat bolus_wrongorder.dat output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n"

rm -f stdout.txt output.dat
$PROGRAM test.par bolus_wrongorder.dat bolus_wrongorder.dat output.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Unit conversions \n"
printf "===================================================================\n"



printf "\n 5.0.0 \n"
printf " Test case 1: Input TAC file time unit is min, parameters per min. \n"
printf " Test case 2: Vb as fraction. \n"
printf " Expected result: Correct output units. \n\n"

rm -f test1.par
printf "# model := 1TCM\n" > test1.par
printf "Parameters	K1[mL/mL/min]	k2[1/min]	Vb\n" >> test1.par
printf "tac1	1.2	1.8	0.1\n" >> test1.par

cp -f bolus_short.dat test_min.dat
tacunit -xconv=min test_min.dat

rm -f stdout.txt output1.dat output.ift
$PROGRAM --verbose -sub test1.par test_min.dat test_min.dat output1.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y t500.svg test_min.dat output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit output1.dat > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.ift timeunit min
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.0.1 \n"
printf " Test case 1: Input TAC file time unit is sec, parameters per sec. \n"
printf " Test case 2: Vb as percentage. \n"
printf " Expected result 1: Correct output units. \n"
printf " Expected result 2: Same output TACs as in simulation with default units. \n\n"

rm -f test2.par
printf "# model := 1TCM\n" > test2.par
printf "Parameters	K1[mL/mL/sec]	k2[1/sec]	Vb[%%]\n" >> test2.par
printf "tac1	0.02	0.03	10\n" >> test2.par

cp -f bolus_short.dat test_sec.dat
tacunit -xconv=sec test_sec.dat

rm -f stdout.txt output2.dat output.ift
$PROGRAM --verbose -sub test2.par test_sec.dat test_sec.dat output2.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y t501.svg test_sec.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit output2.dat > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.ift timeunit sec
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 output1.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.0.2 \n"
printf " Test case 1: Input TAC file time unit is min, parameters per sec. \n"
printf " Test case 2: Vb as percentage. \n"
printf " Expected result 1: Correct output units. \n"
printf " Expected result 2: Same output TACs as in simulation with default units. \n\n"

rm -f stdout.txt output3.dat output.ift
$PROGRAM --verbose -sub test2.par test_min.dat test_min.dat output3.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y t502.svg test_min.dat output3.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit output3.dat > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.ift timeunit min
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 output1.dat output3.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.0.3 \n"
printf " Test case 1: Input TAC file time unit is sec, parameters per min. \n"
printf " Expected result 1: Correct output units. \n"
printf " Expected result 2: Same output TACs as in simulation with default units. \n\n"

rm -f stdout.txt output4.dat output.ift
$PROGRAM --verbose -sub test1.par test_sec.dat test_sec.dat output4.dat &> stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y t503.svg test_sec.dat output4.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit output4.dat > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.ift timeunit sec
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 output1.dat output4.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

