#!/bin/bash
#: Title      : test_svar4tac
#: Date       : 2019-09-13
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../svar4tac$EXT;

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f blood.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


if [ ! -f blood.bld ] || [ ! -f tissue.tac ]; then
  rm -f tissue.tac multi.tac
  fit2dat -c=0,420,1 blood.fit blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f tissue.tac ] ; then
  b2t_h2o -nosub -fpt -voiname=gm blood.bld 100 0.94 1.0 5 30 tissue.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  b2t_h2o -add -nosub -fpt -voiname=wm blood.bld 20 0.94 1.0 3 30 tissue.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt="" -legend=y tacs.svg -l blood.bld tissue.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case: One BTAC in file. \n"
printf " Expected result 1: Noisy curve saved. \n"
printf " Expected result 2: AUC of noisy BTAC is close to AUC of original BTAC. \n\n"

rm -f noisy.bld noisy1.svg

$PROGRAM blood.bld 5 noisy.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="BTAC with and without noise" noisy1.svg -l blood.bld noisy.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -x=y -y=y -w=n -tacnames=y -xunit=y -yunit=y -abs=5 -rel=30 blood.bld noisy.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -w=n -tacnames=n -xunit=n -yunit=n -abs=0.5 -rel=1 blood.bld noisy.bld
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

auc1=$(interpol -i -x=420 -header=no blood.bld)
printf "AUC := %s\n" $auc1 > auc1.ift
auc2=$(interpol -i -x=420 -header=no noisy.bld)
printf "AUC := %s\n" $auc2 > auc2.ift
#cat auc1.ift auc2.ift
iftmatch -abs=200 auc1.ift auc2.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.1 \n"
printf " Test case 1: 5 copies of the single TAC. \n"
printf " Test case 2: 5 copies of TAC made without noise. \n"
printf " Expected result 1: Five noisy and noiseless curves saved. \n"
printf " Expected result 2: AUC of noisy BTACs is close to AUC of original BTAC. \n\n"

rm -f noisy.bld noisy1n.svg

$PROGRAM -x=5 blood.bld 5 noisy.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="BTAC with and without noise" noisy1n.svg -l blood.bld noisy.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

taclist -nv -na noisy.bld > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt tacNr 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

interpol -i -x=300,420 blood.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -x=5 temp.bld 0 temp1.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=300,420 noisy.bld temp2.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
#taclist -nv temp1.bld
#taclist -nv temp2.bld
tacmatch -x=y -y=y -w=n -tacnames=y -xunit=y -yunit=y -abs=100 -rel=10 temp1.bld temp2.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case: Two TTACs in file. \n"
printf " Expected result 1: Noisy curves saved. \n"
printf " Expected result 2: AUC of noisy BTACs is close to AUC of original BTAC. \n\n"

rm -f noisy.tac noisy2.svg

$PROGRAM tissue.tac 5 noisy.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -mt="TTACs with and without noise" noisy2.svg -l tissue.tac noisy.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

interpol -i -x=300,420 tissue.tac temp1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=300,420 noisy.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
#taclist -nv temp1.tac
#taclist -nv temp2.tac
tacmatch -x=y -y=y -w=n -tacnames=y -xunit=y -yunit=y -abs=100 -rel=10 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case 1: 5 copies of the TTACs. \n"
printf " Test case 2: 5 copies of TAC made without noise. \n"
printf " Expected result 1: Five noisy and noiseless curves per original TAC saved. \n"
printf " Expected result 2: AUC of noisy BTACs is close to AUC of original BTAC. \n\n"

rm -f noisy.tac noisy2n.svg

$PROGRAM -x=5 tissue.tac 5 noisy.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="TTACs with and without noise" noisy2n.svg -l tissue.tac noisy.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

taclist -nv -na noisy.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt tacNr 10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

interpol -i -x=300,420 tissue.tac temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -x=5 temp.tac 0 temp1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=300,420 noisy.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
#taclist -nv temp1.tac
#taclist -nv temp2.tac
tacmatch -x=y -y=y -w=n -tacnames=y -xunit=y -yunit=y -abs=100 -rel=10 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.2 \n"
printf " Test case: 5 copies saved as separate files. \n"
printf " Expected result: Five noisy files saved. \n\n"

rm -f noisy2nnnnn.svg

$PROGRAM -separate -x=5 tissue.tac 5 noisy.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="TTACs with and without noise" noisy2nnnnn.svg -l tissue.tac noisy_000?.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

taclist -nv -na noisy_0001.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt tacNr 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -nv -na noisy_0002.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt tacNr 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -nv -na noisy_0003.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt tacNr 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -nv -na noisy_0004.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt tacNr 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -nv -na noisy_0005.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt tacNr 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.0 \n"
printf " Test case: Making 200 noisy copies. \n"
printf " Expected result 1: Noisy curves are saved. \n"
printf " Expected result 2: Estimated noise level is reasonably close to correct. \n\n"

rm -f noisy200.bld

$PROGRAM -x=200 blood.bld 20 noisy200.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmean -cv=noisy200cv.bld noisy200.bld noisy200mean.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -x=y -y=y -w=n -tacnames=n -xunit=y -yunit=y -abs=2 -rel=5 blood.bld noisy200mean.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

dftinteg -avg noisy200cv.bld 10 410 cvmean.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift cvmean.res cvmean.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.02 cvmean.ift Mean 0.2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.3.0 \n"
printf " Test case: Making 100 noisy copies with steady SD. \n"
printf " Expected result 1: Noisy curves are saved. \n"
printf " Expected result 2: Estimated noise level is reasonably close to correct. \n\n"

rm -f noisy100.bld

$PROGRAM -x=100 -minsd=1 blood.bld 0 noisy100.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="BTAC with added SD" noisy100sd.svg -l noisy100.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmean -sd=noisy100sd.bld noisy100.bld noisy100mean.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -x=y -y=y -w=n -tacnames=n -xunit=y -yunit=y -abs=3 -rel=5 blood.bld noisy100mean.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

dftinteg -avg noisy100sd.bld 0 420 sdmean.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift sdmean.res sdmean.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 sdmean.ift Mean 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.4.0 \n"
printf " Test case: Making 160 noisy copies with CV and min SD. \n"
printf " Expected result 1: Noisy curves are saved. \n"
printf " Expected result 2: Estimated noise level is reasonably close to correct. \n\n"

rm -f noisy160.bld

$PROGRAM -x=160 -minsd=0.05 blood.bld 40 noisy160.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="BTAC with added CV and SD" noisy160cvsd.svg -l noisy160.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmean -sd=noisy160sd.bld -cv=noisy160cv.bld noisy160.bld noisy160mean.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -x=y -y=y -w=n -tacnames=n -xunit=y -yunit=y -abs=3 -rel=5 blood.bld noisy160mean.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

dftinteg -avg noisy160sd.bld 0 7 sdmean.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift sdmean.res sdmean.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.02 sdmean.ift Mean 0.05
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

dftinteg -avg noisy160cv.bld 10 410 cvmean.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift cvmean.res cvmean.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.05 cvmean.ift Mean 0.40
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

