/** @file tacmidy.c
 *  @brief Convert TAC y values representing averages during frames with values
 *  at the frame middle time points for SW testing.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
#include "tpcli.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Convert TAC y values representing averages during frames with values",
  "at the frame middle time points, preserving frame AUCs.",
  " ",
  "Usage: @P [Options] tacfile newfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacframe, simframe, tacsetx, tacadd0, tacformat",
  " ",
  "Keywords: TAC, tool, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    ret;
//char  *cptr;
  char   tacfile1[FILENAME_MAX], tacfile2[FILENAME_MAX];
  TAC    tac;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac); tacfile1[0]=tacfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  for(; ai<argc; ai++) {
    if(!tacfile1[0]) {
      strlcpy(tacfile1, argv[ai], FILENAME_MAX); continue;
    } else if(!tacfile2[0]) {
      strlcpy(tacfile2, argv[ai], FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }
  /* Is something missing? */
  if(!tacfile2[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("tacfile1 := %s\n", tacfile1);
    printf("tacfile2 := %s\n", tacfile2);
  }

  /*
   *  Read TAC file
   */
  if(verbose>1) printf("reading %s\n", tacfile1);
  ret=tacRead(&tac, tacfile1, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    printf("xunit := %s\n", unitName(tac.tunit));
    printf("yunit := %s\n", unitName(tac.cunit));
  }
  /* Check that we have frame start and end times */
  if(tac.isframe==0) {
    fprintf(stderr, "Error: file does not contain frame start and end times.\n");
    tacFree(&tac); return(3);
  }
  if(tacXNaNs(&tac)>0) {
    fprintf(stderr, "Error: missing sample times in TAC file.\n");
    tacFree(&tac); return(3);
  }
  /* Check sample number */
  if(tac.sampleNr<3) {
    fprintf(stderr, "Warning: TAC contains only %d sample(s).\n", tac.sampleNr);
  }


  /*
   *  Do the thing
   */
  double y2[tac.sampleNr];
  ret=0;
  for(int i=0; i<tac.tacNr && !ret; i++) {
    ret=liFrameMidValue(tac.x1, tac.x2, tac.c[i].y, tac.sampleNr, y2, verbose-2);
    for(int j=0; j<tac.sampleNr; j++) tac.c[i].y[j]=y2[j];
  }
  if(ret) {
    fprintf(stderr, "Error: cannot calculate y values.\n");
    if(verbose>0) printf("ret := %d\n", ret);
    tacFree(&tac); return(8);
  }
  tac.isframe=0; // prevent accidentally repeating this


  /*
   *  Save TAC data 
   */
  if(verbose>2) printf("writing %s\n", tacfile2);
  FILE *fp; fp=fopen(tacfile2, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile2);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond

